# バッチ設計書 5-start-release

## 概要

本ドキュメントは、Next.jsのリリースプロセスを開始する `start-release` バッチの設計書である。バージョンバンプ、lernaによるバージョン更新、Gitタグ作成、リリースブランチへのプッシュを行う。

### 本バッチの処理概要

`start-release` は、Next.jsの新バージョンリリースを開始するバッチ処理である。リリースタイプ（stable/canary/rc/beta）に応じてバージョン番号を更新し、Gitタグを作成してGitHubにプッシュする。

**業務上の目的・背景**：Next.jsは複数のリリースチャネル（stable、canary、release-candidate、beta）を運用しており、各チャネルに応じたバージョニングとGitタグ管理が必要である。本バッチはlernaのバージョン管理機能を活用し、モノレポ内の全パッケージのバージョンを一括更新する。自動化により人為的ミスを防ぎ、リリースプロセスの一貫性を確保する。

**バッチの実行タイミング**：リリースプロセス開始時に実行。canaryリリースはほぼ毎日、stableリリースは数週間～数ヶ月ごとに実施される。

**主要な処理内容**：
1. リリースタイプとsemverタイプの引数検証
2. RELEASE_BOT_GITHUB_TOKEN の検証
3. Git リモートURLの設定（nextjs-botユーザー）
4. Git ユーザー設定（nextjs-bot）
5. lerna version によるバージョンバンプ（preid付き）
6. GitHub Release の作成（pre-release の場合）

**前後の処理との関連**：前提として `RELEASE_BOT_GITHUB_TOKEN` 環境変数の設定が必要。後続処理として `publish-release` によるnpmパッケージ公開、`publish-native` によるネイティブバイナリ公開が行われる。

**影響範囲**：モノレポ内全パッケージの `package.json`（バージョン更新）、`lerna.json`（バージョン更新）、Gitリポジトリ（タグ作成、コミット）。

## バッチ種別

リリース処理（バージョン管理）

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | canary: ほぼ日次、stable: 数週間～数ヶ月ごと |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 手動 / CI/CD |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| RELEASE_BOT_GITHUB_TOKEN | GitHub APIトークンが環境変数に設定されていること |
| Gitリポジトリ | vercel/next.jsリポジトリのクローンであること |
| release パッケージ | node_modules に release パッケージがインストール済み |
| configstore パッケージ | release パッケージの依存として利用可能 |

### 実行可否判定

- `--release-type` が stable/canary/release-candidate/beta のいずれかであること
- canary以外の場合、`--semver-type` が patch/minor/major のいずれかであること
- `RELEASE_BOT_GITHUB_TOKEN` が設定されていること

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| --release-type | string | Yes | なし | リリースタイプ（stable/canary/release-candidate/beta） |
| --semver-type | string | canary以外はYes | なし | セマンティックバージョンタイプ（patch/minor/major） |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| lerna.json | JSON | 現在のバージョン情報 |
| packages/*/package.json | JSON | 各パッケージのバージョン情報 |
| RELEASE_BOT_GITHUB_TOKEN | 環境変数 | GitHub認証トークン |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| lerna.json | JSON | 更新されたバージョン番号 |
| packages/*/package.json | JSON | 更新されたバージョン番号 |
| Gitタグ | Git | バージョンタグ（例: v15.0.0-canary.18） |
| GitHub Release | GitHub API | リリースエントリ（pre-release時） |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | lerna.json, package.json |
| 出力先 | リポジトリルート、各パッケージディレクトリ |
| 文字コード | UTF-8 |
| 区切り文字 | JSON形式 |

## 処理フロー

### 処理シーケンス

```
1. 引数検証
   └─ release-type と semver-type の妥当性チェック
2. GitHub トークン検証
   └─ RELEASE_BOT_GITHUB_TOKEN の存在チェック
3. configstore にトークン設定
   └─ release パッケージ用のトークン設定
4. Git リモートURL更新
   └─ nextjs-bot ユーザーでの認証URL設定
5. Git ユーザー設定
   └─ user.name: "nextjs-bot", user.email: "it+nextjs-bot@vercel.com"
6. lerna version 実行
   └─ canary: premajor/preminor/prerelease --preid canary
   └─ rc: premajor/preminor/prerelease --preid rc
   └─ beta: premajor/preminor/prerelease --preid beta
   └─ stable: patch/minor/major
7. release コマンド実行（pre-release時）
   └─ pnpm release --pre --skip-questions --show-url
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始: start-release] --> B{release-type検証}
    B -->|不正| C[エラー出力して終了]
    B -->|正常| D{semver-type検証}
    D -->|不正かつcanary以外| C
    D -->|正常| E{GITHUB_TOKEN存在?}
    E -->|なし| C
    E -->|あり| F[configstore にトークン設定]
    F --> G[Git リモートURL・ユーザー設定]
    G --> H{リリースタイプ}
    H -->|canary| I[lerna version prerelease --preid canary]
    H -->|rc| J[lerna version prerelease --preid rc]
    H -->|beta| K[lerna version prerelease --preid beta]
    H -->|stable| L[lerna version semver-type]
    I --> M[release --pre 実行]
    J --> M
    K --> M
    L --> N[バッチ終了]
    M --> N
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベース操作を行わない。GitリポジトリとGitHub APIへの操作を行う。

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 該当なし | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| INVALID_RELEASE_TYPE | バリデーションエラー | 不正なリリースタイプ指定 | stable/canary/rc/betaのいずれかを指定 |
| INVALID_SEMVER_TYPE | バリデーションエラー | 不正なsemverタイプ指定 | patch/minor/majorのいずれかを指定 |
| MISSING_TOKEN | 環境変数エラー | RELEASE_BOT_GITHUB_TOKEN未設定 | 環境変数を設定 |
| GIT_PUSH_FAILED | Git操作エラー | リモートへのプッシュ失敗 | ネットワーク接続とトークン権限を確認 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

リリース失敗時は、作成されたGitタグとバージョンコミットを手動でリバートする必要がある。`git tag -d` でタグを削除し、`git reset` でコミットを取り消す。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | Git操作全体 |
| コミットタイミング | lerna versionが自動コミット |
| ロールバック条件 | push失敗時はローカルコミットのリバートが必要 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1回のリリースで全パッケージのバージョン更新 |
| 目標処理時間 | 数十秒～数分 |
| メモリ使用量上限 | 通常のNode.jsプロセスメモリ |

## 排他制御

リリースプロセスは排他的に実行する必要がある。同時に複数のリリースを開始することは避けなければならない。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 開始ログ | 実行開始時 | "Running pnpm release-{canary/stable}..." |
| 進捗ログ | lerna実行中 | lernaの出力（stdio: pipe経由） |
| 終了ログ | 完了時 | "Release process is finished" |
| エラーログ | エラー発生時 | エラーメッセージ |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 処理時間 | 10分 | CI/CD通知 |
| 終了コード | 0以外 | リリース担当者 |

## 備考

- エントリポイント: `scripts/start-release.js`
- Git設定（user.name, user.email）は nextjs-bot として固定設定される
- canaryリリースでは semver-type に応じて premajor/preminor/prerelease が選択される
- stable リリースでは `pnpm release --pre` は実行されない
- lerna version に `--force-publish -y` が付与され、全パッケージが強制的にバージョン更新される
