# バッチ設計書 8-create-release-branch

## 概要

本ドキュメントは、リリース用ブランチを作成する `create-release-branch` バッチの設計書である。指定されたタグ名からブランチを作成し、CI/CDワークフローの設定を更新してGitHubリポジトリにプッシュする。

### 本バッチの処理概要

`create-release-branch` は、stableリリース用のブランチを作成するバッチ処理である。指定されたGitタグを基点にブランチを作成し、lerna設定やGitHub Actionsワークフローを更新した上でリモートにプッシュし、GitHubのデプロイメント環境ブランチポリシーも更新する。

**業務上の目的・背景**：Next.jsのstableリリースでは、canaryブランチとは別のリリースブランチが必要となる。リリースブランチでは、CI/CDワークフローがcanaryブランチではなくリリースブランチを参照するように更新され、新テストの自動実行を無効化する等の調整が行われる。また、GitHub Environmentsのブランチポリシーも更新し、リリースブランチからのデプロイを許可する。

**バッチの実行タイミング**：stableリリースプロセスの初期段階で、リリースブランチの作成が必要な場合に実行する。

**主要な処理内容**：
1. ブランチ名・タグ名の引数検証
2. RELEASE_BOT_GITHUB_TOKEN の検証
3. Git リモートURL・ユーザー設定
4. ブランチ作成とタグからの reset
5. lerna.json の allowBranch にブランチ名を追加
6. build_and_deploy.yml のブランチ参照を更新
7. build_and_test.yml のブランチ参照を更新、test-new-tests を無効化
8. 変更をコミット・プッシュ
9. GitHub Environments のデプロイメントブランチポリシー更新

**前後の処理との関連**：`start-release` の前に実行されることがある。後続処理として、リリースブランチ上での `start-release` と `publish-release` が行われる。

**影響範囲**：Gitリポジトリ（新ブランチ作成）、lerna.json、GitHub Actionsワークフロー設定、GitHub Environmentsブランチポリシー。

## バッチ種別

リリース処理（ブランチ管理）

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | stableリリースごと |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 手動 / CI/CD |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| RELEASE_BOT_GITHUB_TOKEN | GitHub APIトークンが環境変数に設定 |
| --branch-name | 作成するブランチ名が指定されていること |
| --tag-name | "v" で始まる有効なGitタグ名が指定されていること |

### 実行可否判定

- `--branch-name` が指定されていること（未指定時はエラー）
- `--tag-name` が "v" で始まること（不正時はエラー）
- `RELEASE_BOT_GITHUB_TOKEN` が設定されていること

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| --branch-name | string | Yes | なし | 作成するブランチ名 |
| --tag-name | string | Yes | なし | 基点となるGitタグ名（"v"始まり） |
| RELEASE_BOT_GITHUB_TOKEN | 環境変数 | Yes | なし | GitHub APIトークン |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| Gitタグ | Git | ブランチの基点 |
| lerna.json | JSON | allowBranch設定 |
| .github/workflows/build_and_deploy.yml | YAML | デプロイワークフロー |
| .github/workflows/build_and_test.yml | YAML | テストワークフロー |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| Gitリモート | Git | 新しいリリースブランチ |
| lerna.json | JSON | allowBranchにブランチ名追加 |
| build_and_deploy.yml | YAML | ブランチ参照更新 |
| build_and_test.yml | YAML | ブランチ参照更新、test-new-tests無効化 |
| GitHub Environments | GitHub API | デプロイメントブランチポリシー |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | lerna.json, build_and_deploy.yml, build_and_test.yml |
| 出力先 | リポジトリルート、.github/workflows/ |
| 文字コード | UTF-8 |
| 区切り文字 | JSON / YAML |

## 処理フロー

### 処理シーケンス

```
1. 引数検証
   └─ branchName と tagName の存在・形式チェック
2. GitHub トークン検証
   └─ RELEASE_BOT_GITHUB_TOKEN の存在チェック
3. Git 設定
   └─ リモートURL更新、ユーザー名・メール設定
4. ブランチ作成
   └─ git checkout -b {branchName}
5. タグ取得・リセット
   └─ git fetch origin {tagName} --tags
   └─ git reset --hard {tagName}
6. lerna.json 更新
   └─ command.publish.allowBranch にブランチ名追加
7. build_and_deploy.yml 更新
   └─ refs/heads/canary → refs/heads/{branchName}
8. build_and_test.yml 更新
   └─ ['canary'] → ['{branchName}']
   └─ test-new-tests ジョブを無効化（if: false）
9. コミット・プッシュ
   └─ git add . && git commit && git push
10. GitHub Environments ブランチポリシー更新
    └─ release-stable 環境にブランチポリシー追加
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始: create-release-branch] --> B{引数検証}
    B -->|不正| C[エラー出力して終了]
    B -->|正常| D{GITHUB_TOKEN存在?}
    D -->|なし| C
    D -->|あり| E[Git設定更新]
    E --> F[ブランチ作成・タグリセット]
    F --> G[lerna.json更新]
    G --> H[ワークフロー設定更新]
    H --> I[コミット・プッシュ]
    I --> J[5秒待機]
    J --> K[GitHub Environmentsブランチポリシー更新]
    K --> L{API成功?}
    L -->|成功| M[バッチ終了]
    L -->|失敗| N[エラー出力して終了]
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベース操作を行わない。

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 該当なし | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| MISSING_BRANCH_NAME | バリデーションエラー | --branch-name未指定 | ブランチ名を指定 |
| INVALID_TAG_NAME | バリデーションエラー | タグ名が"v"で始まらない | 正しいタグ名を指定 |
| MISSING_TOKEN | 環境変数エラー | RELEASE_BOT_GITHUB_TOKEN未設定 | 環境変数を設定 |
| GIT_PUSH_FAILED | Git操作エラー | プッシュ失敗 | ネットワーク・トークン権限を確認 |
| ENVIRONMENT_UPDATE_FAILED | GitHub APIエラー | ブランチポリシー更新失敗 | GitHub API権限を確認 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

ブランチ作成後にプッシュが失敗した場合、ローカルブランチを削除して再実行する。GitHub Environments更新が失敗した場合は、GitHub UIから手動でブランチポリシーを追加する。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | Git操作全体 |
| コミットタイミング | 設定変更完了後に一括コミット |
| ロールバック条件 | プッシュ失敗時はローカルブランチ削除が必要 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1回の実行で1ブランチ作成 |
| 目標処理時間 | 数十秒 |
| メモリ使用量上限 | 通常のNode.jsプロセスメモリ |

## 排他制御

同一ブランチ名での重複実行はGitにより自動的に拒否される。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 待機ログ | プッシュ後 | "Waiting 5s before updating branch rules" |
| 成功ログ | API更新後 | "Successfully updated deployment environment branch rules" |
| エラーログ | API失敗時 | ステータスコードとレスポンス本文 |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 処理時間 | 5分 | CI/CD通知 |
| 終了コード | 0以外 | リリース担当者 |

## 備考

- エントリポイント: `scripts/create-release-branch.js`
- GitHub Environments API エンドポイント: `repos/vercel/next.js/environments/release-stable/deployment-branch-policies`
- build_and_test.yml の test-new-tests ジョブは `# test-new-tests-if` / `# test-new-tests-end-if` マーカーで囲まれており、`if: false` に置換される
- プッシュ後にGitHub Environments更新前に5秒の待機がある（GitHub側の反映待ち）
- Git設定（user.name, user.email）は nextjs-bot として固定設定される
