# 機能設計書 13-App Router

## 概要

本ドキュメントは、Next.jsのApp Routerの機能設計を記述する。App Routerは `app/` ディレクトリに基づくファイルシステムベースのルーティング機構であり、レイアウト、ネスト、並列ルート、インターセプトルート等の高度なルーティングパターンをサポートする。

### 本機能の処理概要

App Routerは、Next.jsにおけるモダンなルーティングの中核機能であり、React Server Componentsを基盤としたサーバーサイドレンダリングとクライアントサイドナビゲーションを統合的に提供する。

**業務上の目的・背景**：従来のPages Routerでは実現が困難だった複雑なUIパターン（ネストされたレイアウト、並列ルート、ストリーミング等）を、ファイルシステムの規約に基づいて宣言的に実現する必要があった。App Routerは、React Server Componentsとの深い統合により、パフォーマンスとDXの両方を向上させる。

**機能の利用シーン**：あらゆるNext.jsアプリケーションのルーティング定義で使用される。特にネストされたレイアウト、ローディング状態、エラーバウンダリ、並列ルート等が必要な場面で威力を発揮する。

**主要な処理内容**：
1. `app/` ディレクトリ内のファイル規約（page.tsx、layout.tsx、loading.tsx、error.tsx等）に基づくルート定義の解析
2. LoaderTreeの構築によるルートセグメントのツリー構造管理
3. サーバーサイドでのコンポーネントツリー生成（createComponentTree）
4. RSC（React Server Components）ペイロードの生成とストリーミング配信
5. クライアントサイドでのルーターステート管理とソフトナビゲーション

**関連システム・外部連携**：React Server Components、React Suspense、ストリーミングレンダリング。

**権限による制御**：ルートセグメント単位でのミドルウェアによるアクセス制御が可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | グローバル404ページ (Global Not Found) | 補助機能 | App Routerのルーティング機構を通じてグローバルNot Foundページを表示 |
| 5 | 404ページ (Not Found) | 補助機能 | App Routerのセグメント単位ルーティングでのnotFound()呼び出し連携 |
| 6 | グローバルエラーページ (Global Error) | 補助機能 | App RouterのError Boundary機構との統合 |
| 8 | 403ページ (Forbidden) | 補助機能 | App Routerのforbidden()関数呼び出しとの連携 |
| 9 | 401ページ (Unauthorized) | 補助機能 | App Routerのunauthorized()関数呼び出しとの連携 |
| 10 | デフォルトレイアウト (Default Layout) | 補助機能 | App Routerのレイアウト階層構造でのフォールバックとして機能 |
| 11 | パラレルルートデフォルト | 補助機能 | App Routerのパラレルルート機構での未定義スロット処理 |
| 18 | セグメントエクスプローラー (Route Info) | 補助機能 | App Routerのセグメント構造データを取得して表示 |

## 機能種別

ルーティング / レンダリング制御

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| HTTPリクエスト | BaseNextRequest | Yes | クライアントからのHTTPリクエスト | - |
| pagePath | string | Yes | ルーティングで解決されたページパス | - |
| query | NextParsedUrlQuery | Yes | URLクエリパラメータ | - |
| renderOpts | RenderOpts | Yes | レンダリングオプション | - |
| FlightRouterState | FlightRouterState | No | クライアントルーターの現在のステート（RSCリクエスト時） | - |

### 入力データソース

- HTTPリクエスト（URL、ヘッダー、Cookie等）
- ファイルシステム（app/ディレクトリのファイル構造）
- ビルドマニフェスト（ルートマニフェスト、クライアントリファレンスマニフェスト等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| HTML | ストリーム | 初回アクセス時のHTML応答（Fizzストリーミング） |
| RSCペイロード | Flight形式 | RSCリクエスト時のReact Server Componentsペイロード |
| FlightRouterState | FlightRouterState | 更新後のルーターステート |
| CacheNodeSeedData | CacheNodeSeedData | キャッシュノードのシードデータ |

### 出力先

- HTTPレスポンスとしてクライアントに返却
- クライアントルーターキャッシュの更新

## 処理フロー

### 処理シーケンス

```
1. リクエスト受信とヘッダー解析
   └─ RSCヘッダー、プリフェッチヘッダー、ルーターステートの取得
2. ルートマッチング
   └─ URLパスからLoaderTreeの特定
3. FlightRouterStateの生成/解析
   └─ クライアントステートとサーバーステートの照合
4. コンポーネントツリーの構築
   └─ createComponentTree()でReactコンポーネントツリーを生成
5. レンダリング分岐
   └─ RSCリクエスト: Flightペイロード生成
   └─ 初回アクセス: HTMLストリーミング生成
6. メタデータ処理
   └─ generateMetadata/metadata exportの解決
7. レスポンス送信
   └─ ストリーミングまたはバッファリングされたレスポンスの送信
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B{RSCリクエスト?}
    B -->|Yes| C[FlightRouterState解析]
    B -->|No| D[初回HTMLリクエスト]
    C --> E{プリフェッチ?}
    E -->|Yes| F[プリフェッチ用Flightペイロード生成]
    E -->|No| G[フルFlightペイロード生成]
    D --> H[コンポーネントツリー構築]
    H --> I[Fizzストリーミングレンダリング]
    F --> J[レスポンス送信]
    G --> J
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ファイル規約 | page.tsx/layout.tsx/loading.tsx/error.tsx等の規約に基づいてルートを定義 | 常時 |
| BR-02 | Server Componentデフォルト | app/ディレクトリ内のコンポーネントはデフォルトでServer Component | 常時 |
| BR-03 | ルートグループ | (folder)で囲まれたディレクトリはURLパスに含まれない | 括弧付きフォルダ |
| BR-04 | 動的セグメント | [param]で囲まれたディレクトリは動的パラメータとして扱われる | 角括弧付きフォルダ |
| BR-05 | レイアウトネスト | 親ディレクトリのlayout.tsxは子ルートにも適用される | layout.tsx存在時 |

### 計算ロジック

該当なし。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし。ルーティング機構自体はデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 404 | Not Found | ルートに一致するページが存在しない | not-found.tsxまたはデフォルト404ページを表示 |
| 403 | Forbidden | forbidden()関数が呼び出された | forbidden.tsxを表示 |
| 401 | Unauthorized | unauthorized()関数が呼び出された | unauthorized.tsxを表示 |
| 500 | Server Error | コンポーネントレンダリング中のエラー | error.tsxまたはglobal-error.tsxを表示 |
| - | Redirect | redirect()関数が呼び出された | 指定URLへリダイレクト |

### リトライ仕様

クライアントサイドでのナビゲーション失敗時、ハードナビゲーションへのフォールバックが実装されている。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- ストリーミングレンダリングにより、最初のバイトまでの時間（TTFB）を最小化
- プリフェッチによるナビゲーション高速化
- React Server Componentsによるクライアントバンドルサイズ削減

## セキュリティ考慮事項

- Server Componentのコードはクライアントに送信されない
- CSRF保護機構との統合（Server Actions経由）
- ミドルウェアによるルートレベルの認証・認可

## 備考

- App RouterはReact Server ComponentsとSuspenseを前提とした設計
- Pages Routerとの共存が可能（ハイブリッドモード）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | app-router-types.ts | `packages/next/src/shared/lib/app-router-types.ts` | FlightRouterState、CacheNodeSeedData、FlightData等の型定義を理解 |
| 1-2 | app-dir-module.ts | `packages/next/src/server/lib/app-dir-module.ts` | LoaderTree型でルートセグメントのツリー構造を理解 |

**読解のコツ**: FlightRouterStateは再帰的なツリー構造であり、各ノードがセグメント名と子ノードの辞書を持つ。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | app-render.tsx | `packages/next/src/server/app-render/app-render.tsx` | renderToHTMLOrFlight関数（2349行目）がApp Routerのレンダリングエントリーポイント |

**主要処理フロー**:
1. **2349-2358行目**: renderToHTMLOrFlight関数のシグネチャ
2. **2363行目**: URLの解析
3. **2367-2373行目**: リクエストヘッダーの解析（RSC、プリフェッチ等）
4. **2380-2398行目**: postponed stateの解析（PPR対応）

#### Step 3: コンポーネントツリー構築を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | create-component-tree.tsx | `packages/next/src/server/app-render/create-component-tree.tsx` | createComponentTree関数（49-69行目）でサーバーサイドのコンポーネントツリー構築を把握 |
| 3-2 | create-component-tree.tsx | `packages/next/src/server/app-render/create-component-tree.tsx` | createComponentTreeInternal関数（83行目以降）でLoaderTreeの解析とコンポーネント生成を把握 |

**主要処理フロー**:
- **136-137行目**: parseLoaderTreeでセグメント情報を抽出
- **139-147行目**: modulesからlayout/template/error/loading/not-found等の取得
- **184-192行目**: Loading UIコンポーネントのスタイル・スクリプト解決
- **194-206行目**: LayoutまたはPageモジュールの取得
- **458行目以降**: 並列ルート（parallelRoutes）の再帰的処理

#### Step 4: クライアントサイドルーターを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | layout-router.tsx | `packages/next/src/client/components/layout-router.tsx` | OuterLayoutRouter（492行目）でクライアントサイドのルート切り替えを理解 |
| 4-2 | app-router.tsx | `packages/next/src/client/components/app-router.tsx` | App RouterのクライアントサイドルートProviderを理解 |

### プログラム呼び出し階層図

```
renderToHTMLOrFlight (app-render.tsx)
    |
    +-- parseRequestHeaders
    |
    +-- createWorkStore / createRequestStoreForRender
    |
    +-- createFlightRouterStateFromLoaderTree
    |
    +-- createComponentTree (create-component-tree.tsx)
    |      |
    |      +-- parseLoaderTree
    |      +-- getLayoutOrPageModule
    |      +-- createComponentStylesAndScripts
    |      +-- getLayerAssets
    |      +-- [再帰] createComponentTreeInternal (parallelRoutes)
    |
    +-- renderToInitialFizzStream / continueFizzStream
    |
    +-- walkTreeWithFlightRouterState (RSCペイロード生成)
```

### データフロー図

```
[入力]                       [処理]                              [出力]

HTTPリクエスト -----------> parseRequestHeaders
                                |
app/ディレクトリ ----------> LoaderTree構築
                                |
FlightRouterState --------> createComponentTree ---------> CacheNodeSeedData
(クライアント状態)              |
                           renderToInitialFizzStream -----> HTMLストリーム
                                |
                           walkTreeWithFlightRouterState -> RSCペイロード
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| app-render.tsx | `packages/next/src/server/app-render/app-render.tsx` | ソース | App Routerのレンダリングエントリーポイント |
| create-component-tree.tsx | `packages/next/src/server/app-render/create-component-tree.tsx` | ソース | コンポーネントツリー構築 |
| layout-router.tsx | `packages/next/src/client/components/layout-router.tsx` | ソース | クライアントサイドレイアウトルーター |
| app-router.tsx | `packages/next/src/client/components/app-router.tsx` | ソース | クライアントサイドApp Router |
| app-router-types.ts | `packages/next/src/shared/lib/app-router-types.ts` | ソース | 型定義 |
| app-dir-module.ts | `packages/next/src/server/lib/app-dir-module.ts` | ソース | LoaderTree型・モジュール読み込み |
| walk-tree-with-flight-router-state.tsx | `packages/next/src/server/app-render/walk-tree-with-flight-router-state.tsx` | ソース | Flightデータ生成 |
| create-flight-router-state-from-loader-tree.tsx | `packages/next/src/server/app-render/create-flight-router-state-from-loader-tree.tsx` | ソース | LoaderTreeからFlightRouterState生成 |
| parse-and-validate-flight-router-state.ts | `packages/next/src/server/app-render/parse-and-validate-flight-router-state.ts` | ソース | FlightRouterStateの検証 |
| node-web-streams-helper.ts | `packages/next/src/server/stream-utils/node-web-streams-helper.ts` | ソース | ストリーミングユーティリティ |
