# 機能設計書 17-Server Components

## 概要

本ドキュメントは、Next.js App RouterにおけるReact Server Componentsのサポート機能の設計を記述する。サーバー側でレンダリングされるコンポーネントにより、クライアントバンドルサイズを削減し、サーバーリソースへの直接アクセスを可能にする。

### 本機能の処理概要

Server Components機能は、React Server Components（RSC）をNext.jsのApp Router内で統合的にサポートする中核機能である。

**業務上の目的・背景**：従来のクライアントサイドレンダリングでは、すべてのコンポーネントコードがクライアントに送信される必要があり、バンドルサイズの増大やパフォーマンスの低下を招いていた。Server Componentsにより、サーバー上でのみ実行されるコンポーネントを定義でき、データベースアクセスやファイルシステム操作等のサーバーリソースを直接利用しながら、クライアントに送信するコードを最小化できる。

**機能の利用シーン**：データフェッチを伴うページコンポーネント、データベースクエリを直接実行するコンポーネント、大規模ライブラリ（マークダウンパーサー等）を使用するコンポーネント、静的コンテンツの表示等で利用される。

**主要な処理内容**：
1. App RouterのデフォルトとしてのServer Component判定
2. `isClientReference` によるServer/Client Component分類
3. サーバーサイドでのコンポーネントレンダリングとRSCペイロード生成
4. Client Componentとの境界でのシリアライゼーション（propsの制約）
5. RSCペイロードのストリーミング配信
6. クライアントサイドでのRSCペイロードのデシリアライゼーションとDOM構築

**関連システム・外部連携**：React Flight（RSCプロトコル）、Fizz（ストリーミングSSR）。

**権限による制御**：Server Component内でサーバーサイドの認証チェックが可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | Server Componentsは画面固有のUIではなく、すべてのApp Routerページに適用される基盤機能 |

## 機能種別

レンダリング / コンポーネント管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| コンポーネントモジュール | any | Yes | page.tsx/layout.tsx等のモジュール | isClientReferenceで判定 |
| props | シリアライザブル | Yes | コンポーネントに渡されるprops | JSON互換のシリアライズ可能な値 |
| children | ReactNode | No | 子コンポーネント（Client Componentを含むことが可能） | - |

### 入力データソース

- ファイルシステム（`app/` ディレクトリのコンポーネント）
- ビルドマニフェスト（Server/Clientモジュール分類情報）
- Client Reference Manifest（クライアントコンポーネントの参照マップ）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| RSCペイロード | Flight形式 | React Server Componentsのシリアライズされたレンダリング結果 |
| HTML | ストリーム | RSCペイロードをHTMLに変換した初期表示 |
| Client Reference | モジュール参照 | Client Componentへの参照情報 |

### 出力先

- HTTPレスポンス（RSCペイロードまたはHTML）
- クライアントサイドのReactツリー

## 処理フロー

### 処理シーケンス

```
1. モジュール判定
   └─ isClientReference()でServer/Client Componentを分類
2. Server Componentレンダリング
   └─ サーバー上でReactコンポーネントを実行
3. データフェッチ
   └─ async/awaitによるサーバーサイドデータ取得
4. シリアライゼーション
   └─ RSCペイロードへの変換（Client Component境界でのシリアライズ）
5. ストリーミング配信
   └─ RSCペイロードのチャンク送信
6. クライアントデシリアライゼーション
   └─ RSCペイロードからReactツリーの再構築
```

### フローチャート

```mermaid
flowchart TD
    A[コンポーネントモジュール] --> B{isClientReference?}
    B -->|Yes| C[Client Componentとして処理]
    B -->|No| D[Server Componentとして処理]
    D --> E[サーバーサイドレンダリング]
    E --> F[async/awaitデータフェッチ]
    F --> G[RSCペイロード生成]
    G --> H{初回アクセス?}
    H -->|Yes| I[HTML + RSCペイロード送信]
    H -->|No| J[RSCペイロードのみ送信]
    C --> K[Client Reference生成]
    K --> L[クライアントバンドルに含める]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | デフォルトServer | App Router内のコンポーネントはデフォルトでServer Component | 常時 |
| BR-02 | propsシリアライズ制約 | Server→Client境界のpropsはシリアライズ可能である必要がある（関数・クラスインスタンス不可） | SC→CC境界 |
| BR-03 | asyncサポート | Server Componentはasync/awaitを直接使用できる | Server Component内 |
| BR-04 | インポート制約 | Server ComponentからクライアントAPIは使用不可（useState、useEffect等） | Server Component内 |
| BR-05 | Client Component判定 | `$$typeof === Symbol.for('react.client.reference')` でClient Componentを判定 | モジュール読み込み時 |

### 計算ロジック

該当なし。

## データベース操作仕様

Server Component内で直接データベースアクセスが可能であるが、特定のデータベースとの結合はない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | シリアライゼーションエラー | propsに非シリアライズ可能な値がある | ビルドエラーまたはランタイムエラー |
| - | クライアントAPI使用エラー | Server Component内でuseState等を使用 | コンパイルエラー |
| - | レンダリングエラー | Server Component内の例外 | error.tsxバウンダリでキャッチ |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- Server Componentのコードはクライアントバンドルに含まれない
- 大規模な依存関係（マークダウンパーサー等）をServer Componentで使用することでバンドルサイズを削減
- RSCペイロードのストリーミングにより段階的なUI表示を実現

## セキュリティ考慮事項

- Server Componentのソースコードはクライアントに送信されない
- サーバーサイドの秘密情報（API key等）をServer Component内で安全に使用可能
- ただし、propsとしてClient Componentに渡したデータはクライアントに公開される

## 備考

- React Flight プロトコルによりServer Componentのレンダリング結果がシリアライズされる
- Server ComponentとClient Componentのツリーは任意に入れ子にできる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | client-and-server-references.ts | `packages/next/src/lib/client-and-server-references.ts` | isClientReference関数（45-48行目）、isServerReference関数（15-19行目）でServer/Client判定の仕組みを理解 |
| 1-2 | app-router-types.ts | `packages/next/src/shared/lib/app-router-types.ts` | RSCPayload型、FlightData型を確認 |

**読解のコツ**: `$$typeof === Symbol.for('react.client.reference')` がClient Componentの識別子。'use client'ディレクティブがない場合、デフォルトでServer Componentとなる。

#### Step 2: コンポーネント判定とツリー構築

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | create-component-tree.tsx | `packages/next/src/server/app-render/create-component-tree.tsx` | isClientComponent判定（741行目）とClientPageRoot/ClientSegmentRootの使い分け |

**主要処理フロー**:
1. **8-10行目**: isClientReference、isUseCacheFunctionのインポート
2. **741行目**: `isClientComponent = isClientReference(layoutOrPageMod)` でClient判定
3. **768-786行目**: Client Component時はClientPageRootでラップ
4. **878-896行目**: Client Component時はClientSegmentRootでラップ

#### Step 3: RSCペイロード生成

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | app-render.tsx | `packages/next/src/server/app-render/app-render.tsx` | renderToHTMLOrFlight（2349行目）でのレンダリング分岐 |
| 3-2 | flight-render-result.ts | `packages/next/src/server/app-render/flight-render-result.ts` | FlightRenderResult型でRSCペイロードの出力形式を理解 |

#### Step 4: マニフェスト参照

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | manifests-singleton.ts | `packages/next/src/server/app-render/manifests-singleton.ts` | getClientReferenceManifest、getServerModuleMapでモジュール参照を管理 |

### プログラム呼び出し階層図

```
renderToHTMLOrFlight (app-render.tsx)
    |
    +-- createComponentTree
    |      |
    |      +-- isClientReference (client-and-server-references.ts)
    |      |      └─ $$typeof判定でServer/Client分類
    |      |
    |      +-- Server Component: 直接レンダリング
    |      |
    |      +-- Client Component: ClientPageRoot/ClientSegmentRootでラップ
    |
    +-- React Flight シリアライゼーション
    |      └─ getClientReferenceManifest (manifests-singleton.ts)
    |      └─ getServerModuleMap
    |
    +-- renderToInitialFizzStream / FlightRenderResult
```

### データフロー図

```
[入力]                        [処理]                          [出力]

コンポーネントモジュール ---> isClientReference判定
                                   |
                              Server Component --------> サーバーサイドレンダリング ----> RSCペイロード
                                   |                                                         |
                              Client Component --------> Client Reference生成 ----------> クライアントバンドル
                                   |
                              React Flight シリアライゼーション --------> HTMLストリーム + RSCペイロード
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| client-and-server-references.ts | `packages/next/src/lib/client-and-server-references.ts` | ソース | Server/Client Reference判定 |
| create-component-tree.tsx | `packages/next/src/server/app-render/create-component-tree.tsx` | ソース | コンポーネントツリー構築とSC/CC分岐 |
| app-render.tsx | `packages/next/src/server/app-render/app-render.tsx` | ソース | RSCレンダリングエントリーポイント |
| flight-render-result.ts | `packages/next/src/server/app-render/flight-render-result.ts` | ソース | RSCペイロード結果型 |
| manifests-singleton.ts | `packages/next/src/server/app-render/manifests-singleton.ts` | ソース | マニフェスト管理 |
| interop-default.ts | `packages/next/src/server/app-render/interop-default.ts` | ソース | デフォルトexport互換処理 |
