# 機能設計書 28-getStaticProps

## 概要

本ドキュメントは、Next.js Pages RouterにおけるgetStaticProps機能の設計を記述する。getStaticPropsは、ビルド時（またはISR再検証時）にサーバーサイドでデータを取得し、静的ページのpropsとして事前レンダリングするAPIである。

### 本機能の処理概要

**業務上の目的・背景**：多くのWebページは頻繁に変更されないデータを表示する。getStaticPropsは、ビルド時にデータを取得してHTMLを事前生成するSSG（Static Site Generation）を実現し、CDN配信による高速なページロードを可能にする。ISR（Incremental Static Regeneration）と組み合わせることで、デプロイ後も段階的にページを再生成できる。

**機能の利用シーン**：ブログ記事の一覧・詳細ページ、商品カタログページ、ドキュメントサイト、CMSからのコンテンツ表示、マーケティングページなど、比較的変更頻度が低いコンテンツの表示。

**主要な処理内容**：
1. ビルド時（またはISR再検証時）にgetStaticProps関数を実行
2. 戻り値のバリデーション（props/redirect/notFoundの排他チェック）
3. revalidateオプションによるISR間隔の設定
4. propsのシリアライズとHTMLの事前レンダリング
5. 動的ルートでのgetStaticPathsとの連携

**関連システム・外部連携**：Pages Routerのビルドシステム、ISR（Incremental Static Regeneration）、CDNキャッシュ、外部CMS/APIとの連携。

**権限による制御**：プレビューモード（Draft Mode）により、下書きコンテンツをSSGページ上でプレビュー可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | getStaticPropsはPages Routerの全SSGページに適用される |

## 機能種別

データ取得 / SSGデータフェッチ / ビルド時処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| context.params | Record<string, string> | No | 動的ルートパラメータ | 動的ルートの場合 |
| context.preview | boolean | No | プレビューモードフラグ | Draft Mode有効時true |
| context.previewData | PreviewData | No | プレビューデータ | Draft Mode有効時 |
| context.locale | string | No | 現在のロケール | i18n有効時 |
| context.locales | string[] | No | サポートロケール一覧 | i18n有効時 |
| context.defaultLocale | string | No | デフォルトロケール | i18n有効時 |

### 入力データソース

- ビルド時のコンテキスト情報
- 外部API/CMS/データベース（ユーザー実装部分）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| props | Record<string, any> | ページコンポーネントに渡すprops（シリアライズ可能） |
| revalidate | number \| false | ISR再検証間隔（秒）。falseの場合は再検証なし |
| redirect | { destination: string, permanent: boolean } | リダイレクト指定 |
| notFound | boolean | 404表示指定 |

### 出力先

- ビルド時に生成されるHTMLファイルと`.json`データファイル
- ISR再検証時に更新されるキャッシュ

## 処理フロー

### 処理シーケンス

```
1. ビルド時実行
   └─ ビルドプロセスが各SSGページのgetStaticPropsを実行
2. コンテキスト構築
   └─ params、preview情報、locale情報を含むコンテキストを構築
3. getStaticProps実行
   └─ ユーザー定義関数にコンテキストを渡して実行
4. 戻り値バリデーション
   └─ props/redirect/notFoundの排他チェック、revalidateの型チェック
5. プレレンダリング
   └─ 取得したpropsでHTMLを事前生成
6. ISR設定
   └─ revalidateオプションが設定されている場合、ISR間隔を記録
```

### フローチャート

```mermaid
flowchart TD
    A[ビルド開始 / ISR再検証] --> B[コンテキスト構築]
    B --> C["getStaticProps(context)"]
    C --> D{戻り値}
    D -->|redirect| E[リダイレクト設定]
    D -->|notFound: true| F[fallbackページ生成]
    D -->|props| G[propsバリデーション]
    G --> H{revalidate設定?}
    H -->|number| I[ISR間隔記録]
    H -->|false/未設定| J[静的ページとして固定]
    I --> K[HTML + JSONファイル生成]
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-28-01 | ビルド時実行 | getStaticPropsはビルド時（またはISR再検証時）に実行される | 常時 |
| BR-28-02 | サーバー専用 | getStaticPropsのコードはクライアントバンドルに含まれない | 常時 |
| BR-28-03 | revalidate型制限 | revalidateはnumber（秒）またはfalse | 常時 |
| BR-28-04 | UNSTABLE_REVALIDATE互換 | `unstable_revalidate`は非推奨、`revalidate`を使用 | 旧APIからの移行時 |
| BR-28-05 | getStaticPaths連携 | 動的ルートではgetStaticPathsが必須 | isDynamic && isSSG |
| BR-28-06 | propsシリアライズ | propsはJSON.stringifyでシリアライズ可能でなければならない | 常時 |

### 計算ロジック

ISR再検証間隔：`revalidate: 60` は60秒ごとにページを再生成。`revalidate: false`は再検証なし（ビルド時のみ生成）。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし。getStaticProps自体はデータベースに直接アクセスしないが、ユーザー実装部分でデータベースアクセスが行われる。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | GSP_NO_RETURNED_VALUE | 戻り値がオブジェクトでない | props/redirect/notFoundを含むオブジェクトを返す |
| - | SSG_GET_INITIAL_PROPS_CONFLICT | getInitialPropsと共存 | いずれかを削除 |
| - | UNSTABLE_REVALIDATE_RENAME_ERROR | 非推奨のunstable_revalidateを使用 | revalidateに変更 |
| - | シリアライズエラー | propsがシリアライズ不可 | 関数、Date、undefined等を除外 |

### リトライ仕様

ISR再検証時にgetStaticPropsが失敗した場合、以前のキャッシュが引き続き配信される（stale-while-revalidate戦略）。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- ビルド時に事前生成されたHTMLはCDN配信に最適化
- ISRにより再デプロイなしでコンテンツ更新が可能
- `revalidate`の最小値はフレームワークの制約なし（1秒から設定可能）

## セキュリティ考慮事項

- getStaticProps内のコードはサーバーサイドのみで実行
- ビルド時に生成されたHTMLとJSONは公開ディレクトリに配置されるため、propsに機密情報を含めない
- プレビューモードのCookieベースのアクセス制御

## 備考

- App RouterではServer Componentsの直接的なデータフェッチが推奨
- getStaticPropsはPages Router専用のAPI
- `fallback`オプション（getStaticPaths）との組み合わせで、ビルド時に生成されていないページのオンデマンド生成が可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

getStaticPropsの型定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | types.ts | `packages/next/src/types.ts` | `GetStaticProps`型と`GetStaticPropsContext`型、`GetStaticPropsResult`型 |

**読解のコツ**: `GetStaticPropsResult<P>`は`{ props: P, revalidate?: number | false }`、`{ redirect: Redirect }`、`{ notFound: true }`のunion型。

#### Step 2: ビルド時の実行フローを理解する

ビルドプロセスでのgetStaticProps実行。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | render.tsx | `packages/next/src/server/render.tsx` | renderToHTMLImpl内でのgetStaticProps呼び出しフロー |

**主要処理フロー**:
1. **519行目**: `isSSG = !!getStaticProps`で判定
2. **565行目**: getInitialPropsとの共存チェック
3. **583-601行目**: getStaticPaths関連のバリデーション

#### Step 3: ISR再検証を理解する

Incremental Static Regenerationの再検証メカニズム。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | render.tsx | `packages/next/src/server/render.tsx` | revalidateオプションの処理と検証 |

### プログラム呼び出し階層図

```
next build [src/build/index.ts]
    |
    +-- renderToHTMLImpl() [render.tsx]
            |
            +-- isSSG判定
            +-- getStaticPaths連携チェック
            |
            +-- getStaticProps(context) ← ユーザー定義関数
            |       +-- 戻り値バリデーション
            |       +-- revalidate設定記録
            |       +-- isSerializableProps() 検証
            |
            +-- プレレンダリング
                    +-- HTML + JSON生成
```

### データフロー図

```
[入力]                       [処理]                             [出力]

ビルドコンテキスト      ──> getStaticProps(context)          ──> {props, revalidate}
(params, locale等)           |
                             |
{props}                ──> isSerializableProps() 検証        ──> validatedProps
                             |
validatedProps         ──> プレレンダリング                  ──> .html + .json ファイル
                             |
revalidate設定         ──> ISR設定記録                       ──> prerender-manifest.json
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| render.tsx | `packages/next/src/server/render.tsx` | ソース | getStaticProps実行とプレレンダリング |
| types.ts | `packages/next/src/types.ts` | ソース | GetStaticProps型定義 |
| module.ts | `packages/next/src/server/route-modules/pages/module.ts` | ソース | PagesUserlandModuleでのgetStaticProps参照 |
| is-serializable-props.ts | `packages/next/src/lib/is-serializable-props.ts` | ソース | propsのシリアライズ検証 |
| index.ts | `packages/next/src/build/index.ts` | ソース | ビルドプロセスでのSSGページ生成 |
| constants.ts | `packages/next/src/lib/constants.ts` | ソース | エラーメッセージ定数 |
