# 機能設計書 33-カスタムエラーページ

## 概要

本ドキュメントは、Next.jsのPages Routerにおけるカスタムエラーページ（`_error.tsx`）機能の設計を記述する。HTTPステータスコードに応じたエラーメッセージの表示を行うデフォルトエラーページコンポーネントであり、開発者がカスタマイズ可能である。

### 本機能の処理概要

**業務上の目的・背景**：Webアプリケーションにおいてエラー発生時にユーザーに適切なフィードバックを提供することはUXの観点で重要である。カスタムエラーページは、HTTPエラー（400、404、405、500等）に対応する表示を提供し、ブランディングやナビゲーションを含むカスタムエラー画面の実装を可能にする。

**機能の利用シーン**：サーバーサイドでエラーが発生した場合（500エラー）、存在しないページへのアクセス（404エラー）、不正なリクエスト（400エラー）、許可されていないHTTPメソッドの使用（405エラー）、クライアントサイドでのランタイムエラー発生時に表示される。

**主要な処理内容**：
1. `getInitialProps`によるステータスコードの判定（res.statusCode / err.statusCode）
2. ステータスコードに対応するエラーメッセージの表示
3. ダークモード対応のスタイリング
4. ホスト名情報の取得と表示（クライアントサイドエラー時）

**関連システム・外部連携**：Pages Routerのルーティング機構、サーバーサイドレンダリング（SSR）と連携する。

**権限による制御**：権限による制御は存在しない。全てのユーザーに対して同一のエラー表示を行う。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | エラーページ (_error) | 主画面 | HTTPステータスコードに応じたエラーメッセージ表示の主処理 |

## 機能種別

エラー表示 / UI表示

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| statusCode | number | Yes | HTTPステータスコード | 正の整数 |
| hostname | string | No | リクエスト元のホスト名 | - |
| title | string | No | カスタムエラータイトル | - |
| withDarkMode | boolean | No | ダークモード対応の有無（デフォルト: true） | - |

### 入力データソース

- `getInitialProps`のコンテキスト（NextPageContext）からres.statusCode、err.statusCode
- ブラウザの`window.location.hostname`（クライアントサイド）
- サーバーサイドでは`getRequestMeta(req, 'initURL')`からホスト名を取得

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ReactElement | JSX.Element | エラー表示UI（ステータスコード + エラーメッセージ） |
| ErrorProps | ErrorProps | statusCode、hostname、title、withDarkModeを含むprops |

### 出力先

- サーバーサイドレンダリング経由でクライアントに送信されるHTML

## 処理フロー

### 処理シーケンス

```
1. エラー発生の検知
   └─ サーバーサイド: res.statusCodeまたはerr.statusCodeから取得
   └─ クライアントサイド: err.statusCodeから取得（デフォルト404）
2. getInitialPropsの実行
   └─ statusCodeの決定: res.statusCode > err.statusCode > 404
   └─ hostnameの取得（クライアント: window.location、サーバー: initURL）
3. Errorコンポーネントのレンダリング
   └─ ステータスコードに対応するメッセージの選択
   └─ ダークモード対応CSSの注入
   └─ エラー表示UIの描画
```

### フローチャート

```mermaid
flowchart TD
    A[エラー発生] --> B{サーバーサイド?}
    B -->|Yes| C[res.statusCodeを確認]
    B -->|No| D[err.statusCodeを確認]
    C --> E{statusCode存在?}
    E -->|Yes| F[statusCode = res.statusCode]
    E -->|No| G{err存在?}
    D --> G
    G -->|Yes| H[statusCode = err.statusCode]
    G -->|No| I[statusCode = 404]
    F --> J[hostname取得]
    H --> J
    I --> J
    J --> K[Errorコンポーネントレンダリング]
    K --> L{statusCode存在?}
    L -->|Yes| M[statusCode: title 形式で表示]
    L -->|No| N[クライアントサイドエラーメッセージ表示]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-33-01 | ステータスコード優先順位 | res.statusCode > err.statusCode > 404の優先順位でステータスコードを決定する | 常時 |
| BR-33-02 | デフォルトメッセージマッピング | 400="Bad Request"、404="This page could not be found"、405="Method Not Allowed"、500="Internal Server Error" | statusCodesマップに定義されたコード |
| BR-33-03 | ダークモードデフォルト有効 | withDarkModeはデフォルトでtrueであり、prefers-color-schemeメディアクエリに対応する | withDarkModeが未指定またはtrue |

### 計算ロジック

ステータスコード決定ロジック:
```
statusCode = (res && res.statusCode) ? res.statusCode : (err ? err.statusCode : 404)
```

## データベース操作仕様

該当なし。カスタムエラーページはデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | Bad Request | 不正なリクエスト | エラーメッセージを表示 |
| 404 | Not Found | ページが見つからない | エラーメッセージを表示 |
| 405 | Method Not Allowed | 許可されていないHTTPメソッド | エラーメッセージを表示 |
| 500 | Internal Server Error | サーバー内部エラー | エラーメッセージを表示 |

### リトライ仕様

リトライは行わない。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- エラーページはシンプルなReactコンポーネントであり、外部依存を持たないため、高速にレンダリングされる
- インラインCSSを使用しており、追加のCSS読み込みは不要

## セキュリティ考慮事項

- サーバーサイドのエラー詳細情報（スタックトレース等）はクライアントに送信しない
- ホスト名情報はクライアントサイドエラー時のデバッグ目的でのみ使用される

## 備考

- Pages Routerの`_error.tsx`に相当するApp Routerの機能は`error.tsx`である
- 404ページ専用のカスタマイズには`pages/404.tsx`を使用することも可能
- `displayName`は`'ErrorPage'`に設定されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | _error.tsx | `packages/next/src/pages/_error.tsx` | ErrorProps型定義を確認する |

**読解のコツ**: **12-17行目**に`ErrorProps`型が定義されている。`statusCode`（必須）、`hostname`（オプション）、`title`（オプション）、`withDarkMode`（オプション、デフォルトtrue）の4つのプロパティを持つ。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | _error.tsx | `packages/next/src/pages/_error.tsx` | Errorクラスの実装全体を確認する |

**主要処理フロー**:
1. **5-10行目**: `statusCodes`マップ - HTTPステータスコードとメッセージの対応定義
2. **19-43行目**: `_getInitialProps`関数 - ステータスコードの判定ロジック。**24-25行目**でres.statusCode、err.statusCodeの優先順位を処理。**29-40行目**でhostnameの取得（クライアント: window.location.hostname、サーバー: getRequestMetaからinitURL経由）
3. **81-156行目**: `Error`クラス - render()メソッドでエラーUIを描画。**88-92行目**でtitleの決定、**94-154行目**でReactコンポーネントとしてのレンダリング

#### Step 3: スタイリングを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | _error.tsx | `packages/next/src/pages/_error.tsx` | スタイル定義を確認する |

**主要処理フロー**:
- **45-76行目**: CSSスタイル定義。Flexboxによる中央配置、system-uiフォントファミリー使用
- **104-129行目**: インラインCSS。ダークモード時の`prefers-color-scheme: dark`メディアクエリ対応

### プログラム呼び出し階層図

```
PagesRouteModule.render()
    │
    ├─ renderToHTMLImpl()
    │      ├─ Error.getInitialProps(ctx)
    │      │      ├─ res.statusCode / err.statusCode → statusCode決定
    │      │      └─ getRequestMeta(req, 'initURL') → hostname取得
    │      │
    │      └─ App.render()
    │             └─ Error.render()
    │                    ├─ Head → <title>設定
    │                    └─ エラーUI描画
    │
    └─ Document.getInitialProps()
```

### データフロー図

```
[入力]                     [処理]                      [出力]

res.statusCode ────▶ _getInitialProps() ─▶ ErrorProps
err.statusCode ────▶                      │
req (initURL) ─────▶                      ├─ statusCode
window.location ───▶                      └─ hostname
                                                │
ErrorProps ────────▶ Error.render() ────▶ エラー表示UI
statusCodes map ───▶                      │
                                          └─ HTML + CSS
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| _error.tsx | `packages/next/src/pages/_error.tsx` | ソース | デフォルトエラーページコンポーネント |
| utils.ts | `packages/next/src/shared/lib/utils.ts` | ソース | NextPageContext型定義 |
| head.tsx | `packages/next/src/shared/lib/head.tsx` | ソース | next/headコンポーネント（titleタグ設定用） |
| request-meta.ts | `packages/next/src/server/request-meta.ts` | ソース | getRequestMeta関数（サーバーサイドURL取得） |
| render.tsx | `packages/next/src/server/render.tsx` | ソース | ページレンダリング処理 |
