# 機能設計書 35-静的サイト生成（SSG）

## 概要

本ドキュメントは、Next.jsにおける静的サイト生成（SSG: Static Site Generation）機能の設計を記述する。SSGはビルド時にHTMLを事前生成し、CDN配信に最適化されたページを提供する機能である。

### 本機能の処理概要

**業務上の目的・背景**：コンテンツがリクエストごとに変化しないページ（ブログ記事、製品ページ、ドキュメント等）では、ビルド時に静的HTMLを生成することで、CDNからの高速配信とサーバー負荷の軽減を実現できる。SSGはSSRに比べてTTFBが大幅に短縮され、最高のパフォーマンスを提供する。

**機能の利用シーン**：Pages Routerでは`getStaticProps`と`getStaticPaths`を使用して静的ページを生成する。動的ルートでは`getStaticPaths`で事前生成するパスを定義し、`getStaticProps`でページデータを取得する。ビルド時（`next build`）に実行される。

**主要な処理内容**：
1. ビルド時の`getStaticPaths`実行による事前生成パスの決定
2. 各パスに対する`getStaticProps`の実行とデータ取得
3. ReactコンポーネントのHTMLへのレンダリング
4. 生成されたHTMLとJSONデータの`.next`ディレクトリへの保存
5. fallback処理（blocking / true / false）による未生成パスの処理

**関連システム・外部連携**：ビルドシステム（`next build`）、プリレンダーマニフェスト、ファイルシステムキャッシュ。

**権限による制御**：SSG自体に権限制御は存在しない。ビルド時に実行されるため、ランタイムの認証は不要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 17 | ルート情報パネル (Route Type) | 参照画面 | Staticルートの判定情報を表示 |

## 機能種別

ビルド処理 / データ取得 / HTML生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| page | string | Yes | ページパス | 有効なルートパス |
| getStaticProps | GetStaticProps | Yes | 静的データ取得関数 | - |
| getStaticPaths | GetStaticPaths | No | 動的ルートの事前生成パス定義関数 | 動的ルートの場合は必須 |
| locales | string[] | No | i18nロケール一覧 | - |
| defaultLocale | string | No | デフォルトロケール | - |

### 入力データソース

- `getStaticProps`の戻り値（props、revalidate、redirect、notFound）
- `getStaticPaths`の戻り値（paths、fallback）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| HTMLファイル | string | 事前生成された静的HTMLページ |
| JSONファイル | string | ページデータ（getStaticPropsの戻り値） |
| prerender-manifest.json | PrerenderManifest | プリレンダー設定（revalidate値等） |

### 出力先

- `.next/server/pages/` ディレクトリ（ビルド出力）
- CDN経由でクライアントに配信

## 処理フロー

### 処理シーケンス

```
1. next buildの実行
   └─ ページの解析とデータフェッチ方式の判定
2. getStaticPathsの実行（動的ルートの場合）
   └─ paths配列の取得
   └─ fallback値の決定
3. 各パスに対するgetStaticPropsの実行
   └─ データの取得
   └─ revalidate値の確認
4. Reactコンポーネントのレンダリング
   └─ 取得したデータでHTMLを生成
5. 出力ファイルの保存
   └─ HTML、JSON、prerender-manifestの出力
```

### フローチャート

```mermaid
flowchart TD
    A[next build] --> B{動的ルート?}
    B -->|Yes| C[getStaticPaths実行]
    B -->|No| D[単一パスで生成]
    C --> E[paths配列取得]
    E --> F{各pathに対して}
    F --> G[getStaticProps実行]
    D --> G
    G --> H{戻り値チェック}
    H -->|redirect| I[リダイレクト設定]
    H -->|notFound| J[404ページ生成]
    H -->|props| K[HTMLレンダリング]
    K --> L[HTML + JSONファイル保存]
    L --> M[prerender-manifest更新]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-35-01 | getStaticPathsの必須性 | 動的ルート（[param]）でSSGを使用する場合、getStaticPathsは必須 | 動的ルートかつgetStaticProps定義時 |
| BR-35-02 | fallbackモード | false: 未定義パスは404、true: フォールバックページ表示後クライアント生成、blocking: SSR風に生成 | getStaticPathsのfallback値 |
| BR-35-03 | パスパラメータ検証 | getStaticPathsが返すパスパラメータはルート定義と一致する必要がある | 動的ルート |
| BR-35-04 | 戻り値形式 | getStaticPathsは`{ paths: [], fallback: boolean \| 'blocking' }`を返す必要がある | 常時 |
| BR-35-05 | SSGとgetInitialPropsの排他 | getStaticPropsとgetInitialPropsは同時に使用できない | 同一ページ |

### 計算ロジック

revalidate値が指定された場合は、ISR（Incremental Static Regeneration）として動作する。

## データベース操作仕様

該当なし。SSG自体はデータベース操作を行わないが、`getStaticProps`内で開発者がデータベースアクセスを実装する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Error | getStaticPathsの戻り値が無効な形式の場合 | `{ paths: [], fallback: boolean }`を返す |
| - | Error | 動的SSGページにgetStaticPathsが未定義の場合 | getStaticPathsを定義する |
| - | Error | getStaticPathsが非動的ページで定義された場合 | 動的ルートでのみ使用する |
| - | Error | getStaticPathsがgetStaticPropsなしで定義された場合 | getStaticPropsも定義する |
| - | Error | パスパラメータがルート定義と不一致の場合 | 正しいパラメータ名を使用する |

### リトライ仕様

ビルド時にリトライは行わない。エラー発生時はビルドが失敗する。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- ビルド時間はgetStaticPathsのパス数とgetStaticPropsの処理時間に依存する
- 配信時はCDNキャッシュにより最速のTTFBを実現
- revalidateを指定することでISRによる段階的再生成が可能

## セキュリティ考慮事項

- getStaticPropsはビルド時にのみ実行されるため、秘密情報をビルド環境変数として安全に使用可能
- 生成されたHTML/JSONは公開されるため、機密情報を含めないこと

## 備考

- App Routerでは、`generateStaticParams`がgetStaticPathsに相当する
- `revalidate`オプションを指定するとISR（No.36）として動作する
- `fallback: 'blocking'`は将来的に非推奨となる可能性がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | types.ts | `packages/next/src/build/static-paths/types.ts` | StaticPathsResult、PrerenderedRouteの型定義を確認する |

**読解のコツ**: SSGに関わる型定義は複数ファイルに分散している。`GetStaticProps`と`GetStaticPaths`の型は`packages/next/src/types.ts`に定義されている。

#### Step 2: getStaticPathsの実行を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | pages.ts | `packages/next/src/build/static-paths/pages.ts` | buildPagesStaticPathsの実装を確認する |

**主要処理フロー**:
1. **12-24行目**: `buildPagesStaticPaths`関数のシグネチャ - page、getStaticPaths、locales等を受け取る
2. **26-27行目**: ルート正規表現とマッチャーの取得
3. **31-37行目**: getStaticPathsの実行（localesとdefaultLocaleを引数として渡す）
4. **43-51行目**: 戻り値の基本検証
5. **65-75行目**: fallback値の検証（boolean または 'blocking'）

#### Step 3: レンダリングフローを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | render.tsx | `packages/next/src/server/render.tsx` | SSG判定とレンダリング処理を確認する |

**主要処理フロー**:
- **519行目**: `isSSG = !!getStaticProps` - SSG判定
- **520行目**: `isBuildTimeSSG = isSSG && renderOpts.isBuildTimePrerendering` - ビルド時SSG判定
- **565-601行目**: データフェッチ方式の競合チェック

### プログラム呼び出し階層図

```
next build
    │
    ├─ buildPagesStaticPaths()
    │      ├─ getRouteRegex() / getRouteMatcher()
    │      └─ getStaticPaths({ locales, defaultLocale })
    │
    └─ renderToHTMLImpl() [各パスに対して]
           ├─ getStaticProps(ctx)
           ├─ renderPageTree(App, Component, props)
           └─ HTML / JSON出力
```

### データフロー図

```
[入力]                      [処理]                        [出力]

ルート定義 ──────▶ getStaticPaths() ──▶ paths配列
                                           │
各path ─────────▶ getStaticProps() ──▶ props
                                           │
props + ────────▶ renderToHTMLImpl() ──▶ HTMLファイル
Component                                  │
                                      __NEXT_DATA__ ──▶ JSONファイル
                                           │
revalidate値 ───▶ prerender-manifest ──▶ manifest.json
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| pages.ts | `packages/next/src/build/static-paths/pages.ts` | ソース | Pages Router用getStaticPaths実行 |
| app.ts | `packages/next/src/build/static-paths/app.ts` | ソース | App Router用generateStaticParams実行 |
| types.ts | `packages/next/src/build/static-paths/types.ts` | ソース | SSG関連型定義 |
| utils.ts | `packages/next/src/build/static-paths/utils.ts` | ソース | SSGユーティリティ |
| render.tsx | `packages/next/src/server/render.tsx` | ソース | SSRレンダリング（SSGのHTML生成でも使用） |
| index.ts | `packages/next/src/build/index.ts` | ソース | ビルドプロセスの中核 |
