# 機能設計書 63-コード分割

## 概要

本ドキュメントは、Next.jsのコード分割（Code Splitting）機能の設計について記述する。コード分割は、ビルド時にJavaScriptバンドルをページ・ルート単位で自動的に分割し、ユーザーが必要とするコードのみを配信する仕組みである。

### 本機能の処理概要

Next.jsのコード分割は、Webpack/Turbopack/Rspackのバンドラーが提供するコード分割機能を活用し、ページ単位・ルート単位で自動的にJavaScriptチャンクを分離する機能である。

**業務上の目的・背景**：シングルページアプリケーション（SPA）では、全ページのコードを1つのバンドルにまとめると初回ロードが著しく遅くなる。コード分割により、各ページで必要最小限のJavaScriptのみを配信し、初回ロード時間とTime to Interactive（TTI）を大幅に改善する。

**機能の利用シーン**：Next.jsアプリケーションのすべてのビルドで自動的に適用される。開発者は特別な設定なしにコード分割の恩恵を受けられる。追加で`next/dynamic`を使用することで、コンポーネント単位のより細かい分割も可能。

**主要な処理内容**：
1. ページ単位の自動コード分割：各ページのエントリーポイントから依存グラフを辿り、ページ固有のチャンクを生成
2. 共有モジュールのチャンク分離：複数ページで共有されるモジュールをcommons/frameworkチャンクに分離
3. ランタイムチャンクの分離：Webpackランタイムコードを専用チャンクに分離
4. CSS分割：ページごとのCSS抽出と分離
5. App Routerでのルートセグメント単位分割：レイアウト・ページ・コンポーネントのセグメント単位チャンク生成

**関連システム・外部連携**：Webpack SplitChunksPlugin、Turbopackのチャンク分割機能、Rspackの分割設定と連携する。

**権限による制御**：特になし。ビルド時に自動的に実行される。

## 関連画面

コード分割は全ページに影響を与えるビルド時処理であり、特定の画面には紐づかない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | 全ページ/ルート | 対象 | ページ遷移時に必要なチャンクのみをロード |

## 機能種別

ビルド最適化 / バンドル分割

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| splitChunks設定 | object | No | next.config.jsのwebpack設定でカスタマイズ可能 | Webpack SplitChunksPlugin仕様に準拠 |
| experimental.optimizePackageImports | string[] | No | 特定パッケージのインポート最適化 | パッケージ名の配列 |

### 入力データソース

- ページファイル（pages/ディレクトリまたはapp/ディレクトリのエントリーポイント）
- next.config.jsのwebpack設定（カスタマイズ時）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ページチャンク | .js ファイル | 各ページ固有のJavaScriptバンドル |
| commonチャンク | .js ファイル | 複数ページで共有されるモジュールのバンドル |
| frameworkチャンク | .js ファイル | React等のフレームワークライブラリのバンドル |
| runtimeチャンク | .js ファイル | Webpackランタイムコード |
| CSSチャンク | .css ファイル | ページごとのスタイルシート |

### 出力先

`.next/static/chunks/`ディレクトリに各チャンクファイルが出力される。

## 処理フロー

### 処理シーケンス

```
1. ビルド開始（next build）
   └─ Webpack/Turbopack設定の構築
2. エントリーポイントの収集
   └─ pages/またはapp/ディレクトリからページファイルを列挙
3. 依存グラフの構築
   └─ 各エントリーポイントからモジュール依存関係を解析
4. SplitChunksPluginによるチャンク分割
   └─ 共有モジュールの検出とchunkGroupへの割り当て
5. チャンクの最適化
   └─ 最小サイズ・最大サイズ制約に基づくチャンクの結合/分離
6. ビルドマニフェストの生成
   └─ 各ページに必要なチャンクの対応表を記録
7. チャンクファイルの出力
   └─ .next/static/chunks/配下にファイル出力
```

### フローチャート

```mermaid
flowchart TD
    A[next build開始] --> B[エントリーポイント収集]
    B --> C[依存グラフ構築]
    C --> D[SplitChunksPlugin実行]
    D --> E{チャンク種別判定}
    E -->|フレームワーク| F[framework-*.js]
    E -->|共有モジュール| G[commons-*.js]
    E -->|ページ固有| H[pages/page-*.js]
    E -->|ランタイム| I[webpack-*.js]
    F --> J[ビルドマニフェスト生成]
    G --> J
    H --> J
    I --> J
    J --> K[.next/static/chunks/に出力]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-63-01 | ページ単位自動分割 | 各ページは独立したチャンクとして生成される | 全てのページ |
| BR-63-02 | フレームワークチャンク分離 | react、react-dom等のフレームワークライブラリは専用チャンクに分離される | Webpackビルド時 |
| BR-63-03 | 共有モジュール閾値 | 2つ以上のページで使用されるモジュールは共有チャンクに分離される | SplitChunksPlugin設定による |
| BR-63-04 | CSS自動分割 | 各ページのCSSは独立したスタイルシートとして抽出される | CSS importを含むページ |

### 計算ロジック

Webpack SplitChunksPluginのデフォルト設定に基づく分割基準：
- `chunks: 'all'` - 全ての種類のチャンクを分割対象とする
- `minSize` / `maxSize` - チャンクの最小/最大サイズ制約
- `minChunks: 2` - 2つ以上のチャンクで共有されるモジュールを分離

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ビルドエラー | チャンク分割時にモジュール解決に失敗した場合 | ビルドログでエラー箇所を特定し、依存関係を修正 |

### リトライ仕様

該当なし。ビルドプロセスの再実行が必要。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- ページ単位のチャンク分割により、各ページの初回ロード時間を最小化する
- フレームワークチャンクの分離により、ブラウザキャッシュの効率を向上させる
- 共有モジュールチャンクにより、ページ遷移時の追加ダウンロード量を削減する

## セキュリティ考慮事項

- Subresource Integrity（SRI）が有効な場合、各チャンクにハッシュが付与される

## 備考

- Next.jsではコード分割はゼロコンフィグで動作する
- `next/dynamic`を使用することで、ページ内のコンポーネント単位の追加分割が可能
- App Routerではルートセグメント単位でのより細粒度の分割が行われる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: Webpack設定を理解する

コード分割の中核はWebpack設定のSplitChunks設定にある。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | webpack-config.ts | `packages/next/src/build/webpack-config.ts` | getBaseWebpackConfig関数内のsplitChunks設定を確認。cacheGroups定義がチャンク分割ルールを決定 |

**読解のコツ**: Webpackの`optimization.splitChunks`設定はcacheGroupsの中で分割ルールを定義している。`framework`グループ（React等）、`lib`グループ（共有ライブラリ）、`commons`グループ（共通モジュール）の優先順位に注目。

#### Step 2: エントリーポイント構成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | entries.ts | `packages/next/src/build/entries.ts` | createEntrypoints関数。ページファイルからWebpackエントリーポイントを生成する |

**読解のコツ**: `runDependingOnPageType`関数が、App Router（app/）とPages Router（pages/）のエントリーポイント生成を分岐する。

#### Step 3: ビルドマニフェスト生成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | build-manifest-plugin.ts | `packages/next/src/build/webpack/plugins/build-manifest-plugin.ts` | BuildManifestPlugin。各ページに必要なチャンクの対応表を生成 |

### プログラム呼び出し階層図

```
next build [src/build/index.ts]
    │
    ├─ getBaseWebpackConfig() [src/build/webpack-config.ts]
    │      ├─ optimization.splitChunks設定
    │      │      ├─ cacheGroups.framework（React等）
    │      │      ├─ cacheGroups.lib（共有ライブラリ）
    │      │      └─ cacheGroups.commons（共通モジュール）
    │      │
    │      └─ createEntrypoints() [src/build/entries.ts]
    │             └─ ページファイル列挙
    │
    ├─ Webpackビルド実行
    │      └─ SplitChunksPlugin
    │             └─ チャンク分割処理
    │
    └─ BuildManifestPlugin [src/build/webpack/plugins/build-manifest-plugin.ts]
           └─ ページ→チャンク対応表生成
```

### データフロー図

```
[入力]                    [処理]                          [出力]

pages/index.tsx    ──▶  エントリーポイント   ──▶  chunks/pages/index-*.js
pages/about.tsx    ──▶  依存グラフ解析       ──▶  chunks/pages/about-*.js
app/page.tsx       ──▶  SplitChunks分割     ──▶  chunks/app/page-*.js
                        │
shared modules     ──▶  共有チャンク分離     ──▶  chunks/commons-*.js
react/react-dom    ──▶  フレームワーク分離   ──▶  chunks/framework-*.js
webpack runtime    ──▶  ランタイム分離       ──▶  chunks/webpack-*.js
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| webpack-config.ts | `packages/next/src/build/webpack-config.ts` | ソース | Webpack設定構築（splitChunks設定を含む） |
| entries.ts | `packages/next/src/build/entries.ts` | ソース | エントリーポイント生成 |
| build-manifest-plugin.ts | `packages/next/src/build/webpack/plugins/build-manifest-plugin.ts` | ソース | ビルドマニフェスト生成 |
| index.ts | `packages/next/src/build/index.ts` | ソース | ビルドプロセスのエントリーポイント |
