# 機能設計書 64-ツリーシェイキング

## 概要

本ドキュメントは、Next.jsのツリーシェイキング（Tree Shaking）機能の設計について記述する。ツリーシェイキングは、ビルド時にアプリケーションで実際に使用されていないコード（Dead Code）を自動的に除去し、配信するJavaScriptバンドルのサイズを最小化する最適化機能である。

### 本機能の処理概要

ツリーシェイキングは、ESモジュール（import/export）の静的解析に基づき、未使用のエクスポートを特定し、最終バンドルから除去する機能である。Webpack、Turbopack、Rspackの各バンドラーが実行する。

**業務上の目的・背景**：ライブラリの一部の関数やコンポーネントのみを使用している場合でも、ツリーシェイキングがなければライブラリ全体がバンドルに含まれてしまう。これはバンドルサイズの肥大化を招き、ページロード時間の増加につながる。ツリーシェイキングにより、使用しているコードのみを含む最小限のバンドルを生成する。

**機能の利用シーン**：プロダクションビルド（`next build`）で自動的に実行される。開発者が明示的に有効化する必要はない。特に大規模なライブラリ（lodash、date-fns、Material-UI等）からの部分インポート時に効果が大きい。

**主要な処理内容**：
1. ESモジュールの静的解析による使用/未使用エクスポートの判定
2. サイドエフェクトの解析（sideEffects設定の参照）
3. 未使用コードのマーキングと除去（TerserPlugin/SWC Minifyによる最終除去）
4. `optimizePackageImports`による特定パッケージのバレルファイル最適化
5. Server Componentsでの自動的なクライアントバンドルからのサーバー専用コード除去

**関連システム・外部連携**：Webpack（usedExports、sideEffects最適化）、Turbopack、Rspackの各バンドラーの最適化機能。SWC Minifier/TerserPluginによる最終的なDead Code Elimination。

**権限による制御**：特になし。ビルド時に自動的に実行される。

## 関連画面

ツリーシェイキングは全ページに影響を与えるビルド時最適化であり、特定の画面には紐づかない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | 全ページ/コンポーネント | 対象 | 未使用コードの除去によるバンドルサイズ削減 |

## 機能種別

ビルド最適化 / Dead Code Elimination

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| mode | string | - | Webpackモード。'production'でツリーシェイキングが有効化 | 'production' \| 'development' |
| sideEffects | boolean \| string[] | No | package.jsonのsideEffectsフィールド | - |
| experimental.optimizePackageImports | string[] | No | バレルファイル最適化対象パッケージリスト | パッケージ名の配列 |

### 入力データソース

- アプリケーションのソースコード（ESモジュールのimport/export文）
- 依存パッケージのpackage.json（sideEffectsフィールド）
- next.config.jsの設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 最適化済みバンドル | .js ファイル | 未使用コードが除去された最小限のJavaScriptバンドル |

### 出力先

`.next/static/chunks/`ディレクトリおよび`.next/server/`ディレクトリに出力される。

## 処理フロー

### 処理シーケンス

```
1. Webpackモジュール解析
   └─ ESモジュールのimport/export文を静的解析
2. 使用エクスポートの判定（usedExports）
   └─ 各モジュールから実際に使用されているエクスポートを特定
3. サイドエフェクト解析
   └─ package.jsonのsideEffectsフィールドを参照し、副作用のないモジュールを特定
4. モジュール連結（Module Concatenation）
   └─ 関連モジュールを1つのスコープに結合（Scope Hoisting）
5. Dead Code Elimination
   └─ TerserPlugin/SWC Minifyで未使用コードを最終除去
6. 最適化済みバンドル出力
```

### フローチャート

```mermaid
flowchart TD
    A[ビルド開始] --> B[ESモジュール静的解析]
    B --> C[usedExportsの判定]
    C --> D{sideEffectsフラグ}
    D -->|false: 副作用なし| E[未使用モジュール全体を除去可能]
    D -->|true/未指定| F[未使用エクスポートのみマーク]
    E --> G[Module Concatenation]
    F --> G
    G --> H[Terser/SWC Minify]
    H --> I[Dead Code Elimination]
    I --> J[最適化済みバンドル出力]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-64-01 | ESモジュール必須 | ツリーシェイキングはESモジュール（import/export）構文のみに適用される。CommonJS（require/module.exports）は対象外 | import/exportを使用するモジュール |
| BR-64-02 | sideEffectsフラグの尊重 | package.jsonでsideEffects: falseが指定されたパッケージは、未使用モジュール全体を安全に除去できる | パッケージにsideEffectsフィールドが設定されている場合 |
| BR-64-03 | プロダクションモード限定 | 完全なツリーシェイキングはプロダクションビルド（mode: 'production'）でのみ実行される | next build実行時 |
| BR-64-04 | optimizePackageImportsの動作 | 指定パッケージのバレルファイル（index.ts等）を解析し、使用されているエクスポートのみをインポートするよう変換する | experimental.optimizePackageImportsに指定されたパッケージ |

### 計算ロジック

バンドラーの内部アルゴリズムにより、モジュール依存グラフ上で使用されているエクスポートを追跡し、到達不能なコードを判定する。

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 予期しないコード除去 | sideEffectsの誤設定により必要なコードが除去された場合 | package.jsonのsideEffects設定を修正するか、特定ファイルを除外リストに追加 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- ツリーシェイキングにより、配信バンドルサイズを大幅に削減できる
- 大規模なライブラリ（lodash全体など）からの部分インポートで特に効果が大きい
- optimizePackageImportsにより、バレルファイルの解析コストを削減できる

## セキュリティ考慮事項

- Server Componentsのツリーシェイキングにより、サーバー専用コード（データベース接続情報等）がクライアントバンドルに含まれることを防止する

## 備考

- CommonJSモジュールはツリーシェイキングの対象外となる。ESモジュールへの移行が推奨される
- `next/dynamic`と組み合わせることで、コード分割とツリーシェイキングの両方の恩恵を得られる
- Webpackの`optimization.usedExports`と`optimization.sideEffects`が内部的に有効化される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: Webpack最適化設定を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | webpack-config.ts | `packages/next/src/build/webpack-config.ts` | optimization設定内のusedExports、sideEffects、concatenateModules（Scope Hoisting）設定を確認 |

**読解のコツ**: Webpackのoptimizationフィールドで`usedExports: true`が設定されていることを確認する。これがツリーシェイキングの基盤となる。`concatenateModules`（ModuleConcatenationPlugin）はScope Hoistingを有効にし、ツリーシェイキングの効率を高める。

#### Step 2: パッケージインポート最適化を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | optimize-barrel-exports.ts | `packages/next/src/build/webpack/loaders/next-barrel-loader.ts` | バレルファイル最適化ローダー。optimizePackageImportsで指定されたパッケージのバレルファイルを解析し、使用エクスポートのみをインポート |

**読解のコツ**: バレルファイル（index.ts）は`export { A } from './a'; export { B } from './b';`のように多数のre-exportを行う。このローダーは使用されていないre-exportを除去し、未使用モジュールの解析コストを削減する。

#### Step 3: ミニファイ・Dead Code Eliminationを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | webpack-config.ts | `packages/next/src/build/webpack-config.ts` | minimizer設定。TerserPluginまたはSWC Minifyの設定を確認 |

### プログラム呼び出し階層図

```
next build [src/build/index.ts]
    │
    ├─ getBaseWebpackConfig() [src/build/webpack-config.ts]
    │      ├─ optimization.usedExports = true
    │      ├─ optimization.sideEffects = true
    │      ├─ optimization.concatenateModules = true
    │      └─ optimization.minimizer（Terser/SWC）
    │
    ├─ Webpackビルド実行
    │      ├─ モジュール解析（import/export追跡）
    │      ├─ usedExports判定
    │      ├─ sideEffects判定（package.json参照）
    │      ├─ ModuleConcatenationPlugin（Scope Hoisting）
    │      └─ TerserPlugin/SWC Minify（Dead Code Elimination）
    │
    └─ 最適化済みバンドル出力
```

### データフロー図

```
[入力]                          [処理]                          [出力]

ソースコード             ──▶  ESモジュール解析
  import { A } from 'lib'       │
                                ▼
package.json             ──▶  sideEffects判定
  "sideEffects": false          │
                                ▼
                              usedExports追跡
                                │
                                ▼
                              Module Concatenation
                                │
                                ▼
                              Terser/SWC Minify     ──▶  最適化済みバンドル
                              (Dead Code Elimination)       （未使用コード除去済み）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| webpack-config.ts | `packages/next/src/build/webpack-config.ts` | ソース | Webpack最適化設定（usedExports、sideEffects、minimizer） |
| next-barrel-loader.ts | `packages/next/src/build/webpack/loaders/next-barrel-loader.ts` | ソース | バレルファイル最適化ローダー |
| index.ts | `packages/next/src/build/index.ts` | ソース | ビルドプロセスのエントリーポイント |
