# 機能設計書 66-開発サーバー

## 概要

本ドキュメントは、Next.jsの開発サーバー（DevServer）の設計について記述する。DevServerはNextNodeServerを継承した開発専用サーバーであり、HMR（Hot Module Replacement）、エラーオーバーレイ、ファイル監視、オンデマンドコンパイルを統合した開発体験を提供する。

### 本機能の処理概要

DevServerは、`next dev`コマンドで起動される開発用サーバーであり、ファイル変更の検知とリアルタイム反映、エラー表示、静的パス解析などの開発支援機能を提供する。

**業務上の目的・背景**：Webアプリケーション開発において、コード変更を即座に確認できる環境は生産性に直結する。DevServerは、ファイル保存時の自動コンパイル、ブラウザへのリアルタイム反映（HMR）、わかりやすいエラー表示を統合し、高速な開発イテレーションを実現する。

**機能の利用シーン**：ローカル開発環境で`next dev`コマンドにより起動し、開発中のアプリケーションの動作確認・デバッグに使用する。

**主要な処理内容**：
1. 開発用サーバーの初期化（DevBundlerServiceとの連携、静的パスキャッシュの設定）
2. ファイル変更の検知とオンデマンドコンパイル
3. Hot Module Replacement（HMR）によるブラウザへのリアルタイム反映
4. エラーオーバーレイ（ReactDevOverlay）による開発者向けエラー表示
5. ルーティングのリアルタイム更新（ページ追加/削除の自動検知）
6. Static Pathsの動的解析（getStaticPaths等の開発時実行）
7. Server Components HMRキャッシュの管理

**関連システム・外部連携**：DevBundlerService（バンドラーサービス）、HotReloader（HMR管理）、On-Demand Entry Handler（オンデマンドコンパイル）、DevTools（開発者ツール）。

**権限による制御**：特になし。開発環境でのみ使用される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 12 | 開発オーバーレイ (Dev Overlay) | 補助機能 | 開発サーバー実行時にのみ表示されるオーバーレイUI |

## 機能種別

サーバー / 開発支援 / リアルタイムコンパイル

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| options.conf | NextConfigComplete | Yes | 完全なNext.js設定オブジェクト | - |
| options.isNextDevCommand | boolean | No | next devコマンドからの起動かどうか | - |
| options.bundlerService | DevBundlerService | Yes | 開発バンドラーサービスインスタンス | - |
| options.startServerSpan | Span | No | サーバー起動のトレーススパン | - |

### 入力データソース

- ファイルシステム（pages/、app/ディレクトリのファイル）
- 開発バンドラー（Webpack/Turbopack/Rspack）からのコンパイル結果
- WebSocket経由のHMRメッセージ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| HTTPレスポンス | ServerResponse | 開発用のHTMLレスポンス（ReactDevOverlay付き） |
| HMRメッセージ | WebSocket | ブラウザへのHMR更新通知 |
| コンパイル結果 | webpack.Stats | 各コンパイラのビルド結果 |

### 出力先

HTTPレスポンスとしてブラウザに送信。WebSocket経由でHMRメッセージを送信。

## 処理フロー

### 処理シーケンス

```
1. DevServer初期化（constructor）
   └─ NextNodeServer(super)初期化、DevBundlerService設定
   └─ pagesDir/appDir検出、静的パスキャッシュ初期化
   └─ Server Components HMRキャッシュ初期化（実験的機能）
2. サーバー起動（prepare()）
   └─ ルートマッチャーの初期化、ページスキャン
3. リクエスト受信
   └─ ready状態を待機後、リクエスト処理
4. ページのオンデマンドコンパイル
   └─ ensurePage()でDevBundlerService経由のコンパイル実行
5. レンダリング
   └─ ReactDevOverlayでエラーオーバーレイ付きレンダリング
6. HMR処理
   └─ WebSocket経由でブラウザにモジュール更新を通知
```

### フローチャート

```mermaid
flowchart TD
    A[next dev起動] --> B[DevServer初期化]
    B --> C[prepare: ルートスキャン]
    C --> D[ready状態]
    D --> E[リクエスト受信]
    E --> F{ページコンパイル済み?}
    F -->|No| G[ensurePage: オンデマンドコンパイル]
    F -->|Yes| H[レンダリング実行]
    G --> H
    H --> I[ReactDevOverlay付きレスポンス]
    I --> J[レスポンス送信]
    K[ファイル変更検知] --> L[再コンパイル]
    L --> M[HMRメッセージ送信]
    M --> N[ブラウザ更新]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-66-01 | Server継承 | DevServerはNextNodeServer（Server）を継承し、dev: trueで初期化する | 常時 |
| BR-66-02 | Error.stackTraceLimit拡張 | 開発環境ではスタックトレースの上限を50に拡張する | コンストラクタ実行時 |
| BR-66-03 | 静的パスキャッシュ | LRUCache（5MB）でgetStaticPathsの結果をキャッシュする | 静的パス解析時 |
| BR-66-04 | 静的パスワーカー | 静的パス解析は別ワーカープロセスで実行される（numWorkers: 1） | getStaticPaths呼び出し時 |
| BR-66-05 | Server Components HMRキャッシュ | serverComponentsHmrCache有効時、サーバーコンポーネントのレンダリング結果をキャッシュする | experimental.serverComponentsHmrCache有効時 |
| BR-66-06 | ReactDevOverlay | Pages Routerのレンダリング時にReactDevOverlayでエラーオーバーレイを付与する | 開発モード時 |

### 計算ロジック

静的パスキャッシュのサイズ計算：`JSON.stringify(value.staticPaths)?.length || 1`

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| WrappedBuildError | ビルドエラー | コンパイル時にエラーが発生した場合 | エラーオーバーレイに表示 |
| PageNotFoundError | 404 | 存在しないページにアクセスした場合 | 開発モードのエラーページを表示 |
| MiddlewareNotFoundError | 500 | ミドルウェアファイルが見つからない場合 | エラーをログ出力 |

### リトライ仕様

ファイル保存時に自動的に再コンパイルが行われる。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- オンデマンドコンパイルにより、アクセスされたページのみをビルドし起動時間を短縮する
- 静的パスワーカーのworkerThreadsを設定で制御可能
- LRUCacheにより静的パス解析結果を効率的にキャッシュする

## セキュリティ考慮事項

- 開発サーバーは本番環境での使用を意図していない
- NODE_OPTIONSから--inspect/--inspect-brkフラグを除外し、ワーカープロセスでのデバッガー競合を回避する

## 備考

- DevServerは`packages/next/src/server/dev/next-dev-server.ts`に定義される
- PagesDevOverlayBridgeは遅延ロードされる（require時のみ）
- interception routes rewritesの自動生成機能を含む

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | next-dev-server.ts | `packages/next/src/server/dev/next-dev-server.ts` | Options型定義（97-114行目）。isNextDevCommand、bundlerService、startServerSpanの各フィールド |

**読解のコツ**: DevServerのOptions型はServerOptionsを拡張し、confをNextConfigComplete型に昇格させている。bundlerServiceがバンドラーとの通信を担う。

#### Step 2: コンストラクタを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | next-dev-server.ts | `packages/next/src/server/dev/next-dev-server.ts` | DevServerクラスのconstructor（173-199行目）。初期化処理の全体像 |

**主要処理フロー**:
1. **174-177行目**: Error.stackTraceLimitを50に設定
2. **178行目**: super({ ...options, dev: true })でNextNodeServerを初期化
3. **179-182行目**: nextConfig、bundlerService、startServerSpanの設定
4. **183-184行目**: renderOpts.dev = true、ErrorDebug = ReactDevOverlayの設定
5. **185-192行目**: 静的パスLRUCache（5MB）の初期化
6. **194-196行目**: pagesDir/appDirの検出
7. **198-199行目**: serverComponentsHmrCache有効時のHMRキャッシュ初期化

#### Step 3: Static Pathsワーカーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | next-dev-server.ts | `packages/next/src/server/dev/next-dev-server.ts` | getStaticPathsWorker()（144-171行目）。jest-workerベースの別プロセス実行 |

### プログラム呼び出し階層図

```
DevServer [next-dev-server.ts]
    │ extends
    ├─ NextNodeServer (Server) [next-server.ts]
    │      │ extends
    │      └─ BaseServer [base-server.ts]
    │
    ├─ DevBundlerService [lib/dev-bundler-service.ts]
    │      └─ ensurePage() → バンドラーへのコンパイル要求
    │
    ├─ ReactDevOverlay [next-devtools/userspace/pages/pages-dev-overlay-setup.ts]
    │      └─ Pages Routerのエラーオーバーレイ
    │
    ├─ static-paths-worker [dev/static-paths-worker.ts]
    │      └─ getStaticPaths解析（別ワーカープロセス）
    │
    └─ LRUCache [lib/lru-cache.ts]
           └─ 静的パスキャッシュ（5MB上限）
```

### データフロー図

```
[入力]                          [処理]                            [出力]

HTTPリクエスト            ──▶  DevServer
                                │
ファイル変更              ──▶  DevBundlerService ──▶ HMRメッセージ ──▶ ブラウザ更新
                                │
pages/ / app/            ──▶  ルートスキャン      ──▶ ルーティング更新
                                │
getStaticPaths           ──▶  Worker             ──▶ LRUCache
                                │
コンパイルエラー          ──▶  ReactDevOverlay    ──▶ エラーオーバーレイ表示
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| next-dev-server.ts | `packages/next/src/server/dev/next-dev-server.ts` | ソース | DevServerクラス本体 |
| next-server.ts | `packages/next/src/server/next-server.ts` | ソース | 基底クラス（NextNodeServer） |
| dev-bundler-service.ts | `packages/next/src/server/lib/dev-bundler-service.ts` | ソース | 開発バンドラーサービス |
| static-paths-worker.ts | `packages/next/src/server/dev/static-paths-worker.ts` | ソース | 静的パス解析ワーカー |
| pages-dev-overlay-setup.ts | `packages/next/src/next-devtools/userspace/pages/pages-dev-overlay-setup.ts` | ソース | Pages Router用エラーオーバーレイ |
| hot-reloader-types.ts | `packages/next/src/server/dev/hot-reloader-types.ts` | ソース | HMRメッセージ型定義 |
| log-requests.ts | `packages/next/src/server/dev/log-requests.ts` | ソース | 開発時リクエストログ |
