# 機能設計書 80-URLPattern

## 概要

本ドキュメントは、Next.jsにおけるURLPattern APIの設計について記述する。URLPatternはWeb標準のURLPatternを再エクスポートし、Middleware内でURLパターンマッチングを行うためのAPIである。

### 本機能の処理概要

URLPatternは、Web標準の`URLPattern` APIをNext.jsから利用可能にするためのモジュールである。Next.jsのspec-extension/url-pattern.tsファイルは、グローバルスコープの`URLPattern`クラスが利用可能な場合はそれをそのまま再エクスポートし、利用不可の場合（Node.js環境等）は`undefined`を返す。URLPatternはURLの各コンポーネント（protocol, hostname, pathname, search, hash等）に対してパターンマッチングを行い、動的セグメントのキャプチャが可能である。

**業務上の目的・背景**：Middleware内でURLのパターンマッチングを行う場合、正規表現を手動で構築するのは複雑でエラーが起きやすい。URLPatternはWeb標準のAPIとして、URLの各部分（プロトコル、ホスト名、パス名、クエリ、フラグメント）に対する構造化されたパターンマッチングを提供し、ルーティングロジックの実装を簡素化する。NextRequestの旧`page`プロパティの代替として推奨されている。

**機能の利用シーン**：Middleware内でのURLパスマッチング、動的パラメータの抽出、特定のURLパターンに対する条件分岐、APIルートのマッチング。

**主要な処理内容**：
1. グローバルスコープの`URLPattern`の可用性チェック
2. 利用可能な場合はグローバル`URLPattern`を`GlobalURLPattern`として再エクスポート
3. `URLPattern`は`test()`, `exec()`メソッドによるパターンマッチングを提供（Web標準仕様）

**関連システム・外部連携**：Web標準URLPattern API、Edge Runtime（URLPatternがネイティブサポート）

**権限による制御**：特になし。URLPatternは純粋なパターンマッチングユーティリティである。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | URLPatternは画面に直接関連しない。サーバーサイドのURLマッチングに使用される |

## 機能種別

URLパターンマッチング / ユーティリティ

## 入力仕様

### 入力パラメータ

#### URLPatternコンストラクタ（Web標準）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| input | URLPatternInit \| string | Yes | URLパターン定義 | - |
| baseURL | string | No | ベースURL | - |

#### test()

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| input | URLPatternInput | Yes | テスト対象のURL | - |

#### exec()

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| input | URLPatternInput | Yes | マッチング対象のURL | - |

### 入力データソース

Middleware内でプログラマティックに構築されるURLパターンおよびリクエストURL。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| URLPattern \| undefined | typeof URLPattern \| undefined | グローバルURLPatternクラス、またはundefined（Node.js環境等で未対応の場合） |

#### test()の出力

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | boolean | パターンにマッチするかどうか |

#### exec()の出力

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | URLPatternResult \| null | マッチ結果（各コンポーネントのキャプチャグループ含む）。マッチしない場合null |

### 出力先

Middleware内のコードで使用される。

## 処理フロー

### 処理シーケンス

```
1. url-pattern.tsモジュールの読み込み
   └─ グローバルスコープのURLPatternの可用性チェック
2. URLPatternの利用可能性判定
   └─ typeof URLPattern === 'undefined' の場合: undefinedをエクスポート
   └─ それ以外の場合: グローバルURLPatternをエクスポート
3. ユーザーコードでのURLPattern使用
   └─ new URLPattern({ pathname: '/api/:id' }) でパターン作成
   └─ pattern.test(url) / pattern.exec(url) でマッチング実行
```

### フローチャート

```mermaid
flowchart TD
    A[url-pattern.ts 読み込み] --> B{typeof URLPattern === undefined?}
    B -->|Yes| C[undefined をエクスポート]
    B -->|No| D[globalThis.URLPattern をエクスポート]
    D --> E[ユーザーコードで使用]
    E --> F[new URLPattern pattern]
    F --> G{test / exec}
    G -->|test| H[boolean 返却]
    G -->|exec| I[URLPatternResult / null 返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-80-01 | 環境依存 | URLPatternはEdge Runtimeではネイティブサポートされるが、Node.js環境では利用不可の場合がある | ランタイム環境による |
| BR-80-02 | NextRequest.page代替 | URLPatternはNextRequestの旧pageプロパティの代替として推奨される | Middleware開発時 |
| BR-80-03 | Web標準準拠 | URLPatternはWeb標準のURL Pattern APIに準拠し、その仕様に従う | 常時 |

### 計算ロジック

URLPatternの具体的なマッチングロジックはWeb標準仕様に準拠しており、Next.js固有の計算ロジックは存在しない。

## データベース操作仕様

### 操作別データベース影響一覧

URLPatternはデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TypeError | 利用不可 | Node.js環境等でURLPatternが利用できない場合にnew URLPattern()を呼ぶ | Edge Runtime環境で使用するか、ポリフィルを導入する |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- URLPatternの生成とマッチングはネイティブ実装（Edge Runtime）では高速
- パターンの事前コンパイルにより、繰り返しマッチングの効率化が可能

## セキュリティ考慮事項

- URLPatternは入力URLのパース・マッチングのみを行い、副作用はない
- 過度に複雑なパターンによるReDoS（Regular Expression Denial of Service）リスクは、ネイティブ実装では一般的に低い

## 備考

- url-pattern.tsは5行のシンプルなモジュールであり、グローバルURLPatternの再エクスポートのみを行う
- `@ts-expect-error`コメントはNode.js型定義にURLPatternが含まれていないためのTypeScript型エラー抑制
- `next/server`からインポートして使用する
- URLPatternの詳細な仕様はWeb標準仕様（https://urlpattern.spec.whatwg.org/）を参照

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: モジュールの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | url-pattern.ts | `packages/next/src/server/web/spec-extension/url-pattern.ts` | 全5行のモジュール。typeof URLPatternによる可用性チェックとエクスポートのロジックを確認 |

**読解のコツ**: このモジュールはWeb標準APIの再エクスポートに過ぎない。`@ts-expect-error`はNode.jsのTypeScript型定義にURLPatternが含まれていないために必要。実際のURLPatternの実装はランタイム環境（Edge Runtime等）が提供する。

**主要処理フロー**:
1. **1-3行目**: グローバルURLPatternの可用性チェック。undefinedの場合はundefined、それ以外はURLPatternを`GlobalURLPattern`に代入
2. **5行目**: `GlobalURLPattern`を`URLPattern`としてnamed export

#### Step 2: 利用箇所を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | request.ts | `packages/next/src/server/web/spec-extension/request.ts` | NextRequestの旧pageプロパティ（89-91行目）がRemovedPageErrorをスローし、URLPatternの使用を推奨している箇所を確認 |

### プログラム呼び出し階層図

```
url-pattern.ts
    |
    +-- typeof URLPattern check
    |       +-- undefined -> export undefined
    |       +-- defined  -> export globalThis.URLPattern
    |
    [ユーザーコード]
        +-- import { URLPattern } from 'next/server'
        +-- new URLPattern({ pathname: '/api/:id' })
        +-- pattern.test(url) / pattern.exec(url)
```

### データフロー図

```
[入力]                       [処理]                        [出力]

グローバルスコープ -------> url-pattern.ts            URLPattern | undefined
  URLPattern                  |                            |
  (Edge Runtime等)            +-> typeof チェック            |
                              +-> 再エクスポート             |
                                                            v
URLパターン文字列 -------> URLPattern                  boolean / Result
  { pathname: '...' }    コンストラクタ                      |
                              |                            |
対象URL -----------------> test() / exec() ------------>+
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| url-pattern.ts | `packages/next/src/server/web/spec-extension/url-pattern.ts` | ソース | URLPattern再エクスポートモジュール（5行） |
| request.ts | `packages/next/src/server/web/spec-extension/request.ts` | ソース | NextRequestの旧pageプロパティでURLPattern推奨メッセージ |
