# 機能設計書 89-SWC（コンパイラ）

## 概要

本ドキュメントは、Next.jsにおけるRustベースのJavaScript/TypeScriptコンパイラSWCの統合に関する機能設計書である。Babelの代替として高速なトランスパイル・コード変換・最適化を提供する。

### 本機能の処理概要

**業務上の目的・背景**：従来のBabelベースのトランスパイルは大規模プロジェクトでボトルネックとなることがあった。SWCはRustで実装された高速コンパイラであり、Next.jsのデフォルトトランスパイラとして採用され、ビルド速度の大幅な改善を実現する。また、SWCはNext.js固有の変換（Server Components、Server Actions、styled-jsx等）もネイティブにサポートする。

**機能の利用シーン**：`next build`、`next dev`のビルド処理時にすべてのJavaScript/TypeScriptファイルのトランスパイルに使用される。WebpackのSWCローダー、Turbopackの内部コンパイラ、Jest トランスフォーマーとしても機能する。

**主要な処理内容**：
1. ネイティブバインディング（プラットフォーム別）またはWASMバインディングの読み込み
2. JavaScript/TypeScriptのトランスパイル（ES最新構文 -> ターゲット環境対応）
3. JSX/TSXの変換（React自動ランタイムモード）
4. Next.js固有の変換（Server Components、Server Actions、styled-jsx、emotion、styled-components等）
5. コード最適化（removeConsole、reactRemoveProperties、modularizeImports等）
6. SWCプラグインのサポート
7. ソースマップ生成

**関連システム・外部連携**：`@next/swc-*`プラットフォーム別バイナリパッケージ、Webpack SWCローダー、Turbopack

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面を持たないコンパイル処理機構である |

## 機能種別

コード変換 / ビルド処理 / 最適化

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| filename | string | Yes | 変換対象ファイルのパス | 有効なファイルパス |
| development | boolean | Yes | 開発モードかどうか | - |
| hasReactRefresh | boolean | No | React Refresh有効かどうか | - |
| globalWindow | boolean | No | windowオブジェクトが利用可能か | - |
| esm | boolean | No | ESモジュール出力かどうか | - |
| compilerOptions | NextConfig['compiler'] | No | コンパイラ設定（styled-jsx、emotion等） | - |
| swcPlugins | Array | No | SWCプラグインの配列 | [name, options]のタプル配列 |
| bundleLayer | WebpackLayerName | No | Webpackレイヤー名 | - |
| serverComponents | boolean | No | Server Components有効かどうか | - |

### 入力データソース

- JavaScript/TypeScriptソースファイル
- `next.config.js`のcompiler設定
- `jsconfig.json` / `tsconfig.json`（パスエイリアス、デコレータ設定等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | string | トランスパイル済みJavaScriptコード |
| map | string | ソースマップ（設定時） |
| Module（AST） | object | パース済みAST（静的解析用） |

### 出力先

- Webpack/Turbopack/Rspackバンドルパイプライン
- ファイルシステム（.next/ディレクトリ）

## 処理フロー

### 処理シーケンス

```
1. SWCバインディングの読み込み
   └─ プラットフォーム判定（OS + アーキテクチャ）
   └─ ネイティブバインディング試行
   └─ 失敗時: WASMバインディングフォールバック
2. パーサーオプションの構築
   └─ getParserOptions(): TypeScript/ECMAScript構文選択、デコレータ設定
3. ベースSWCオプションの構築
   └─ getBaseSWCOptions(): JSX設定、optimizer設定、プラグイン設定
4. Next.js固有変換の設定
   └─ serverComponents: RSCレイヤー判定
   └─ serverActions: Server Actions変換設定
   └─ styledJsx, emotion, styledComponents: CSS-in-JS変換
   └─ modularizeImports: インポート最適化
5. コード変換の実行
   └─ bindings.transform(source, options)
6. 結果の返却
   └─ トランスパイル済みコード + ソースマップ
```

### フローチャート

```mermaid
flowchart TD
    A[ソースファイル入力] --> B[パーサーオプション構築]
    B --> C{TS/TSX?}
    C -->|Yes| D[TypeScript構文パーサー]
    C -->|No| E[ECMAScript構文パーサー]
    D --> F[ベースSWCオプション構築]
    E --> F
    F --> G[Next.js固有変換設定]
    G --> H[SWCプラグイン設定]
    H --> I[bindings.transform実行]
    I --> J[トランスパイル済みコード + ソースマップ]
    J --> K[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-89-01 | SWCデフォルト | SWCがデフォルトトランスパイラ、Babel設定ファイル存在時のみBabelフォールバック | 全ビルド |
| BR-89-02 | パーサー自動選択 | .ts/.tsxはTypeScript構文、それ以外はECMAScript構文で解析 | ファイル拡張子に基づく |
| BR-89-03 | JSX自動ランタイム | React JSXの変換は自動ランタイムモード（`runtime: 'automatic'`） | 全JSX変換 |
| BR-89-04 | CSS-in-JSサーバー制限 | emotion、styled-componentsはReact Serverレイヤーでは無効化 | Server Components |
| BR-89-05 | CJS出力判定 | .cjsファイルとNext.js内部モジュールはCommonJSとして出力 | 該当ファイル |
| BR-89-06 | externalHelpers | pnp環境でない場合はSWCヘルパーを外部モジュールとして参照 | 非pnp環境 |

### 計算ロジック

特になし。

## データベース操作仕様

本機能はデータベースに対する操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Error | ネイティブバインディング読み込み失敗（WASMも失敗） | プラットフォーム対応のバイナリをインストール |
| - | Warning | SWCバージョン不一致 | Next.jsバージョンに合わせてSWCを更新 |
| - | Error | SWCプラグインのrequire.resolve失敗 | プラグインパッケージをインストール |

### リトライ仕様

バインディング読み込みにおいて、ネイティブ -> WASMダウンロード -> WASMフォールバックの自動リトライが行われる。

## トランザクション仕様

トランザクション管理は不要。

## パフォーマンス要件

- SWCはBabelと比較して数倍から数十倍の高速なトランスパイルを実現
- LRUキャッシュ（500エントリ）によるAST解析結果のキャッシュ
- Rustスタックサイズはデフォルト8MB（大規模npmパッケージのコンパイル対応）

## セキュリティ考慮事項

- SWCプラグインは任意のコードを実行できるため、信頼できるプラグインのみを使用すべき
- Server Actionsの暗号化処理にSWCの変換が関与する

## 備考

- SWCバインディングは`@next/swc-{platform}-{arch}`パッケージとして配布される
- プラットフォームサポート: macOS (x64/arm64), Linux (x64/arm64), Windows (x64/ia32/arm64), FreeBSD (x64)
- 非サポートプラットフォームではWASMフォールバックが使用される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | types.ts | `packages/next/src/build/swc/types.ts` | SWCバインディングの主要型定義（Binding、Project、Endpoint等） |
| 1-2 | generated-native.d.ts | `packages/next/src/build/swc/generated-native.d.ts` | ネイティブバインディングのAPI型定義 |

**読解のコツ**: SWCバインディングはRust側のAPIをNode.jsから呼び出すnapi-rsブリッジを使用している。型定義は自動生成される。

#### Step 2: バインディング読み込みを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | index.ts | `packages/next/src/build/swc/index.ts` | **68-94行目**: `getSupportedArchTriples`でプラットフォーム対応トリプルを取得 |
| 2-2 | index.ts | `packages/next/src/build/swc/index.ts` | **197-269行目**: `loadBindings`関数でネイティブ/WASMバインディングの読み込み |

**主要処理フロー**:
- **68-94行目**: darwin, win32, linux, freebsd, android向けのサポートトリプル定義
- **131-139行目**: バージョン不一致チェック
- **146-154行目**: WASM優先対象プラットフォーム（FreeBSD, Android, 32bit Windows等）
- **197-269行目**: バインディング読み込みフロー（ネイティブ -> lockfileパッチ -> ダウンロード -> WASMフォールバック）

#### Step 3: SWCオプション構築を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | options.ts | `packages/next/src/build/swc/options.ts` | **39-54行目**: `getParserOptions` - パーサー設定（TypeScript/ECMAScript選択） |
| 3-2 | options.ts | `packages/next/src/build/swc/options.ts` | **56-252行目**: `getBaseSWCOptions` - JSX、optimizer、プラグイン等の全オプション構築 |

**主要処理フロー**:
- **40-41行目**: ファイル拡張子による構文選択（.ts/.tsx -> typescript）
- **119-158行目**: JIT設定（baseUrl、paths、externalHelpers、parser、transform）
- **146-158行目**: React変換設定（importSource、runtime: automatic、refresh）
- **159-171行目**: optimizer設定（globalWindow判定、NODE_ENV定義）
- **179-202行目**: modularizeImports設定
- **203-215行目**: CSS-in-JS設定（emotion、styled-components、サーバーレイヤー制限）
- **216-236行目**: Server Components、Server Actions設定

#### Step 4: CSS-in-JSオプションのヘルパー関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | options.ts | `packages/next/src/build/swc/options.ts` | **254-270行目**: `getStyledComponentsOptions` |
| 4-2 | options.ts | `packages/next/src/build/swc/options.ts` | **272-300行目**: `getEmotionOptions` |

### プログラム呼び出し階層図

```
Webpack SWCローダー / Turbopack
    |
    +-- loadBindings()                   # バインディング読み込み
    |       +-- loadNative()             # ネイティブバインディング試行
    |       |       +-- require(@next/swc-{platform}-{arch})
    |       +-- loadWasm()               # WASMフォールバック
    |               +-- downloadWasmSwc()  # WASM自動ダウンロード
    |
    +-- getParserOptions()               # パーサー設定
    |       +-- TypeScript/ECMAScript判定
    |       +-- デコレータ設定
    |
    +-- getBaseSWCOptions()              # SWC変換オプション
    |       +-- JSX設定
    |       +-- optimizer設定
    |       +-- CSS-in-JS設定
    |       +-- Server Components/Actions設定
    |       +-- modularizeImports設定
    |       +-- SWCプラグイン設定
    |
    +-- bindings.transform(source, options)  # コード変換実行
    |
    +-- bindings.parse(source, options)      # AST解析（静的解析用）
```

### データフロー図

```
[入力]                    [処理]                          [出力]

ソースファイル ────────▶ getParserOptions() ────┐
jsconfig.json ────────▶ getBaseSWCOptions() ───┤
next.config.js ───────▶                        ├──▶ bindings.transform()
compiler設定               SWCオプション構築     │         |
                                                │    トランスパイル済みJS
                                                │    ソースマップ
                                                │
ソースファイル ────────▶ bindings.parse() ──────┘──▶ AST（静的解析用）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| index.ts | `packages/next/src/build/swc/index.ts` | ソース | SWCバインディング読み込み・API公開 |
| options.ts | `packages/next/src/build/swc/options.ts` | ソース | SWC変換オプションの構築 |
| types.ts | `packages/next/src/build/swc/types.ts` | ソース | SWC/Turbopack型定義 |
| helpers.ts | `packages/next/src/build/swc/helpers.ts` | ソース | SWCヘルパー関数 |
| generated-native.d.ts | `packages/next/src/build/swc/generated-native.d.ts` | 型定義 | ネイティブバインディングの自動生成型定義 |
| generated-wasm.d.ts | `packages/next/src/build/swc/generated-wasm.d.ts` | 型定義 | WASMバインディングの自動生成型定義 |
| jest-transformer.ts | `packages/next/src/build/swc/jest-transformer.ts` | ソース | Jest用SWCトランスフォーマー |
| install-bindings.ts | `packages/next/src/build/swc/install-bindings.ts` | ソース | バインディングのインストール処理 |
| next-swc-loader.ts | `packages/next/src/build/webpack/loaders/next-swc-loader.ts` | ソース | Webpack用SWCローダー |
