# 機能設計書 95-エラーオーバーレイ

## 概要

本ドキュメントは、Next.jsのエラーオーバーレイ機能の設計について記述する。エラーオーバーレイは開発時にコンパイルエラー・ランタイムエラー・ハイドレーションエラーをブラウザ上に分かりやすく表示する機能である。

### 本機能の処理概要

**業務上の目的・背景**：開発中にエラーが発生した場合、開発者がブラウザのコンソールやターミナルを切り替えることなく、直接ブラウザ画面上でエラーの詳細情報を確認できるようにすることで、デバッグ効率を大幅に向上させる。

**機能の利用シーン**：
- コンパイルエラー発生時にターミナル風のエラー出力を表示する場合
- Reactコンポーネントのランタイムエラー発生時にスタックトレースとソースコードフレームを表示する場合
- SSRとCSRの出力不一致（ハイドレーションエラー）発生時に差分を視覚的に表示する場合
- エラーメッセージからエディタを直接開いてファイルの該当行にジャンプする場合

**主要な処理内容**：
1. ビルドエラー（BuildError）のターミナル風表示 - コンパイラ出力のANSIカラー除去と整形
2. ランタイムエラー（RuntimeError）のスタックトレース表示 - ソースマップ解決済みスタック
3. ハイドレーションエラーの疑似HTML差分表示 - PseudoHtmlDiffコンポーネント
4. コールスタック表示 - CallStackコンポーネント
5. コードフレーム表示 - エラー発生箇所のソースコードを構文ハイライト付きで表示
6. エディタリンク - ファイルパスクリックでエディタの該当行を開く

**関連システム・外部連携**：HMR WebSocket（エラー情報の受信）、ソースマップ（スタックトレース解決）、エディタ連携（launch-editor）。

**権限による制御**：開発モード（`next dev`）でのみ動作。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 15 | ビルドエラー画面 | 主機能 | コンパイルエラー発生時のターミナル風エラー出力表示処理 |
| 16 | ランタイムエラー画面 | 主機能 | ハイドレーションエラー・ランタイムエラーの種別に応じたUI表示処理 |
| 12 | 開発オーバーレイ (Dev Overlay) | 補助機能 | ErrorOverlayコンポーネントによるエラー情報のオーバーレイ表示 |

## 機能種別

UI表示 / エラー解析

## 入力仕様

### 入力パラメータ

#### BuildError

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| message | string | Yes | ビルドエラーメッセージ（ANSIカラーコード含む） | - |
| versionInfo | object | Yes | Next.jsバージョン情報 | - |

#### RuntimeError (Errors)

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| runtimeErrors | ReadyRuntimeError[] | Yes | 処理済みランタイムエラーの配列 | - |
| getSquashedHydrationErrorDetails | function | Yes | ハイドレーションエラー詳細取得関数 | - |
| debugInfo | DebugInfo | Yes | デバッグ情報 | - |
| onClose | function | Yes | オーバーレイを閉じるコールバック | - |

### 入力データソース

HMR WebSocketを通じたサーバーからのビルドエラー通知、ブラウザのwindow.onerrorやError Boundaryで捕捉されたランタイムエラー。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| エラーオーバーレイUI | React Component | エラー詳細を表示するオーバーレイ画面 |
| エラー情報テキスト | string | AI向けにフォーマットされたエラー情報（コピー可能） |

### 出力先

ブラウザ上のShadow DOM内（DevToolsオーバーレイフレームワーク内）。

## 処理フロー

### 処理シーケンス

```
1. エラーの受信
   └─ HMRまたはError Boundaryからエラー情報を受信
2. エラー種別の判定
   └─ ビルドエラー / ランタイムエラー / ハイドレーションエラーを判定
3. エラー情報の整形
   └─ ANSIカラー除去、ソースマップ解決、コードフレーム生成
4. UIレンダリング
   └─ ErrorOverlayLayoutに応じたレイアウトでエラー情報を表示
5. ユーザーインタラクション
   └─ エラーナビゲーション、エディタ連携、コピー機能
```

### フローチャート

```mermaid
flowchart TD
    A[エラー受信] --> B{エラー種別}
    B -->|ビルドエラー| C[BuildError]
    B -->|ランタイムエラー| D[Errors]
    C --> E[stripAnsi でANSI除去]
    E --> F[Terminal コンポーネントで表示]
    D --> G{ハイドレーションエラー?}
    G -->|Yes| H[PseudoHtmlDiff で差分表示]
    G -->|No| I[GenericErrorDescription]
    H --> J[ErrorOverlayLayout]
    I --> J
    F --> J
    J --> K[コールスタック表示]
    K --> L[エディタリンク提供]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ビルドエラー優先 | ビルドエラーがある場合はランタイムエラーより優先して表示 | 同時発生時 |
| BR-02 | ANSIカラー除去 | エラーメッセージのANSIカラーコードはstripAnsiで除去 | 常時 |
| BR-03 | エラーソース識別 | getErrorSourceによりエラーがサーバー由来かクライアント由来か判別 | ランタイムエラー時 |
| BR-04 | 環境名ラベル表示 | error.environmentNameが設定されている場合、ラベルとして表示 | サーバーエラー時 |
| BR-05 | AI向けエラー情報生成 | エラー情報をAIプロンプト向けにフォーマットした文字列を生成 | BuildError時 |

### 計算ロジック

ビルドエラーメッセージの解析ロジック（getErrorTextFromBuildErrorMessage）：
- マルチライン文字列を改行で分割
- 2行目（インデックス1）をエラーメッセージとして抽出
- `Error: ` プレフィックスを除去

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | SyntaxError | コンパイルエラー | BuildErrorコンポーネントで表示 |
| - | TypeError等 | ランタイムエラー | Errorsコンポーネントで表示 |
| - | HydrationError | SSR/CSR不一致 | PseudoHtmlDiffで差分表示 |

### リトライ仕様

該当なし（エラー表示機能のため）。

## トランザクション仕様

該当なし。

## パフォーマンス要件

エラーオーバーレイはエラー発生時にのみレンダリングされ、通常の開発フローへのパフォーマンス影響はない。

## セキュリティ考慮事項

- エラースタックトレースにはソースコードのパス情報が含まれるが、開発モードでのみ表示されるため問題ない
- エディタ連携はローカル環境でのみ動作する

## 備考

- React 19のハイドレーションエラーとReact 18のハイドレーションエラーで異なるエラー解析ロジックが使用される（react-19-hydration-error.ts / react-18-hydration-error.ts）
- Storybookでのコンポーネント開発がサポートされている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | get-error-by-type.ts | `packages/next/src/next-devtools/dev-overlay/utils/get-error-by-type.ts` | ReadyRuntimeError型の構造 |
| 1-2 | hydration-error.ts | `packages/next/src/next-devtools/shared/hydration-error.ts` | HydrationErrorState型の構造 |

**読解のコツ**: ReadyRuntimeErrorはエラーオブジェクトにソースマップ解決済みのフレーム情報を付加した型。useFrames hookと組み合わせて使用される。

#### Step 2: ビルドエラー表示を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | build-error.tsx | `packages/next/src/next-devtools/dev-overlay/container/build-error.tsx` | BuildErrorコンポーネントの実装 |

**主要処理フロー**:
1. **11-27行目**: getErrorTextFromBuildErrorMessageでビルドエラーメッセージから表示テキストを抽出
2. **29-60行目**: BuildErrorコンポーネントがTerminalコンポーネントとErrorOverlayLayoutでUI構成
3. **40-60行目**: AI向けエラー情報生成ロジック

#### Step 3: ランタイムエラー表示を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | errors.tsx | `packages/next/src/next-devtools/dev-overlay/container/errors.tsx` | ランタイムエラー表示のメインコンポーネント |
| 3-2 | component-stack-pseudo-html.tsx | `packages/next/src/next-devtools/dev-overlay/container/runtime-error/component-stack-pseudo-html.tsx` | ハイドレーションエラー差分表示 |
| 3-3 | render-error.tsx | `packages/next/src/next-devtools/dev-overlay/container/runtime-error/render-error.tsx` | エラーレンダリングのラッパー |

**主要処理フロー**:
- **42-44行目** (errors.tsx): HydrationErrorDescriptionがハイドレーションエラーの説明テキストを表示
- **46-60行目** (errors.tsx): GenericErrorDescriptionが一般的なランタイムエラーを表示、環境名プレフィックスの処理

### プログラム呼び出し階層図

```
ErrorOverlay (error-overlay.tsx)
    │
    ├─ BuildError (build-error.tsx)
    │      ├─ stripAnsi
    │      ├─ Terminal (terminal.tsx)
    │      └─ ErrorOverlayLayout
    │
    └─ Errors (errors.tsx)
           ├─ HydrationErrorDescription
           │      └─ HotlinkedText
           ├─ GenericErrorDescription
           │      └─ HotlinkedText
           ├─ PseudoHtmlDiff (component-stack-pseudo-html.tsx)
           ├─ RuntimeError (runtime-error/index.tsx)
           ├─ CallStack (call-stack.tsx)
           └─ ErrorOverlayLayout
```

### データフロー図

```
[入力]                          [処理]                          [出力]

HMR ビルドエラー ────────▶ BuildError ─────────────▶ ターミナル風表示
                                │
                                └─ stripAnsi ──────▶ ANSIコード除去

HMR/Error Boundary ─────▶ Errors ─────────────────▶ エラーリスト表示
ランタイムエラー              │
                             ├─ isHydrationError? ─▶ PseudoHtmlDiff
                             │
                             └─ useFrames ─────────▶ ソースマップ解決
                                                         │
                                                         ▼
                                                    CodeFrame表示
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| build-error.tsx | `packages/next/src/next-devtools/dev-overlay/container/build-error.tsx` | ソース | ビルドエラー表示コンポーネント |
| errors.tsx | `packages/next/src/next-devtools/dev-overlay/container/errors.tsx` | ソース | ランタイムエラー表示コンポーネント |
| render-error.tsx | `packages/next/src/next-devtools/dev-overlay/container/runtime-error/render-error.tsx` | ソース | エラーレンダリングラッパー |
| component-stack-pseudo-html.tsx | `packages/next/src/next-devtools/dev-overlay/container/runtime-error/component-stack-pseudo-html.tsx` | ソース | ハイドレーションエラー差分表示 |
| terminal.tsx | `packages/next/src/next-devtools/dev-overlay/components/terminal/terminal.tsx` | ソース | ターミナル風表示コンポーネント |
| call-stack.tsx | `packages/next/src/next-devtools/dev-overlay/components/call-stack/call-stack.tsx` | ソース | コールスタック表示 |
| get-error-by-type.ts | `packages/next/src/next-devtools/dev-overlay/utils/get-error-by-type.ts` | ソース | エラー種別判定・ReadyRuntimeError生成 |
| react-19-hydration-error.ts | `packages/next/src/next-devtools/shared/react-19-hydration-error.ts` | ソース | React 19ハイドレーションエラー解析 |
| react-18-hydration-error.ts | `packages/next/src/next-devtools/shared/react-18-hydration-error.ts` | ソース | React 18ハイドレーションエラー解析 |
| use-open-in-editor.ts | `packages/next/src/next-devtools/dev-overlay/utils/use-open-in-editor.ts` | ソース | エディタ連携フック |
