# 通知設計書 40-App Routerルートアナウンサー

## 概要

本ドキュメントは、Next.jsのApp Routerにおけるルートアナウンサーの設計について記載する。App Routerでのページ遷移時にスクリーンリーダー向けにページタイトルまたはh1テキストをaria-live領域でアナウンスする通知機能である。

### 本通知の処理概要

App Routerルートアナウンサーは、クライアントサイドナビゲーション（ページ遷移）が発生した際に、スクリーンリーダー（JAWS、NVDA、VoiceOver等）に対してページの変更を通知する。視覚的な表示は行わず、アクセシビリティ支援技術のみが認識できるaria-live領域を使用する。

**業務上の目的・背景**：シングルページアプリケーション（SPA）では、クライアントサイドナビゲーションによるページ遷移はブラウザのフルリロードを伴わないため、スクリーンリーダーがページ変更を認識できない問題がある。WAI-ARIAのaria-live属性を使用することで、DOM変更をスクリーンリーダーに通知し、視覚障害のあるユーザーがページ遷移を認識できるようにする。これはWebアクセシビリティの基本的な要件であり、WCAG 2.1のガイドラインに沿った実装である。

**通知の送信タイミング**：`FlightRouterState`（App Routerのルーターツリー状態）が変更された場合にuseEffectが発火し、ページタイトルの変更を検出してアナウンスする。初回読み込み時はスクリーンリーダーが自動的にアナウンスするため、タイトル変更が発生した2回目以降の遷移からアナウンスされる。

**通知の受信者**：スクリーンリーダーを使用している視覚障害のあるユーザー。通常の視覚的なUIへの影響はない（CSSで非表示化されている）。

**通知内容の概要**：現在のページタイトル（`document.title`）を優先的にアナウンスする。タイトルが設定されていない場合は最初の`<h1>`要素のテキストをアナウンスする。いずれも取得できない場合はアナウンスしない。

**期待されるアクション**：スクリーンリーダーユーザーは、アナウンスされたページタイトルにより現在のページを把握し、コンテンツの読み上げを継続する。

## 通知種別

ブラウザUI / アクセシビリティ（aria-liveリージョン）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（Reactのstate更新によるDOM変更 → スクリーンリーダーが検出） |
| 優先度 | 高（aria-live="assertive"により即座にアナウンス） |
| リトライ | 無し |

### 送信先決定ロジック

すべてのブラウザクライアントに対してaria-live領域が生成される。スクリーンリーダーが有効な場合のみアナウンスが行われる。アナウンス対象のテキストは以下の優先順位で決定される：
1. `document.title`（ページタイトル）
2. `document.querySelector('h1').innerText`（最初のh1要素のテキスト）
3. アナウンスなし（テキストが取得できない場合）

## 通知テンプレート

### アクセシビリティ通知の場合

| 項目 | 内容 |
|-----|------|
| DOM要素 | カスタム要素`<next-route-announcer>`内のShadow DOM |
| ARIA属性 | `aria-live="assertive"`, `role="alert"` |
| 表示 | CSSで非表示（`clip:rect(0 0 0 0); width:1px; height:1px;`） |
| 形式 | テキストコンテンツ（createPortalで注入） |

### 本文テンプレート

```
{currentTitle}
```

`currentTitle`は`document.title`または`<h1>`要素のテキスト。

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| tree | FlightRouterState（ルーターツリー状態） | App Routerの内部状態 | Yes |
| currentTitle | アナウンスするページタイトル | document.title or document.querySelector('h1') | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| ルーター状態変更 | tree propの変更 | previousTitle !== currentTitle | FlightRouterStateが変更され、かつタイトルが前回と異なる場合 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| 初回ロード | `previousTitle.current === undefined`の場合（初回はスクリーンリーダーが自動アナウンスするため） |
| タイトル未変更 | `previousTitle.current === currentTitle`の場合、アナウンスしない |
| タイトル取得不可 | `document.title`も`<h1>`も取得できない場合、空文字列がセットされる |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[App Routerページ遷移] --> B[tree propが変更]
    B --> C[useEffect発火]
    C --> D{document.title存在?}
    D -->|Yes| E[currentTitle = document.title]
    D -->|No| F{h1要素存在?}
    F -->|Yes| G[currentTitle = h1.innerText]
    F -->|No| H[currentTitle = 空文字]
    E --> I{初回ロード?}
    G --> I
    H --> I
    I -->|Yes| J[previousTitleを更新のみ]
    I -->|No| K{タイトル変更あり?}
    K -->|Yes| L[setRouteAnnouncement実行]
    K -->|No| J
    L --> M[aria-live領域にテキスト注入]
    M --> N[スクリーンリーダーがアナウンス]
    J --> O[previousTitleを更新]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（データベースは使用しない。DOMを直接参照する）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| DOM要素の取得失敗 | Shadow DOMが利用できないブラウザの場合 | `getAnnouncerNode()`内で新規要素を作成するため、通常は発生しない |
| コンテナの重複 | 既存のアナウンサー要素が存在する場合 | `document.getElementsByName(ANNOUNCER_TYPE)[0]`で既存要素を再利用する |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（ページ遷移ごとに1回） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（アプリケーション稼働中は常時有効）

## セキュリティ考慮事項

- アナウンス内容はページタイトルまたはh1テキストのみであり、機密データは含まれない
- Shadow DOMを使用することで、アプリケーションのCSSがアナウンサー要素に影響しない
- 本コンポーネントは本番ビルドにも含まれる（アクセシビリティ機能は開発環境限定ではない）

## 備考

- `AppRouterAnnouncer`はApp Router（`app/`ディレクトリ）専用。Pages Router（`pages/`ディレクトリ）には別途`RouteAnnouncer`コンポーネント（通知No.41）が存在する
- カスタム要素`<next-route-announcer>`を使用し、Shadow DOMでDOMのカプセル化を実現。これにより、アプリケーションのスタイルがアナウンサーに影響しない
- aria-live="assertive"は、現在のスクリーンリーダーの読み上げを中断してでもアナウンスすることを意味する。role="alert"も同様の即時アナウンスのセマンティクスを提供する
- CSSスタイルは完全にオフスクリーン（`clip:rect(0 0 0 0)`, `width:1px`, `height:1px`等）に設定されており、視覚的な影響はない
- `createPortal`を使用してアナウンサーノードにテキストを注入する
- コンポーネントのアンマウント時にクリーンアップ処理があり、`<next-route-announcer>`要素がDOMから削除される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、入力となるFlightRouterStateと、DOM構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | app-router-announcer.tsx | `packages/next/src/client/components/app-router-announcer.tsx` | 定数定義（5-6行目）。`ANNOUNCER_TYPE = 'next-route-announcer'`, `ANNOUNCER_ID = '__next-route-announcer__'` |

**読解のコツ**: `FlightRouterState`はApp Routerのルーターツリー状態を表す型で、ページ遷移のたびに変更される。この変更がuseEffectの依存配列に含まれることで、ページ遷移検出のトリガーとなる。

#### Step 2: アナウンサーノードの作成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | app-router-announcer.tsx | `packages/next/src/client/components/app-router-announcer.tsx` | `getAnnouncerNode()`関数（8-28行目）。既存要素の再利用または新規作成、Shadow DOMの使用 |

**主要処理フロー**:
1. **9行目**: `document.getElementsByName(ANNOUNCER_TYPE)[0]`で既存要素を検索
2. **10行目**: 既存要素がありShadow DOMの子要素がある場合は再利用
3. **13行目**: なければ`document.createElement(ANNOUNCER_TYPE)`で新規作成
4. **15-16行目**: `ariaLive = 'assertive'`, `role = 'alert'`を設定
5. **17-20行目**: CSSでオフスクリーン化（`clip:rect(0 0 0 0)`, `width:1px`等）
6. **23-24行目**: Shadow DOM作成、アナウンサーを追加
7. **25行目**: `document.body.appendChild(container)`でDOMに追加

#### Step 3: アナウンスロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | app-router-announcer.tsx | `packages/next/src/client/components/app-router-announcer.tsx` | `AppRouterAnnouncer`コンポーネント（30-70行目）。2つのuseEffectとcreatePortalによるアナウンス |

**主要処理フロー**:
- **33-42行目**: 1つ目のuseEffect - アナウンサーノードの初期化とクリーンアップ
- **47-67行目**: 2つ目のuseEffect（依存配列: [tree]） - タイトル変更の検出とアナウンス
- **49行目**: `document.title`の取得を優先
- **52-54行目**: タイトルがなければ`document.querySelector('h1')`のテキストを取得
- **60-62行目**: 初回ロードでない場合かつタイトルが変更された場合にのみ`setRouteAnnouncement`を呼ぶ
- **66行目**: `previousTitle.current`を更新
- **69行目**: `createPortal(routeAnnouncement, portalNode)`でアナウンサーにテキストを注入

### プログラム呼び出し階層図

```
AppRouterAnnouncer [app-router-announcer.tsx:30]
    |
    +-- useEffect(初期化) [app-router-announcer.tsx:33]
    |      +-- getAnnouncerNode() [app-router-announcer.tsx:8]
    |             +-- document.createElement(ANNOUNCER_TYPE)
    |             +-- container.attachShadow({ mode: 'open' })
    |             +-- document.body.appendChild(container)
    |
    +-- useEffect(tree依存) [app-router-announcer.tsx:47]
    |      +-- document.title 取得
    |      +-- document.querySelector('h1') 取得
    |      +-- setRouteAnnouncement()
    |
    +-- createPortal() [React]
           +-- portalNode (アナウンサーDiv)
```

### データフロー図

```
[入力]                       [処理]                              [出力]

FlightRouterState        useEffect発火                      aria-live領域
(tree prop)      ------> document.title取得              --> テキスト変更
                         or h1テキスト取得                    |
                         previousTitleと比較                   v
                         setRouteAnnouncement              スクリーンリーダー
                         createPortalでDOM更新              アナウンス
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| app-router-announcer.tsx | `packages/next/src/client/components/app-router-announcer.tsx` | ソース | AppRouterAnnouncerコンポーネント本体。アナウンサーノード作成、タイトル取得、アナウンスロジック |
