# 非機能要件定義書

## 概要

本ドキュメントは、Next.jsフレームワーク（`packages/next`パッケージ）のソースコードから読み取れる非機能要件を定義したものである。Next.jsはReactベースのフルスタックWebフレームワークであり、開発サーバー、プロダクションサーバー、ビルドシステムを含む。本書では、キャッシュ設定、タイムアウト設定、リトライ設定、ロギング・監視設定、セキュリティ要件、可用性要件、拡張性・保守性要件について、コードベースの実装から抽出した内容を記載する。

## 性能要件

### レスポンスタイム

| 処理種別 | 目標値 | 備考 |
| --- | --- | --- |
| 静的ページ生成タイムアウト | 60秒以内 | `staticPageGenerationTimeout`のデフォルト値。これを超えるとビルドが失敗する（`config-shared.ts` L1518） |
| 画像最適化タイムアウト | 7秒以内 | `imgOptTimeoutInSeconds`のデフォルト値（`config-shared.ts` L1595） |
| プロキシリクエストタイムアウト | 30秒以内 | `proxyRequest`関数でのデフォルトタイムアウト。明示的に`null`を指定した場合は無制限（`proxy-request.ts` L32） |
| テレメトリ送信タイムアウト | 5秒以内 | `AbortSignal.timeout(5000)`によるテレメトリAPIリクエストのタイムアウト（`post-telemetry-payload.ts` L19-20） |
| プロセス検索タイムアウト | 250ミリ秒以内 | ポート使用中のプロセス検索に対する制限（`start-server.ts` L44） |

### スループット

| 項目 | 目標値 | 備考 |
| --- | --- | --- |
| 静的生成の最大同時実行数 | 8ページ | `staticGenerationMaxConcurrency`のデフォルト値（`config-shared.ts` L1640） |
| 静的生成のワーカー当たり最小ページ数 | 25ページ | `staticGenerationMinPagesPerWorker`のデフォルト値（`config-shared.ts` L1641） |
| CPUワーカー数 | OS CPUコア数 - 1（最小1） | `cpus`設定のデフォルト算出ロジック（`config-shared.ts` L1588-1592） |
| 画像最適化の同時実行数 | sharp依存（デフォルト: sharpの並行数 / 2） | `imgOptConcurrency`のデフォルトは`null`。sharpの並行数を環境に応じて1/2（本番）または1/4（開発）に削減（`image-optimizer.ts` L64-77） |

### キャッシュ設定

#### インメモリキャッシュ

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| 最大メモリサイズ | 50 MB | `cacheMaxMemorySize`のデフォルト値。0に設定するとインメモリキャッシュを完全に無効化（`config-shared.ts` L1483） |
| レスポンスキャッシュTTL | 10秒 | `DEFAULT_TTL_MS`。`NEXT_PRIVATE_RESPONSE_CACHE_TTL`環境変数で設定可能（`response-cache/index.ts` L44-47） |
| レスポンスキャッシュ最大エントリ数 | 150 | `DEFAULT_MAX_SIZE`。`NEXT_PRIVATE_RESPONSE_CACHE_MAX_SIZE`環境変数で設定可能（`response-cache/index.ts` L53-56） |
| LRUキャッシュ方式 | サイズベースLRU | カスタムサイズ計算関数をサポートするLRU（Least Recently Used）実装（`lru-cache.ts`） |

#### キャッシュライフプロファイル

| プロファイル名 | stale | revalidate | expire | 備考 |
| --- | --- | --- | --- | --- |
| default | staleTimes.static（デフォルト300秒） | 15分 | INFINITE_CACHE | デフォルトキャッシュプロファイル（`config-shared.ts` L1527-1531） |
| seconds | 30秒 | 1秒 | 1分 | 短期キャッシュ（`config-shared.ts` L1532-1536） |
| minutes | 5分 | 1分 | 1時間 | 分単位キャッシュ（`config-shared.ts` L1537-1541） |
| hours | 5分 | 1時間 | 1日 | 時間単位キャッシュ（`config-shared.ts` L1542-1546） |
| days | 5分 | 1日 | 1週間 | 日単位キャッシュ（`config-shared.ts` L1547-1551） |
| weeks | 5分 | 1週間 | 1ヶ月 | 週単位キャッシュ（`config-shared.ts` L1552-1556） |
| max | 5分 | 1ヶ月 | 1年 | 最長キャッシュ（`config-shared.ts` L1557-1561） |

#### クライアントルーターキャッシュ（staleTimes）

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| dynamic | 0秒 | 動的ページのクライアントキャッシュ保持時間（`config-shared.ts` L1633） |
| static | 300秒（5分） | 静的プリフェッチのクライアントキャッシュ保持時間（`config-shared.ts` L1634） |

#### 画像最適化キャッシュ

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| 最小キャッシュTTL | 14400秒（4時間） | `minimumCacheTTL`のデフォルト値（`image-config.ts` L157） |
| キャッシュバージョン | 4 | 画像キャッシュの内部バージョン管理（`image-optimizer.ts` L56） |

#### オンデマンドエントリーキャッシュ（開発モード）

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| 最大非アクティブ時間 | 60秒 | `onDemandEntries.maxInactiveAge`のデフォルト値（`config-shared.ts` L1496） |
| ページバッファ長 | 5 | 同時にメモリに保持するページ数（`config-shared.ts` L1497） |

#### Turbopackファイルシステムキャッシュ

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| 開発モード | 有効（true） | `turbopackFileSystemCacheForDev`のデフォルト値（`config-shared.ts` L1653） |
| ビルドモード | 無効（false） | `turbopackFileSystemCacheForBuild`のデフォルト値（`config-shared.ts` L1654） |

#### キャッシュハンドラー設定

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| デフォルトハンドラー | インメモリLRUキャッシュ | `cache-handlers/default.ts`で実装。エラー発生エントリは3回までリトライ後にエラーコンテンツを返却 |
| カスタムハンドラーパス | 環境変数で指定可能 | `NEXT_CACHE_HANDLER_PATH`、`NEXT_DEFAULT_CACHE_HANDLER_PATH`、`NEXT_REMOTE_CACHE_HANDLER_PATH`、`NEXT_STATIC_CACHE_HANDLER_PATH` |
| キャッシュタグ最大数 | 128 | `NEXT_CACHE_TAG_MAX_ITEMS`（`constants.ts` L32） |
| キャッシュタグ最大長 | 256文字 | `NEXT_CACHE_TAG_MAX_LENGTH`（`constants.ts` L33） |

#### 期限切れキャッシュ制御

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| expireTime | 31536000秒（1年） | SWRキャッシュの有効期限。`NEXT_PRIVATE_CDN_CONSUMED_SWR_CACHE_CONTROL`環境変数設定時はundefined（`config-shared.ts` L1515-1517） |
| INFINITE_CACHE | 0xfffffffe | revalidate=falseの場合に使用される値。V8 SMIとして最適化可能（`constants.ts` L43） |

## 可用性要件

| 項目 | 目標値 | 備考 |
| --- | --- | --- |
| 稼働率 | コード上で明示的な目標値の定義なし | フレームワークとしての稼働率目標はアプリケーション側で定義が必要 |
| 計画停止時間 | コード上で明示的な定義なし | デプロイ戦略に依存 |
| 障害復旧時間（RTO） | コード上で明示的な定義なし | インフラストラクチャ層に依存 |
| 目標復旧時点（RPO） | コード上で明示的な定義なし | データ永続化層に依存 |

### HTTP接続維持

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| HTTP Keep-Alive | 有効（true） | `httpAgentOptions.keepAlive`のデフォルト値（`config-shared.ts` L1508-1510） |
| Keep-Aliveタイムアウト | 設定可能 | `keepAliveTimeout`パラメータでサーバーのKeep-Aliveタイムアウトを制御可能（`start-server.ts` L269-270） |

### レスポンス圧縮

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| gzip圧縮 | 有効（true） | `compress`のデフォルト値（`config-shared.ts` L1490）。`setupCompression`によるレスポンス圧縮を実施 |

### エラーリカバリ

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| 静的生成リトライ回数 | 未定義（デフォルト） | `staticGenerationRetryCount`（`config-shared.ts` L1638） |
| テレメトリ送信リトライ | 最大1回（minTimeout: 500ms, factor: 1） | `post-telemetry-payload.ts` L37。エラー時は無視 |
| デフォルトキャッシュのエラーリトライ | 3回 | エラーエントリは3回MISSを返した後、エラーコンテンツを返却。期限は`Math.min(30, entry.expiration)`（`cache-handlers/default.ts` L27-30） |
| ISRフラッシュ | ディスク書き込み有効（true） | `isrFlushToDisk`のデフォルト値（`config-shared.ts` L1599） |

### DistDirロック

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| lockDistDir | 有効（true） | 複数プロセスによる同時書き込みを防止。`next dev`および`next build`時にロックを取得（`config-shared.ts` L889-899） |

## セキュリティ要件

### Server Action保護

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| Server Actionの暗号化 | AES暗号化 | `encryption.ts`でServer Actionのバウンド引数をAES暗号化。IVは16バイト |
| CSRF保護 | allowedOriginsで設定可能 | `serverActions.allowedOrigins`でリバースプロキシ経由のオリジンを許可リストに追加可能（`config-shared.ts` L687） |
| RSCリクエストヘッダー検証 | 有効（テスト/Canaryビルド時） | `validateRSCRequestHeaders`はテストモードまたは非安定ビルドでデフォルト有効（`config-shared.ts` L1629-1631） |
| Body Sizeリミット | 設定可能 | `serverActions.bodySizeLimit`でServer Actionのボディサイズを制限可能（`config-shared.ts` L679） |

### 画像最適化セキュリティ

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| Content Security Policy | `script-src 'none'; frame-src 'none'; sandbox;` | 画像レスポンスのデフォルトCSP（`image-config.ts` L163） |
| Content Disposition | attachment | 画像ダウンロード時のデフォルトディスポジション（`image-config.ts` L164） |
| SVG許可 | 無効（false） | `dangerouslyAllowSVG`のデフォルト値。XSSリスクのためデフォルトで無効（`image-config.ts` L162） |
| ローカルIP許可 | 無効（false） | `dangerouslyAllowLocalIP`のデフォルト値（`image-config.ts` L161） |
| リモートパターン | 空配列（全てブロック） | `remotePatterns`のデフォルト値。外部画像は明示的にパターンを定義する必要がある（`image-config.ts` L166） |
| 最大リダイレクト回数 | 3回 | 画像取得時の最大リダイレクト数（`image-config.ts` L159） |
| 最大レスポンスボディサイズ | 50 MB | 画像レスポンスボディの最大サイズ（`image-config.ts` L160） |

### HTTPヘッダーセキュリティ

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| x-powered-by | 有効（true） | `poweredByHeader`のデフォルト値。無効化推奨（`config-shared.ts` L1489） |
| crossOrigin | 未設定 | `<script>`タグの`crossorigin`属性。`anonymous`または`use-credentials`を設定可能（`config-shared.ts` L1260） |
| サブリソースインテグリティ（SRI） | 設定可能 | `experimental.sri`でSRIアルゴリズムを指定可能（`config-shared.ts` L428-430） |
| ETag生成 | 有効（true） | `generateEtags`のデフォルト値（`config-shared.ts` L1487） |
| React最大ヘッダー長 | 6000 | `reactMaxHeadersLength`のデフォルト値（`config-shared.ts` L1507） |

### プロキシセキュリティ

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| プロキシボディサイズ上限 | 10 MB | `proxyClientMaxBodySize`のデフォルト値（`config-shared.ts` L1650） |
| クロスサイト保護 | 有効 | `blockCrossSite`関数によるクロスサイトリクエストのブロック（`router-server.ts` L51） |
| 内部ヘッダーフィルタリング | 有効 | `filterInternalHeaders`による内部ヘッダーの除去（`router-server.ts` L50） |

## 拡張性要件

### アーキテクチャ

| 項目 | 内容 | 備考 |
| --- | --- | --- |
| 出力モード | `standalone` / `export` / デフォルト | `output`設定で出力形式を切替可能。`standalone`はDockerデプロイ向け（`config-shared.ts` L1324-1330） |
| ランタイム | Node.js / Edge | `nodejs`および`edge`ランタイムをサポート（`config-shared.ts` L240） |
| カスタムキャッシュハンドラー | 設定可能 | `cacheHandler`/`cacheHandlers`でファイルシステム以外のキャッシュバックエンドを指定可能（`config-shared.ts` L1117-1124） |
| カスタムサーバー | サポート | `customServer`フラグで独自HTTPサーバーの使用が可能（`router-server.ts` L91） |
| Turbopackメモリ制限 | 設定可能 | `turbopackMemoryLimit`でターゲットメモリ制限をバイト単位で指定可能（`config-shared.ts` L465） |

### バンドル最適化

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| 大規模ページデータ警告閾値 | 128 KB | `largePageDataBytes`のデフォルト値（`config-shared.ts` L1614） |
| WebpackビルドワーカーによるメモリOptimization | 設定可能 | `webpackBuildWorker`/`webpackMemoryOptimizations`でメモリ最適化が可能（`config-shared.ts` L631-640） |
| サーバーコード最小化 | 有効（true） | `serverMinification`のデフォルト値（`config-shared.ts` L1575） |

## 保守性要件

### ロギング設定

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| サーバーファンクションログ | 有効（true） | `logging.serverFunctions`のデフォルト値（`config-shared.ts` L1512） |
| フェッチログ（fullUrl） | 設定可能 | `logging.fetches.fullUrl`で完全URLのログ出力を制御可能（`config-shared.ts` L294） |
| フェッチログ（HMRリフレッシュ） | 設定可能 | `logging.fetches.hmrRefreshes`でHMRキャッシュ復元時のログを制御可能（`config-shared.ts` L299） |
| 受信リクエストログ | 有効（デフォルト） | `logging.incomingRequests`で無効化またはパターンベースのフィルタリングが可能（`config-shared.ts` L306） |
| ブラウザ-ターミナルログ転送 | warn（デフォルト） | `logging.browserToTerminal`で警告とエラーをターミナルに転送（`config-shared.ts` L320-321） |
| アボート後のログ非表示 | 無効（false） | `hideLogsAfterAbort`。レンダリングアボート後のログを非表示にする設定（`config-shared.ts` L907） |

### デバッグ機能

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| デバッグモード | `DEBUG=next:*`環境変数 | `setupDebug`を使用したデバッグログの有効化 |
| キャッシュデバッグ | `NEXT_PRIVATE_DEBUG_CACHE`環境変数 | デフォルトキャッシュハンドラーの詳細デバッグ（`cache-handlers/default.ts` L59-61） |
| SWCトレースプロファイリング | 無効（false） | `swcTraceProfiling`のデフォルト値（`config-shared.ts` L1611） |
| スローモジュール検出 | 未定義 | `slowModuleDetection`でビルド時間閾値を設定可能（`config-shared.ts` L805-811） |

### コード品質

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| TypeScriptビルドエラー無視 | 無効（false） | `typescript.ignoreBuildErrors`のデフォルト値（`config-shared.ts` L1474） |
| React Strict Mode | null（未設定） | `reactStrictMode`のデフォルト値（`config-shared.ts` L1506） |
| 型付きルート | 無効（false） | `typedRoutes`のデフォルト値（`config-shared.ts` L1477） |

## 運用・監視要件

### テレメトリ

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| テレメトリエンドポイント | `https://telemetry.nextjs.org/api/v1/record` | テレメトリデータの送信先（`post-telemetry-payload.ts` L25） |
| テレメトリ送信タイムアウト | 5秒 | `AbortSignal.timeout(5000)`（`post-telemetry-payload.ts` L19-20） |
| テレメトリリトライ | 最大1回（minTimeout: 500ms） | 送信失敗時は1回リトライ。それでも失敗した場合はエラーを無視（`post-telemetry-payload.ts` L37） |
| テレメトリ無効化 | `NEXT_TELEMETRY_DISABLED=1`環境変数 | テレメトリの無効化が可能 |

### トレーシング

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| OpenTelemetry統合 | サポート | `@opentelemetry/api`との統合。ユーザー独自バージョンの使用も可能（`tracer.ts` L20-36） |
| クライアントトレースメタデータ | 設定可能 | `clientTraceMetadata`でトレーシング伝播用のメタタグを設定（`config-shared.ts` L645） |
| パフォーマンスプレフィックス | `NEXT_OTEL_PERFORMANCE_PREFIX`環境変数 | OpenTelemetryパフォーマンスメトリクスのプレフィックス設定（`tracer.ts` L16） |

### 計装（Instrumentation）

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| Instrumentationフック | `instrumentation.ts`ファイル | プロジェクトルートの`instrumentation.ts`（または`src/instrumentation.ts`）でサーバー起動時のフック処理を定義可能（`constants.ts` L54） |

### MCP（Model Context Protocol）サーバー

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| MCPサーバー | 有効（true） | `experimental.mcpServer`のデフォルト値。`/_next/mcp`にAIアシスタント向けのコードインテリジェンスサーバーを公開（`config-shared.ts` L1652） |

## 備考

- 本ドキュメントはNext.jsフレームワークのソースコード（`packages/next/src/`配下）から非機能要件を抽出したものである。Next.jsはフレームワークであり、実際のアプリケーションの非機能要件は、アプリケーション固有の設定やデプロイメント環境に大きく依存する。
- レートリミット機能はフレームワーク内蔵されておらず、アプリケーション層またはインフラストラクチャ層（CDN、ロードバランサー等）で別途実装が必要である。
- 可用性要件（稼働率、RTO、RPO等）はフレームワークレベルでは定義されておらず、デプロイ先のインフラストラクチャに依存する。
- キャッシュ設定値はすべて`next.config.js`で上書き可能であり、ここに記載した値はデフォルト値である。
- 環境変数による設定変更が多数サポートされており、本番環境ではこれらの値の適切な設定が重要である。
