# API設計書

## 概要

本ドキュメントは、Node.jsランタイムが提供するHTTP APIの設計を記載しています。Node.jsのHTTPモジュールは、HTTPサーバーおよびHTTPクライアント機能を提供し、ストリーミングベースの非同期I/Oを特徴としています。

## 共通仕様

### モジュールインポート

```javascript
// CommonJS
const http = require('node:http');

// ES Modules
import * as http from 'node:http';
```

### プロトコル

- HTTP/1.0
- HTTP/1.1

### 共通ヘッダー

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| Host | ○ (HTTP/1.1) | リクエスト先のホスト名 |
| Content-Type | - | リクエスト/レスポンスボディの形式 |
| Content-Length | - | ボディのバイト長 |
| Transfer-Encoding | - | chunked等の転送エンコーディング |
| Connection | - | keep-alive / close |

### HTTPステータスコード

| ステータスコード | 説明 |
| --- | --- |
| 100 | Continue |
| 101 | Switching Protocols |
| 102 | Processing |
| 103 | Early Hints |
| 200 | OK |
| 201 | Created |
| 204 | No Content |
| 301 | Moved Permanently |
| 302 | Found |
| 304 | Not Modified |
| 400 | Bad Request |
| 401 | Unauthorized |
| 403 | Forbidden |
| 404 | Not Found |
| 408 | Request Timeout |
| 413 | Payload Too Large |
| 422 | Unprocessable Entity |
| 429 | Too Many Requests |
| 431 | Request Header Fields Too Large |
| 500 | Internal Server Error |
| 502 | Bad Gateway |
| 503 | Service Unavailable |
| 504 | Gateway Timeout |

## API一覧

| カテゴリ | API | 説明 |
| --- | --- | --- |
| Server | http.createServer() | HTTPサーバーインスタンスを作成 |
| Server | server.listen() | サーバーを指定ポートで起動 |
| Server | server.close() | サーバーを停止 |
| Server | server.setTimeout() | タイムアウトを設定 |
| Server | server.closeAllConnections() | 全接続を強制終了 |
| Server | server.closeIdleConnections() | アイドル接続を終了 |
| Client | http.request() | HTTPリクエストを発行 |
| Client | http.get() | HTTP GETリクエストを発行 |
| Agent | new http.Agent() | 接続プール管理エージェントを作成 |
| Agent | agent.createConnection() | 新規接続を作成 |
| Agent | agent.destroy() | エージェントを破棄 |
| Utility | http.validateHeaderName() | ヘッダー名を検証 |
| Utility | http.validateHeaderValue() | ヘッダー値を検証 |
| Utility | http.setMaxIdleHTTPParsers() | パーサープールサイズを設定 |
| Utility | http.setGlobalProxyFromEnv() | 環境変数からプロキシを設定 |

## 各APIエンドポイント定義

### Server API

#### 1. http.createServer([options][, requestListener])

HTTPサーバーインスタンスを新規作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| モジュール | `http` |
| 戻り値 | `http.Server` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| options | Object | - | {} | サーバー設定オプション |
| options.IncomingMessage | Class | - | http.IncomingMessage | カスタムIncomingMessageクラス |
| options.ServerResponse | Class | - | http.ServerResponse | カスタムServerResponseクラス |
| options.insecureHTTPParser | boolean | - | false | 非標準HTTPパーサーを許可 |
| options.maxHeaderSize | number | - | 16384 | 最大ヘッダーサイズ（バイト） |
| options.requireHostHeader | boolean | - | true | Hostヘッダーを必須とする |
| options.joinDuplicateHeaders | boolean | - | false | 重複ヘッダーをカンマ区切りで結合 |
| options.highWaterMark | number | - | 16384 | ストリームのhighWaterMark |
| options.requestTimeout | number | - | 300000 | リクエストタイムアウト（ミリ秒） |
| options.headersTimeout | number | - | 60000 | ヘッダー受信タイムアウト（ミリ秒） |
| options.keepAliveTimeout | number | - | 5000 | キープアライブタイムアウト（ミリ秒） |
| options.connectionsCheckingInterval | number | - | 30000 | 接続チェック間隔（ミリ秒） |
| requestListener | Function | - | - | requestイベントリスナー |

**使用例**

```javascript
const http = require('node:http');

const server = http.createServer((req, res) => {
  res.writeHead(200, { 'Content-Type': 'text/plain' });
  res.end('Hello World\n');
});

server.listen(3000, '127.0.0.1', () => {
  console.log('Server running at http://127.0.0.1:3000/');
});
```

---

#### 2. server.listen([port][, host][, backlog][, callback])

指定されたポートとホストで接続のリスニングを開始します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.Server` |
| 戻り値 | `http.Server` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| port | number | - | - | リスニングポート番号 |
| host | string | - | '0.0.0.0' | バインドするホスト |
| backlog | number | - | 511 | 保留中の接続キューの最大長 |
| callback | Function | - | - | listeningイベント時のコールバック |

**イベント**

| イベント名 | 説明 |
| --- | --- |
| listening | サーバーがリスニング開始時に発火 |
| connection | 新規TCP接続確立時に発火 |
| request | HTTPリクエスト受信時に発火 |
| error | エラー発生時に発火 |

---

#### 3. server.close([callback])

サーバーへの新規接続受付を停止し、既存接続が終了次第サーバーを閉じます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.Server` |
| 戻り値 | `http.Server` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| callback | Function | - | closeイベント時のコールバック |

---

#### 4. server.setTimeout([msecs][, callback])

ソケットのタイムアウト値を設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.Server` |
| 戻り値 | `http.Server` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| msecs | number | - | 0 | タイムアウト（ミリ秒）。0で無効化 |
| callback | Function | - | - | timeoutイベントリスナー |

---

#### 5. server.closeAllConnections()

サーバーに接続されている全てのソケットを強制的に閉じます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.Server` |
| 戻り値 | void |

---

#### 6. server.closeIdleConnections()

アイドル状態（リクエスト処理中でない）の接続のみを閉じます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.Server` |
| 戻り値 | void |

---

### ServerResponse API

#### 7. response.writeHead(statusCode[, statusMessage][, headers])

レスポンスヘッダーを送信します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.ServerResponse` |
| 戻り値 | `http.ServerResponse` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| statusCode | number | ○ | HTTPステータスコード（100-999） |
| statusMessage | string | - | ステータスメッセージ |
| headers | Object \| Array | - | レスポンスヘッダー |

**エラーレスポンス**

| エラー | 説明 |
| --- | --- |
| ERR_HTTP_HEADERS_SENT | ヘッダーが既に送信済み |
| ERR_HTTP_INVALID_STATUS_CODE | 無効なステータスコード |
| ERR_INVALID_CHAR | ステータスメッセージに無効な文字 |

---

#### 8. response.write(chunk[, encoding][, callback])

レスポンスボディのチャンクを送信します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.ServerResponse` |
| 戻り値 | boolean |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| chunk | string \| Buffer | ○ | 送信するデータ |
| encoding | string | - | エンコーディング（デフォルト: 'utf8'） |
| callback | Function | - | データフラッシュ後のコールバック |

---

#### 9. response.end([data][, encoding][, callback])

レスポンスの送信を完了します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.ServerResponse` |
| 戻り値 | `http.ServerResponse` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| data | string \| Buffer | - | 最後に送信するデータ |
| encoding | string | - | エンコーディング |
| callback | Function | - | 完了時のコールバック |

---

#### 10. response.writeContinue([callback])

HTTP 100 Continueレスポンスを送信します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.ServerResponse` |
| 戻り値 | void |

---

#### 11. response.writeEarlyHints(hints[, callback])

HTTP 103 Early Hintsレスポンスを送信します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.ServerResponse` |
| 戻り値 | void |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| hints | Object | ○ | Linkヘッダー等のヒント情報 |
| hints.link | string \| Array | - | Link headerの値 |
| callback | Function | - | 送信完了後のコールバック |

---

### Client API

#### 12. http.request(url[, options][, callback])

HTTPリクエストを発行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| モジュール | `http` |
| 戻り値 | `http.ClientRequest` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| url | string \| URL | - | - | リクエストURL |
| options.agent | http.Agent \| boolean | - | http.globalAgent | 使用するエージェント |
| options.auth | string | - | - | Basic認証（user:password形式） |
| options.createConnection | Function | - | - | カスタム接続作成関数 |
| options.defaultPort | number | - | 80 | デフォルトポート |
| options.family | number | - | - | IPファミリー（4または6） |
| options.headers | Object | - | {} | リクエストヘッダー |
| options.host | string | - | 'localhost' | ホスト名 |
| options.hostname | string | - | - | ホスト名（hostの別名） |
| options.insecureHTTPParser | boolean | - | false | 非標準HTTPパーサー許可 |
| options.localAddress | string | - | - | バインドするローカルアドレス |
| options.localPort | number | - | - | ローカルポート |
| options.lookup | Function | - | dns.lookup | DNS解決関数 |
| options.maxHeaderSize | number | - | 16384 | 最大ヘッダーサイズ |
| options.method | string | - | 'GET' | HTTPメソッド |
| options.path | string | - | '/' | リクエストパス |
| options.port | number | - | 80 | ポート番号 |
| options.protocol | string | - | 'http:' | プロトコル |
| options.setHost | boolean | - | true | Hostヘッダー自動設定 |
| options.signal | AbortSignal | - | - | アボートシグナル |
| options.socketPath | string | - | - | Unixドメインソケットパス |
| options.timeout | number | - | - | タイムアウト（ミリ秒） |
| options.uniqueHeaders | Array | - | - | 重複禁止ヘッダーリスト |
| callback | Function | - | - | responseイベントリスナー |

**使用例**

```javascript
const http = require('node:http');

const options = {
  hostname: 'example.com',
  port: 80,
  path: '/api/data',
  method: 'POST',
  headers: {
    'Content-Type': 'application/json',
  },
};

const req = http.request(options, (res) => {
  console.log(`STATUS: ${res.statusCode}`);
  res.setEncoding('utf8');
  res.on('data', (chunk) => {
    console.log(`BODY: ${chunk}`);
  });
});

req.on('error', (e) => {
  console.error(`Problem with request: ${e.message}`);
});

req.write(JSON.stringify({ key: 'value' }));
req.end();
```

---

#### 13. http.get(url[, options][, callback])

HTTP GETリクエストを発行します。req.end()が自動的に呼び出されます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| モジュール | `http` |
| 戻り値 | `http.ClientRequest` |

**パラメータ**

http.request()と同一（methodは常にGET）

**使用例**

```javascript
http.get('http://example.com/', (res) => {
  const { statusCode } = res;
  let data = '';

  res.on('data', (chunk) => { data += chunk; });
  res.on('end', () => {
    console.log(data);
  });
}).on('error', (err) => {
  console.error('Error:', err.message);
});
```

---

### ClientRequest API

#### 14. request.setTimeout(timeout[, callback])

リクエストのタイムアウトを設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.ClientRequest` |
| 戻り値 | `http.ClientRequest` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| timeout | number | ○ | タイムアウト（ミリ秒） |
| callback | Function | - | timeoutイベントリスナー |

---

#### 15. request.setNoDelay([noDelay])

Nagleアルゴリズムを無効化します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.ClientRequest` |
| 戻り値 | void |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| noDelay | boolean | - | true | 遅延無効化フラグ |

---

#### 16. request.abort()

リクエストを中止します（非推奨）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.ClientRequest` |
| 戻り値 | void |
| 備考 | request.destroy()の使用を推奨 |

---

### Agent API

#### 17. new http.Agent([options])

HTTP接続プールを管理するエージェントを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.Agent` |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| options.keepAlive | boolean | - | false | ソケットを再利用 |
| options.keepAliveMsecs | number | - | 1000 | TCPキープアライブ初期遅延 |
| options.agentKeepAliveTimeoutBuffer | number | - | 1000 | サーバーキープアライブヒントからの減算値 |
| options.maxSockets | number | - | Infinity | ホストあたり最大ソケット数 |
| options.maxTotalSockets | number | - | Infinity | 全体最大ソケット数 |
| options.maxFreeSockets | number | - | 256 | 空きソケット最大数 |
| options.scheduling | string | - | 'lifo' | スケジューリング戦略（'fifo' \| 'lifo'） |
| options.timeout | number | - | - | ソケットタイムアウト |
| options.proxyEnv | Object | - | - | プロキシ環境変数 |
| options.defaultPort | number | - | 80 | デフォルトポート |
| options.protocol | string | - | 'http:' | プロトコル |

**使用例**

```javascript
const http = require('node:http');

const agent = new http.Agent({
  keepAlive: true,
  maxSockets: 10,
  maxFreeSockets: 5,
});

http.get({
  hostname: 'example.com',
  port: 80,
  path: '/',
  agent: agent,
}, (res) => {
  // handle response
});
```

---

#### 18. agent.createConnection(options[, callback])

新規ソケット接続を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.Agent` |
| 戻り値 | `stream.Duplex` |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| options | Object | ○ | net.createConnection()と同じ |
| callback | Function | - | ソケット作成完了時コールバック |

---

#### 19. agent.getName([options])

リクエストオプションから一意の名前を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.Agent` |
| 戻り値 | string |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| options.host | string | - | ホスト名 |
| options.port | number | - | ポート番号 |
| options.localAddress | string | - | ローカルアドレス |
| options.family | number | - | IPファミリー |

---

#### 20. agent.destroy()

エージェントが管理する全ソケットを破棄します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス | `http.Agent` |
| 戻り値 | void |

---

### Utility API

#### 21. http.validateHeaderName(name)

ヘッダー名がRFC 7230に準拠しているか検証します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| モジュール | `http` |
| 戻り値 | void（エラー時は例外） |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| name | string | ○ | 検証するヘッダー名 |

---

#### 22. http.validateHeaderValue(name, value)

ヘッダー値がRFC 7230に準拠しているか検証します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| モジュール | `http` |
| 戻り値 | void（エラー時は例外） |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| name | string | ○ | ヘッダー名 |
| value | any | ○ | 検証するヘッダー値 |

---

#### 23. http.setMaxIdleHTTPParsers(max)

アイドル状態のHTTPパーサーの最大数を設定します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| モジュール | `http` |
| 戻り値 | void |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| max | number | ○ | 最大パーサー数（1以上） |

---

#### 24. http.setGlobalProxyFromEnv([env])

環境変数からグローバルプロキシ設定を行います。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| モジュール | `http` |
| 戻り値 | Function（リストア関数） |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| env | Object | - | process.env | 環境変数オブジェクト |
| env.HTTP_PROXY | string | - | - | HTTPプロキシURL |
| env.HTTPS_PROXY | string | - | - | HTTPSプロキシURL |
| env.NO_PROXY | string | - | - | プロキシ除外パターン |

---

### グローバルプロパティ

#### http.METHODS

サポートされるHTTPメソッドのリスト（ソート済み）。

| 型 | 値の例 |
| --- | --- |
| string[] | ['ACL', 'BIND', 'CHECKOUT', 'CONNECT', 'COPY', 'DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT', ...] |

---

#### http.STATUS_CODES

HTTPステータスコードとメッセージのマッピング。

| 型 | 値の例 |
| --- | --- |
| Object | { 200: 'OK', 404: 'Not Found', 500: 'Internal Server Error', ... } |

---

#### http.globalAgent

デフォルトのHTTPエージェントインスタンス。

| 型 | 説明 |
| --- | --- |
| http.Agent | 全てのHTTPクライアントリクエストで使用されるデフォルトエージェント |

---

#### http.maxHeaderSize

最大HTTPヘッダーサイズ（バイト）。デフォルトは16KB。

| 型 | デフォルト値 |
| --- | --- |
| number | 16384 |

---

## 備考

### イベント駆動アーキテクチャ

Node.js HTTPモジュールはイベント駆動型であり、主要なイベントには以下があります。

**サーバーイベント**
- `request`: HTTPリクエスト受信時
- `connection`: 新規TCP接続時
- `upgrade`: HTTPアップグレードリクエスト時
- `connect`: CONNECTメソッドリクエスト時
- `clientError`: クライアントエラー時
- `dropRequest`: リクエスト上限到達時
- `checkContinue`: Expect: 100-continueヘッダー受信時
- `timeout`: タイムアウト時
- `close`: サーバー終了時

**クライアントイベント**
- `response`: レスポンス受信時
- `socket`: ソケット割り当て時
- `connect`: CONNECTメソッドレスポンス時
- `upgrade`: アップグレードレスポンス時
- `continue`: 100 Continue受信時
- `information`: 1xx情報レスポンス受信時
- `timeout`: タイムアウト時
- `error`: エラー発生時
- `close`: 接続終了時

### 診断チャンネル

Node.jsはdiagnostics_channelを通じて以下のHTTPイベントを公開しています。

- `http.server.request.start`: サーバーリクエスト開始
- `http.server.response.created`: レスポンス作成時
- `http.server.response.finish`: レスポンス完了時
- `http.client.request.created`: クライアントリクエスト作成時
- `http.client.request.start`: クライアントリクエスト開始
- `http.client.request.error`: クライアントリクエストエラー
- `http.client.response.finish`: クライアントレスポンス完了

### パフォーマンス考慮事項

- `keepAlive: true`オプションを使用してソケット再利用を有効化
- `maxSockets`を適切に設定して同時接続数を制限
- `requestTimeout`と`headersTimeout`でスローロリス攻撃を防止
- 大きなレスポンスはストリーミングで処理
