# アーキテクチャ設計書

## 概要

本ドキュメントはNode.js（オープンソースのクロスプラットフォームJavaScriptランタイム環境）のアーキテクチャ設計について記述します。Node.jsはV8 JavaScriptエンジンとlibuvライブラリを基盤とし、イベント駆動型の非同期I/Oモデルを採用しています。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md)を参照。

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| V8 JavaScript Engine | C++バインディング | JavaScript実行エンジン |
| libuv | C++バインディング | 非同期I/O、イベントループ |
| OpenSSL | C++バインディング | 暗号化・TLS/SSL処理 |
| ICU (International Components for Unicode) | C++バインディング | 国際化・多言語サポート |
| c-ares | C++バインディング | 非同期DNS解決 |
| llhttp | C++バインディング | HTTP/1.1パーサー |
| nghttp2 | C++バインディング | HTTP/2実装 |
| ngtcp2 | C++バインディング | QUIC実装 |
| zlib | C++バインディング | データ圧縮 |
| Brotli | C++バインディング | Brotli圧縮 |
| SQLite | C++バインディング | 組み込みデータベース |
| npm | プロセス連携 | パッケージマネージャ |

## レイヤー構成

### アーキテクチャスタイル

Node.jsはレイヤードアーキテクチャを採用しており、以下の特徴を持ちます：

- **イベント駆動型アーキテクチャ**: libuvを使用したイベントループベースの非同期処理
- **バインディングレイヤー分離**: JavaScriptとC++の明確な分離
- **モジュラーアーキテクチャ**: 疎結合なビルトインモジュール群

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| JavaScript API Layer (lib/) | ユーザー向けパブリックAPI | fs.js, http.js, net.js, buffer.js, events.js等 |
| Internal JavaScript Layer (lib/internal/) | 内部実装・ユーティリティ | bootstrap/, modules/, streams/, errors.js, validators.js等 |
| Binding Layer | JavaScript-C++間のブリッジ | internalBinding(), process.binding() |
| C++ Core Layer (src/) | Node.jsコアC++実装 | node.cc, env.h, async_wrap.cc, crypto/等 |
| Dependencies Layer (deps/) | 外部ライブラリ | v8/, uv/, openssl/, llhttp/, nghttp2/等 |

### レイヤー間の依存関係ルール

1. **上位レイヤーから下位レイヤーへの単方向依存**
   - JavaScript API Layer -> Internal JavaScript Layer -> Binding Layer -> C++ Core Layer -> Dependencies Layer
2. **許可される参照**
   - パブリックAPI（lib/）は内部API（lib/internal/）を参照可能
   - 内部APIはinternalBinding()を通じてC++バインディングを参照
   - C++コアはV8、libuv等の依存ライブラリを参照
3. **禁止される参照**
   - パブリックAPIからC++バインディングへの直接参照（internalBindingを使用）
   - lib/internal/以外からのlib/internal/モジュールの直接参照
   - ユーザーランドコードからのlib/internal/モジュールへのアクセス

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| プロセス管理 | プロセスライフサイクル、ブートストラップ | lib/internal/bootstrap/, lib/internal/main/, lib/internal/process/ |
| モジュールシステム | CommonJS/ESMローダー | lib/internal/modules/cjs/, lib/internal/modules/esm/ |
| ファイルシステム | ファイル操作 | lib/fs.js, lib/internal/fs/ |
| ネットワーク | TCP/UDP、HTTP、HTTPS | lib/net.js, lib/http.js, lib/https.js, lib/dgram.js |
| 暗号化 | 暗号化・ハッシュ・証明書 | lib/crypto.js, lib/internal/crypto/, src/crypto/ |
| ストリーム | データストリーム処理 | lib/stream.js, lib/internal/streams/ |
| イベント | イベントエミッター | lib/events.js, lib/internal/events/ |
| バッファ | バイナリデータ操作 | lib/buffer.js, lib/internal/buffer.js |
| 子プロセス | プロセス生成・管理 | lib/child_process.js, lib/internal/child_process.js |
| ワーカースレッド | マルチスレッド処理 | lib/worker_threads.js, lib/internal/worker.js |
| タイマー | タイマー・スケジューリング | lib/timers.js, lib/internal/timers.js |
| コンソール | 標準入出力 | lib/console.js, lib/internal/console/ |
| テストランナー | テスト実行フレームワーク | lib/test.js, lib/internal/test_runner/ |
| Inspector | デバッグ・プロファイリング | lib/inspector.js, lib/internal/inspector/ |

### パッケージ構造

```
node/
├── lib/                          # JavaScript実装
│   ├── *.js                      # パブリックAPI（fs.js, http.js, net.js等）
│   └── internal/                 # 内部API
│       ├── bootstrap/            # ブートストラップ処理
│       │   ├── node.js           # メインブートストラップ
│       │   ├── realm.js          # Realmセットアップ
│       │   ├── switches/         # 起動モード別処理
│       │   └── web/              # Web API互換
│       ├── main/                 # エントリーポイント
│       │   ├── run_main_module.js
│       │   ├── worker_thread.js
│       │   ├── repl.js
│       │   └── eval_string.js
│       ├── modules/              # モジュールシステム
│       │   ├── cjs/              # CommonJSローダー
│       │   └── esm/              # ESMローダー
│       ├── streams/              # ストリーム実装
│       ├── crypto/               # 暗号化実装
│       ├── fs/                   # ファイルシステム実装
│       ├── process/              # プロセス関連
│       ├── errors.js             # エラー定義
│       └── validators.js         # バリデーション
├── src/                          # C++実装
│   ├── node.cc                   # メインエントリー
│   ├── env.h                     # Environment定義
│   ├── node_binding.h            # バインディング定義
│   ├── node_realm.h              # Realm実装
│   ├── async_wrap.cc             # 非同期ラッパー
│   ├── crypto/                   # 暗号化C++実装
│   ├── api/                      # 埋め込みAPI
│   └── inspector/                # Inspector実装
├── deps/                         # 依存ライブラリ
│   ├── v8/                       # V8エンジン
│   ├── uv/                       # libuv
│   ├── openssl/                  # OpenSSL
│   ├── llhttp/                   # HTTPパーサー
│   ├── nghttp2/                  # HTTP/2
│   ├── ngtcp2/                   # QUIC
│   ├── cares/                    # c-ares (DNS)
│   ├── icu-small/                # ICU
│   ├── zlib/                     # zlib
│   ├── brotli/                   # Brotli
│   ├── sqlite/                   # SQLite
│   └── npm/                      # npm
├── test/                         # テストコード
│   ├── parallel/                 # 並列テスト
│   ├── sequential/               # 順次テスト
│   ├── cctest/                   # C++テスト
│   └── common/                   # テスト共通
├── tools/                        # ビルド・開発ツール
├── doc/                          # ドキュメント
└── benchmark/                    # ベンチマーク
```

### コンポーネント依存関係

主要コンポーネント間の依存関係：

1. **ブートストラップ → モジュールシステム**
   - `lib/internal/bootstrap/node.js` が `lib/internal/modules/` を初期化

2. **モジュールシステム → ファイルシステム**
   - CJS/ESMローダーがファイル読み込みにfsを使用

3. **HTTP → ネットワーク**
   - `lib/http.js` が `lib/net.js` をベースに実装

4. **ストリーム → イベント**
   - すべてのストリームが EventEmitter を継承

5. **暗号化 → バッファ**
   - 暗号化処理がBuffer型を使用

## ミドルウェア構成

### データベース

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| 組み込みDB | SQLite | deps/sqlite | Web Storage API、組み込みDB機能 |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| CompileCache | 内蔵 | モジュールコンパイル結果キャッシュ | セッション中 |
| ResolveCache | 内蔵 | モジュール解決結果キャッシュ | セッション中 |
| V8 Code Cache | V8内蔵 | JavaScriptコードキャッシュ | 永続化可能 |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| MessageChannel | 内蔵 | Worker間通信 |
| MessagePort | 内蔵 | スレッド間メッセージング |

### 検索エンジン

該当なし（Node.jsコアには検索エンジンは含まれない）

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| V8 JavaScript Engine | deps/v8 | JavaScript実行 |
| libuv | deps/uv | 非同期I/O、イベントループ |
| OpenSSL | deps/openssl | 暗号化、TLS/SSL |
| llhttp | deps/llhttp | HTTP/1.1パース |
| nghttp2 | deps/nghttp2 | HTTP/2 |
| ngtcp2 | deps/ngtcp2 | QUIC |
| c-ares | deps/cares | 非同期DNS |
| ICU | deps/icu-small | 国際化 |
| zlib | deps/zlib | 圧縮 |
| Brotli | deps/brotli | Brotli圧縮 |
| Zstandard | deps/zstd | Zstd圧縮 |
| simdjson | deps/simdjson | 高速JSONパース |
| Ada | deps/ada | URL解析 |

## データフロー

### リクエスト処理の流れ

HTTPリクエストを例にした同期リクエストの処理フロー：

1. **libuv イベントループでソケットイベント検出**
   - TCPソケットからデータ到着をイベントループが検出
2. **C++レイヤーでデータ受信**
   - `src/tcp_wrap.cc` でデータを受信
3. **llhttpでHTTPパース**
   - HTTPリクエストを構造化データに変換
4. **JavaScript側にコールバック**
   - `_http_server.js` の `parserOnIncoming` が呼び出される
5. **IncomingMessageオブジェクト生成**
   - リクエストデータを `IncomingMessage` インスタンスとして表現
6. **ユーザーハンドラ呼び出し**
   - `'request'` イベントでユーザーコードに制御を渡す
7. **レスポンス生成**
   - `ServerResponse` オブジェクトでレスポンスを構築
8. **データ送信**
   - ストリームを通じてソケットにデータを書き込み

### 非同期処理の流れ

ファイル読み込みを例にした非同期処理のフロー：

1. **JavaScript APIコール**
   - `fs.readFile()` が呼び出される
2. **内部APIへの委譲**
   - `lib/internal/fs/promises.js` または `lib/fs.js` が処理
3. **C++バインディング呼び出し**
   - `internalBinding('fs')` を通じてC++へ
4. **libuv非同期リクエスト**
   - `uv_fs_read()` でスレッドプールにタスクを投入
5. **I/O完了**
   - スレッドプールでファイル読み込み完了
6. **コールバックキューイング**
   - イベントループの次のイテレーションでコールバックをスケジュール
7. **JavaScriptコールバック実行**
   - Promise解決またはコールバック関数の呼び出し

### データ永続化の流れ

1. **バッファまたは文字列からデータ準備**
2. **ストリーム経由でwriteメソッド呼び出し**
3. **内部バッファリング（highWaterMark制御）**
4. **C++レイヤーへの委譲**
5. **libuv経由でOS syscall実行**
6. **完了コールバックまたはPromise解決**

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| Permission Model | src/permission/, lib/internal/process/permission.js | ファイルシステム、子プロセス、ワーカー等へのアクセス制御 |
| TLS証明書検証 | src/crypto/crypto_tls.cc, lib/tls.js | HTTPS/TLS接続の証明書検証 |
| Policy | lib/internal/policy/ | モジュール整合性検証 |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | console API (lib/console.js) | stdout/stderr |
| 監査ログ | Diagnostic Report (lib/internal/report/) | ファイル |
| アクセスログ | 該当なし（アプリケーション実装） | アプリケーション依存 |
| デバッグログ | util.debuglog() | stderr |
| トレーシング | diagnostics_channel, async_hooks | メモリ/ファイル |

### エラーハンドリング

Node.jsのエラーハンドリングは `lib/internal/errors.js` で集中管理されます。

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| システムエラー | SystemError クラス | errno, syscall, path等を含むエラー |
| 引数エラー | ERR_INVALID_ARG_* | TypeError/RangeError派生 |
| モジュールエラー | ERR_MODULE_* | モジュール解決・実行エラー |
| 暗号化エラー | ERR_CRYPTO_* | 暗号化操作失敗 |
| ネットワークエラー | ECONNREFUSED等 | ソケットエラー |

特徴：
- エラーコードベースの識別（semver-major変更を避ける）
- スタックトレースのフレーム隠蔽（HideStackFramesError）
- 一貫したエラーメッセージフォーマット

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| AsyncContext | AsyncLocalStorage | 非同期コンテキストの伝播 |
| Domain（非推奨） | domain.run() | エラーハンドリングスコープ |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 単一責任原則 | モジュール分割 | 各ビルトインモジュールが特定機能に特化 |
| 開放閉鎖原則 | イベントエミッター | ユーザーがイベントハンドラを追加可能 |
| 依存性逆転原則 | バインディングレイヤー | internalBinding()による抽象化 |
| 最小権限原則 | Permission Model | --experimental-permission による制限 |
| primordials使用 | lib/internal/ | ユーザーランドのモンキーパッチ影響を回避 |

### コーディング規約

1. **JavaScript**
   - ESLint設定（eslint.config.mjs）に準拠
   - 'use strict' の使用
   - primordials経由でのビルトインメソッドアクセス（内部モジュール）
   - JSDocによるドキュメンテーション

2. **C++**
   - Google C++ Style Guide ベース（.clang-format）
   - cpplint による静的解析
   - ヘッダーファイル（.h）と実装ファイル（.cc）の分離
   - インラインヘッダー（*-inl.h）によるテンプレート実装

3. **テスト**
   - test/parallel/ で並列実行可能なテスト
   - test/sequential/ で順次実行が必要なテスト
   - common/モジュールによるテストユーティリティ

## 備考

- Node.jsはオープンソースプロジェクトであり、OpenJS Foundationの支援を受けています
- セマンティックバージョニングに従い、偶数メジャーバージョンがLTS（Long Term Support）となります
- V8とlibuvのサポートプラットフォームのサブセットをサポートします
- 本ドキュメントはNode.jsメインブランチのコード構造に基づいています
