# セキュリティ設計書

## 概要

本ドキュメントは、Node.jsランタイムにおけるセキュリティ設計について記載しています。Node.jsはサーバーサイドJavaScript実行環境として、暗号化、TLS/SSL通信、入力検証など包括的なセキュリティ機能を提供しています。本設計書では、コア暗号化モジュール(`crypto`)、TLS/SSL通信モジュール(`tls`, `https`)、および入力検証機構について詳述します。

## 認証設計

### 認証方式

Node.jsはランタイムレベルでの認証機構は提供していませんが、以下のセキュリティメカニズムをアプリケーションレベルで利用可能にしています。

| 方式 | 概要 | 実装場所 |
| --- | --- | --- |
| TLS/SSL証明書認証 | X.509証明書によるサーバー/クライアント認証 | `lib/tls.js`, `lib/internal/tls/wrap.js` |
| HMAC認証 | Hash-based Message Authentication Code | `lib/internal/crypto/hash.js` |
| デジタル署名認証 | RSA, ECDSA等による署名検証 | `lib/internal/crypto/sig.js` |

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| TLSセッション有効期限 | 設定可能(`sessionTimeout`) | `lib/internal/tls/secure-context.js`で設定 |
| TLSセッション再利用 | 対応 | `_sessionCache`によるセッションキャッシング |
| クライアント再ネゴシエーション制限 | 3回/600秒 | `CLIENT_RENEG_LIMIT`, `CLIENT_RENEG_WINDOW` |
| セッションチケット | 48バイト固定長 | `ticketKeys`による設定 |

## 認可設計

### 権限体系

Node.jsのコアモジュールは、OSレベルの権限モデルを継承します。

| ロール | 権限 | 説明 |
| --- | --- | --- |
| 実行ユーザー | 全権限 | Node.jsプロセスは実行ユーザーの全権限を継承 |
| 信頼されたコード | 無制限 | アプリケーションコードは全API利用可能 |
| 外部入力 | 制限あり | 検証後のみ処理対象 |

### アクセス制御

Node.jsの脅威モデル(`SECURITY.md`)に基づき、以下のアクセス制御ポリシーが適用されます。

- **信頼されるもの**: OS、実行コード、npm依存関係、ファイルシステム
- **信頼されないもの**: ネットワーク経由の受信データ、APIを通じて変換/検証されるデータ

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | `lib/https.js`による完全なHTTPS対応 |
| TLSバージョン | デフォルト最小: TLSv1.2、最大: TLSv1.3 |
| 証明書検証 | `checkServerIdentity`によるSAN/CN検証 |
| 暗号スイート | 設定可能(`--tls-cipher-list`) |
| ALPNプロトコル | HTTP/1.1デフォルト、カスタム設定可能 |

### TLS設定詳細

| 設定項目 | デフォルト値 | 設定方法 |
| --- | --- | --- |
| DEFAULT_MIN_VERSION | TLSv1.2 | `--tls-min-v1.0/1.1/1.2/1.3` |
| DEFAULT_MAX_VERSION | TLSv1.3 | `--tls-max-v1.2/1.3` |
| DEFAULT_ECDH_CURVE | auto | `ecdhCurve`オプション |
| 証明書失効リスト(CRL) | なし | `crl`オプションで設定 |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 対称暗号 | AES-GCM, ChaCha20-Poly1305, AES-CBC/CTR/ECB等(OpenSSL依存) |
| 非対称暗号 | RSA(PKCS1/OAEP), ECDSA, Ed25519, Ed448 |
| ハッシュ | SHA-1, SHA-256, SHA-384, SHA-512, SHA3-256/384/512, SHAKE128/256 |
| 鍵導出 | PBKDF2, scrypt, Argon2, HKDF |
| KEM | ML-KEM(耐量子暗号) |
| MAC | HMAC |

### 暗号化実装詳細

| 機能 | 実装ファイル | 主要クラス/関数 |
| --- | --- | --- |
| 対称暗号 | `lib/internal/crypto/cipher.js` | `Cipheriv`, `Decipheriv` |
| ハッシュ | `lib/internal/crypto/hash.js` | `Hash`, `Hmac`, `hash()` |
| 署名/検証 | `lib/internal/crypto/sig.js` | `Sign`, `Verify` |
| 鍵生成 | `lib/internal/crypto/keygen.js` | `generateKeyPair`, `generateKey` |
| 鍵交換 | `lib/internal/crypto/diffiehellman.js` | `DiffieHellman`, `ECDH` |
| 乱数生成 | `lib/internal/crypto/random.js` | `randomBytes`, `randomUUID`, `randomInt` |
| WebCrypto | `lib/internal/crypto/webcrypto.js` | `subtle` |

### 機密情報管理

Node.jsは以下の機密情報管理機能を提供しています。

- **FIPS対応**: `--force-fips`オプションによるFIPSモード強制
- **セキュアヒープ**: `secureHeapUsed()`による使用状況監視
- **タイミングセーフ比較**: `timingSafeEqual()`によるタイミング攻撃対策
- **環境変数CA証明書**: `NODE_EXTRA_CA_CERTS`による追加CA証明書

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | Node.jsコアでは未提供(アプリケーションレベルで実装) |
| SQLインジェクション | Node.jsコアでは未提供(アプリケーションレベルで実装) |
| CSRF | Node.jsコアでは未提供(アプリケーションレベルで実装) |
| 型検証 | `lib/internal/validators.js`による包括的な型検証 |
| 証明書検証 | SAN/CN検証、ワイルドカード対応 |
| HTTPリクエストスマグリング | HTTPパーサーでの対策(CWE-444) |

### 入力検証機能

`lib/internal/validators.js`で提供される検証関数:

| 関数 | 用途 |
| --- | --- |
| `validateString` | 文字列型検証 |
| `validateNumber` | 数値型検証(範囲指定可能) |
| `validateInteger` | 整数型検証 |
| `validateInt32` | 32ビット符号付き整数検証 |
| `validateUint32` | 32ビット符号なし整数検証 |
| `validateBoolean` | 真偽値検証 |
| `validateObject` | オブジェクト型検証 |
| `validateArray` | 配列型検証 |
| `validateBuffer` | Buffer/TypedArray/DataView検証 |
| `validateEncoding` | エンコーディング検証 |
| `validatePort` | ポート番号検証(0-65535) |
| `validateAbortSignal` | AbortSignal検証 |
| `validateFunction` | 関数型検証 |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| セキュリティ警告 | DHパラメータ警告等 | コンソール出力(アプリケーション依存) |
| 非推奨警告 | DEP0093(fips)、DEP0115(prng)等 | コンソール出力 |

Node.jsコアでは、包括的な監査ログ機能は提供されていません。セキュリティ監査ログはアプリケーションレベルで実装する必要があります。

## 脆弱性報告とセキュリティポリシー

Node.jsは以下のセキュリティポリシーを維持しています。

- **脆弱性報告**: HackerOne経由での報告受付
- **CVE発行**: 重大な脆弱性にはCVE発行
- **バグバウンティ**: 有効な脆弱性報告に対する報奨金プログラム
- **開示ポリシー**: 修正準備完了後、72時間のエンバーゴ期間

### 脆弱性に該当するもの

- 不正な証明書検証(CWE-295)
- HTTPリクエストスマグリング(CWE-444)
- 暗号ステップの欠落(CWE-325)
- 外部によるシステム/設定制御(CWE-15)

### 脆弱性に該当しないもの

- 悪意あるサードパーティモジュール(CWE-1357)
- プロトタイプ汚染攻撃(CWE-1321)
- 制御されていない検索パス要素(CWE-427)

## 備考

- Node.jsは実行コードを信頼するモデルを採用しています。したがって、アプリケーションコードの入力検証は開発者の責任となります。
- 実験的機能やコンパイル時フラグ背後の機能に対する脆弱性はセキュリティ問題として扱われません。
- 本設計書はNode.jsコアの実装に基づいており、アプリケーション固有のセキュリティ要件は別途設計が必要です。
