# バッチ設計書 11-commit-queue.sh

## 概要

本ドキュメントは、Node.jsプロジェクトのGitHub Actionsで使用されるコミットキュー処理スクリプト（commit-queue.sh）の設計仕様を記載する。

### 本バッチの処理概要

このバッチは、Node.jsプロジェクトにおけるPull Request（PR）の自動マージ処理を行うコミットキューシステムの中核を担う。`commit-queue`ラベルが付与されたPRを自動的に検出し、CIの完了を確認した上で、適切なマージ方式（スカッシュ、リベース、または単一コミット）でmainブランチにマージする。

**業務上の目的・背景**：Node.jsのような大規模オープンソースプロジェクトでは、多数のコントリビューターからのPRを効率的に処理する必要がある。手動でのマージ作業は時間がかかり、人為的ミスも発生しやすい。本バッチにより、レビュー完了後のマージ処理を自動化し、メンテナーの負担を軽減するとともに、一貫性のあるマージ処理を実現する。

**バッチの実行タイミング**：GitHub Actionsによるイベント駆動型で実行される。`commit-queue`ラベルがPRに付与されたタイミング、または定期的なスケジュール実行により起動する。

**主要な処理内容**：
1. 対象PRのラベル情報を取得し、CIの状態を確認
2. CI実行中または`request-ci`ラベル付きの場合はスキップ
3. `commit-queue`ラベルを削除し、マージポリシーを決定
4. `git node land`コマンドまたはGitHub APIでマージを実行
5. マージ成功時はコメントを投稿、失敗時は`commit-queue-failed`ラベルを付与

**前後の処理との関連**：CIワークフロー（start-ci.sh）によるテスト完了後に実行される。マージ完了後は、リリースワークフローに引き継がれる可能性がある。

**影響範囲**：mainブランチへのコード統合、PRのステータス変更、GitHubラベル・コメントの更新に影響を与える。

## バッチ種別

CI/CD自動化（Pull Requestマージ処理）

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | イベント駆動 / 定期実行 |
| 実行時刻 | 随時（GitHub Actionsトリガー） |
| 実行曜日 | 全曜日 |
| 実行日 | 毎日 |
| トリガー | GitHub Actions（ラベル付与イベント / スケジュール） |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| GitHub CLI（gh）のインストール | PRの操作に必要 |
| jqコマンドのインストール | JSON解析に必要 |
| git-node CLIのインストール | `git node land`コマンドの実行に必要 |
| リポジトリへの書き込み権限 | mainブランチへのプッシュ権限が必要 |
| GITHUB_TOKEN環境変数 | GitHub API認証用 |

### 実行可否判定

1. PRに`request-ci`ラベルが付いている場合は処理をスキップ（CIの開始待ち）
2. PRのCIチェックが`pending`状態の場合は処理をスキップ（CI実行中）
3. 上記以外の場合、マージ処理を実行

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| OWNER | string | Yes | - | リポジトリオーナー名（第1引数） |
| REPOSITORY | string | Yes | - | リポジトリ名（第2引数） |
| PR番号リスト | string[] | Yes | - | 処理対象のPR番号（第3引数以降） |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| GitHub API | JSON | PRのラベル情報、CIステータス |
| Gitリポジトリ | - | ローカルリポジトリの状態 |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| GitHub PR | コメント | マージ結果のコメント（成功時：コミットSHA、失敗時：エラー詳細） |
| GitHub PR | ラベル | 失敗時に`commit-queue-failed`ラベルを付与 |
| mainブランチ | Git | マージされたコミット |
| 標準出力 | テキスト | 処理ログ |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | output（一時ファイル）、labels.json（一時ファイル） |
| 出力先 | カレントディレクトリ |
| 文字コード | UTF-8 |
| 区切り文字 | N/A |

## 処理フロー

### 処理シーケンス

```
1. Git設定の初期化
   └─ ボット用のメール・ユーザー名を設定
2. 各PRに対してループ処理開始
   └─ 引数で渡されたPR番号を順次処理
3. PRのラベル情報を取得
   └─ `gh pr view`でラベルをJSONで取得
4. request-ciラベルの確認
   └─ 存在する場合はスキップ（CI開始待ち）
5. CIステータスの確認
   └─ pending状態の場合はスキップ（CI実行中）
6. commit-queueラベルの削除
   └─ 処理開始を示すためラベルを除去
7. マージポリシーの決定
   └─ commit-queue-squash → fixupAll
   └─ commit-queue-rebase → リベース
   └─ それ以外 → oneCommitMax
8. マージ処理の実行
   └─ git node land または GitHub API
9. 結果の判定と後処理
   └─ 成功: コメント投稿、PR close
   └─ 失敗: ラベル付与、エラーコメント
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[Git設定初期化]
    B --> C[PRループ開始]
    C --> D{request-ciラベル?}
    D -->|あり| E[スキップ]
    D -->|なし| F{CI pending?}
    F -->|はい| E
    F -->|いいえ| G[commit-queueラベル削除]
    G --> H{マージポリシー判定}
    H -->|squash| I[fixupAllモード]
    H -->|rebase| J[リベースモード]
    H -->|その他| K[oneCommitMaxモード]
    I --> L[git node land実行]
    J --> L
    K --> L
    L --> M{マージ成功?}
    M -->|はい| N[コミットコメント投稿]
    M -->|いいえ| O[失敗ラベル付与]
    O --> P[エラーコメント投稿]
    P --> Q[git node land --abort]
    N --> R{リベースモード?}
    R -->|はい| S[PR close]
    R -->|いいえ| T[次のPRへ]
    S --> T
    Q --> T
    E --> T
    T --> U{残りPRあり?}
    U -->|はい| C
    U -->|いいえ| V[バッチ終了]
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベースを直接操作しない。GitHub APIを通じたPR状態の更新が主な操作となる。

| 処理 | 対象 | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ラベル操作 | GitHub PR | UPDATE | commit-queue, commit-queue-failedラベルの追加・削除 |
| コメント投稿 | GitHub PR | INSERT | マージ結果のコメント追加 |
| マージ実行 | GitHub PR | UPDATE | PRのマージ状態を更新 |
| PRクローズ | GitHub PR | UPDATE | リベースモード時にPRをクローズ |

### テーブル別操作詳細

N/A（データベース操作なし）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | マージ失敗 | git node landコマンドの失敗 | commit-queue-failedラベル付与、エラーコメント投稿 |
| - | プッシュ失敗 | リモートへのプッシュ失敗 | commit-queue-failedラベル付与、エラーコメント投稿 |
| - | API失敗 | GitHub APIのエラー | commit-queue-failedラベル付与、エラーコメント投稿 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0回（即時失敗） |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

### 障害時対応

1. `git node land --abort`でマージ処理を中断
2. `commit-queue-failed`ラベルを付与
3. エラー詳細をPRコメントとして投稿
4. GitHub Actionsの実行URLをコメントに含める
5. 次のPRの処理に移行（他のPRには影響を与えない）

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | PR単位 |
| コミットタイミング | 各PR処理完了時 |
| ロールバック条件 | git node land失敗時に--abortで中断 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1〜10 PR/実行 |
| 目標処理時間 | 5分以内/PR |
| メモリ使用量上限 | GitHub Actions標準リソース |

## 排他制御

- 複数のGitHub Actions実行が同時に同じPRを処理しないよう、ラベルの削除で排他制御
- `commit-queue`ラベル削除後は他の実行インスタンスはそのPRを処理しない

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 開始ログ | PR処理開始時 | 対象PR番号 |
| スキップログ | スキップ時 | スキップ理由（CI待ち、CI実行中） |
| 処理ログ | マージ処理中 | git node landの出力 |
| 成功ログ | マージ成功時 | コミットSHA |
| エラーログ | マージ失敗時 | エラー詳細 |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 処理時間 | 10分/PR | GitHub Actions |
| エラー発生 | 1件以上 | PRコメント、GitHub Actions |

## 備考

- `set -xe`オプションにより、コマンドの実行内容が表示され、エラー発生時は即座に終了する
- ボットアカウント（github-bot@iojs.org）でコミットが作成される
- squashマージの場合はGitHub APIの`PUT /repos/{owner}/{repo}/pulls/{pr}/merge`を使用
- リベースマージの場合は`git push`でリモートにプッシュ後、`gh pr close`でPRをクローズ
