# バッチ設計書 17-update-npm.sh

## 概要

本ドキュメントは、Node.jsプロジェクトの依存関係更新で使用されるnpm更新スクリプト（update-npm.sh）の設計仕様を記載する。

### 本バッチの処理概要

このバッチは、Node.jsのdeps/npmディレクトリに配置されているnpmパッケージを指定バージョンに更新するスクリプトである。npm registryからtarballをダウンロードし、SHA256チェックサムを検証した後、既存のnpmディレクトリを置き換える。

**業務上の目的・背景**：Node.jsはnpmをバンドルして配布しており、npmのバージョンアップに追従する必要がある。手動での更新は手順が複雑でミスが発生しやすい。本バッチにより、npmの更新作業を自動化し、安全かつ確実にバージョンアップを行う。

**バッチの実行タイミング**：新しいnpmバージョンがリリースされた際に手動で実行する。GitHub Actionsによる定期チェックでも利用可能。

**主要な処理内容**：
1. 指定バージョンのnpm tarball URLを取得
2. tarballをダウンロード
3. SHA256チェックサムを検証
4. 既存のdeps/npmディレクトリを削除
5. tarballを展開してdeps/npmに配置
6. maintaining-dependencies.mdのバージョン番号を更新

**前後の処理との関連**：更新後はテストスイートの実行が推奨される。ライセンス更新（license-builder.sh）が必要になる場合がある。

**影響範囲**：deps/npmディレクトリ全体、maintaining-dependencies.mdファイルに影響を与える。

## バッチ種別

依存関係更新（パッケージ更新）

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（新バージョンリリース時） |
| 実行時刻 | 任意 |
| 実行曜日 | 全曜日 |
| 実行日 | 毎日 |
| トリガー | 手動 / GitHub Actions |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Node.jsのインストール | npm viewコマンド実行に必要 |
| curlコマンドのインストール | tarballダウンロードに必要 |
| tarコマンドのインストール | tarball展開に必要 |
| shasum（またはsha256sum）のインストール | チェックサム検証に必要 |
| mktempコマンドのインストール | 一時ディレクトリ作成に必要 |

### 実行可否判定

1. NPM_VERSION（更新対象バージョン）が引数で指定されていること
2. 必要なコマンドがインストールされていること
3. インターネット接続が可能であること

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| NPM_VERSION | string | Yes | - | 更新対象のnpmバージョン（第1引数） |
| NODE | string | No | out/Release/node または system node | Node.js実行ファイルパス（環境変数） |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| コマンドライン引数 | string | npmバージョン |
| npm registry | URL/tarball | npm tarball URL |
| npm registry | tar.gz | npm tarball本体 |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| deps/npm | ディレクトリ | 更新されたnpmパッケージ |
| maintaining-dependencies.md | Markdown | バージョン番号の更新 |
| 標準出力 | テキスト | 処理ログ、NEW_VERSION出力 |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | deps/npm/* |
| 出力先 | deps/npm/ |
| 文字コード | UTF-8 |
| 区切り文字 | N/A |

## 処理フロー

### 処理シーケンス

```
1. 引数チェック
   └─ NPM_VERSIONが指定されているか確認
2. 一時ワークスペースの作成
   └─ mktemp -d
3. cleanup関数の設定
   └─ trap INT TERM EXIT
4. tarball URLの取得
   └─ npm view npm@VERSION dist.tarball
5. tarballのダウンロード
   └─ curl -s "$NPM_TARBALL" > "$NPM_TGZ"
6. SHA256チェックサムの検証
   └─ log_and_verify_sha256sum
7. 既存npmディレクトリの削除
   └─ rm -rf "$DEPS_DIR/npm"
8. 新規npmディレクトリの作成
   └─ mkdir "$DEPS_DIR/npm"
9. tarballの展開
   └─ tar zxvf --strip-component=1 -C deps/npm
10. バージョン情報の更新
    └─ finalize_version_update
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B{NPM_VERSION指定?}
    B -->|いいえ| C[エラー終了 exit 1]
    B -->|はい| D[一時ワークスペース作成]
    D --> E[cleanup trap設定]
    E --> F[tarball URL取得]
    F --> G[tarballダウンロード]
    G --> H[SHA256チェックサム検証]
    H --> I{検証成功?}
    I -->|いいえ| J[エラー終了]
    I -->|はい| K[既存npm削除]
    K --> L[新規npmディレクトリ作成]
    L --> M[tarball展開]
    M --> N[バージョン情報更新]
    N --> O[cleanup実行]
    O --> P[バッチ終了]
    J --> O
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベースを操作しない。ファイルシステム操作のみ。

| 処理 | 対象 | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ディレクトリ削除 | deps/npm | DELETE | 既存npmディレクトリの削除 |
| ディレクトリ作成 | deps/npm | INSERT | 新規npmディレクトリの作成 |
| ファイル展開 | deps/npm/* | INSERT | tarball内容の展開 |
| ファイル更新 | maintaining-dependencies.md | UPDATE | バージョン番号の更新 |

### テーブル別操作詳細

N/A（データベース操作なし）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | 引数エラー | NPM_VERSIONが未指定 | バージョンを指定して再実行 |
| - | ダウンロードエラー | curlコマンド失敗 | ネットワーク接続を確認 |
| - | チェックサムエラー | SHA256検証失敗 | tarballの整合性を確認、再ダウンロード |
| - | 展開エラー | tarコマンド失敗 | ディスク容量・権限を確認 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0回（即時失敗） |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

### 障害時対応

1. エラーメッセージを確認
2. ネットワーク接続を確認
3. 指定バージョンが存在するか確認（npm view npm@VERSION）
4. ディスク容量・権限を確認
5. 一時ファイルが残っている場合は手動削除
6. cleanup関数により一時ワークスペースは自動削除

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 全体（部分的失敗時はdeps/npmが削除状態） |
| コミットタイミング | 処理完了時 |
| ロールバック条件 | N/A（手動での復旧が必要） |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1回/実行 |
| 目標処理時間 | 5分以内 |
| メモリ使用量上限 | 標準シェル環境 |

## 排他制御

- 同時実行は想定しない
- deps/npmディレクトリの排他ロックは行わない

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 開始ログ | 処理開始時 | Making temporary workspace |
| チェックサムログ | 検証時 | SHA256チェックサム値 |
| 完了ログ | 処理完了時 | All done! |
| バージョンログ | 最終行 | NEW_VERSION=xxx |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 処理時間 | 10分 | 標準出力 |

## 備考

- `set -e`オプションにより、エラー発生時は即座に終了する
- utils.shからユーティリティ関数をインポート
  - `log_and_verify_sha256sum`: SHA256チェックサムの検証
  - `finalize_version_update`: バージョン情報の更新とNEW_VERSION出力
- 一時ワークスペースはtrapによりINT/TERM/EXIT時に自動削除
- tarball展開時に`--strip-component=1`で余分なディレクトリ階層を除去
- NODE環境変数でカスタムNode.jsバイナリを指定可能
- 出力の最終行`NEW_VERSION=xxx`はGitHub Actionsの`$GITHUB_ENV`設定用
