# バッチ設計書 32-update-ada.sh

## 概要

本ドキュメントは、Node.jsプロジェクトの依存ライブラリであるada URL解析ライブラリを最新バージョンに更新するシェルスクリプト `update-ada.sh` の設計仕様を記述したものである。

### 本バッチの処理概要

本バッチは、WHATWG URL標準に準拠した高速URL解析ライブラリであるadaをNode.jsの依存ライブラリとして最新バージョンに更新する処理を自動化するものである。

**業務上の目的・背景**：Node.jsのURL処理モジュールはadaライブラリを基盤として実装されている。セキュリティ脆弱性の修正、URL解析仕様への追従、パフォーマンス改善などの理由から、定期的に最新バージョンへの追従が必要である。手動での更新作業は煩雑でエラーが発生しやすいため、自動化スクリプトによって一貫した更新プロセスを提供する。

**バッチの実行タイミング**：手動実行または依存関係更新のCI/CDパイプラインからトリガーされる。新バージョンがリリースされた際に開発者が必要に応じて実行する。

**主要な処理内容**：
1. GitHub APIを使用してadaの最新リリースバージョンを取得
2. 現在のバージョンと比較し、更新が必要か判定
3. 一時ワークスペースを作成してシングルヘッダー形式のソースをダウンロード
4. SHA256チェックサムの検証を実施
5. 既存のGYP/GNビルドファイルとREADME.mdを保持しつつ、ライブラリファイルを置換
6. MITライセンスファイルをダウンロードして配置
7. maintaining-dependencies.mdのバージョン情報を更新

**前後の処理との関連**：本バッチは単独で実行可能であるが、更新後はNode.js全体のビルドとテストが必要となる。更新結果はGitコミットとして記録され、PRを通じてレビューされる。

**影響範囲**：`deps/ada/` ディレクトリ配下のファイル、`doc/contributing/maintaining/maintaining-dependencies.md` のバージョン記載に影響を与える。

## バッチ種別

依存関係更新 / ライブラリ管理

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（新バージョンリリース時） |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 手動 / CI/CDパイプライン |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Node.js実行環境 | `out/Release/node` または `node` コマンドが利用可能であること |
| ネットワーク接続 | GitHub API および GitHubリリースページへのアクセスが可能であること |
| utils.sh | `tools/dep_updaters/utils.sh` が存在すること |
| curl | curlコマンドが利用可能であること |
| unzip | unzipコマンドが利用可能であること |

### 実行可否判定

GitHub APIから取得した最新バージョンと `deps/ada/ada.h` に定義されている現在のバージョン（ADA_VERSION）を比較する。バージョンが同一の場合はスクリプトを終了し、異なる場合は更新処理を続行する。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| NODE | 環境変数 | No | `$BASE_DIR/out/Release/node` または `node` | 使用するNode.js実行ファイルのパス |
| GITHUB_TOKEN | 環境変数 | No | なし | GitHub API認証用トークン（レート制限回避用） |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| GitHub API | JSON | `https://api.github.com/repos/ada-url/ada/releases/latest` から最新バージョン情報を取得 |
| deps/ada/ada.h | C Header | 現在のバージョン情報（ADA_VERSION）を取得 |
| GitHub Releases | ZIP | `https://github.com/ada-url/ada/releases/download/v{VERSION}/singleheader.zip` からソースアーカイブを取得 |
| GitHub Raw | テキスト | `https://raw.githubusercontent.com/ada-url/ada/HEAD/LICENSE-MIT` からライセンスファイルを取得 |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| deps/ada/ada.h | C Header | adaライブラリのシングルヘッダー |
| deps/ada/ada.cpp | C++ Source | adaライブラリの実装ファイル |
| deps/ada/LICENSE-MIT | テキスト | MITライセンスファイル |
| deps/ada/*.gyp | GYP | GYPビルドファイル（既存を維持） |
| deps/ada/*.gn, *.gni | GN | GNビルドファイル（既存を維持） |
| deps/ada/README.md | Markdown | READMEファイル（既存を維持） |
| 標準出力 | テキスト | `NEW_VERSION={version}` の形式で新バージョンを出力 |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | ada.h, ada.cpp, LICENSE-MIT |
| 出力先 | deps/ada/ |
| 文字コード | UTF-8 |
| 区切り文字 | N/A |

## 処理フロー

### 処理シーケンス

```
1. 初期化
   └─ BASE_DIR, DEPS_DIR, NODE変数の設定
   └─ utils.shの読み込み
2. 最新バージョン取得
   └─ GitHub APIでada-url/adaの最新リリースを取得
   └─ tag_nameから'v'プレフィックスを除去してバージョン番号を取得
3. 現在バージョン取得
   └─ deps/ada/ada.hからADA_VERSIONを抽出
4. バージョン比較
   └─ compare_dependency_version関数でバージョンを比較
   └─ 同一の場合はスクリプト終了
5. 一時ワークスペース作成
   └─ mktemp -dで一時ディレクトリを作成
   └─ trapでクリーンアップ関数を設定
6. シングルヘッダーZIPダウンロード
   └─ curlでsingleheader.zipをダウンロード
   └─ SHA256チェックサムを検証
7. ライセンスファイルダウンロード
   └─ curlでLICENSE-MITをダウンロード
8. ビルドファイル保持
   └─ 既存の*.gyp, *.gn, *.gni, README.mdをワークスペースに移動
9. 既存ファイル削除・新規ファイル配置
   └─ deps/adaを削除
   └─ ワークスペースをdeps/adaとして移動
10. バージョン情報更新
    └─ finalize_version_update関数でドキュメント更新
    └─ 新バージョンを標準出力
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[環境変数・utils.sh読み込み]
    B --> C[GitHub APIで最新バージョン取得]
    C --> D[現在バージョンをada.hから取得]
    D --> E{バージョン比較}
    E -->|同一| F[スキップして終了]
    E -->|異なる| G[一時ワークスペース作成]
    G --> H[singleheader.zipダウンロード]
    H --> I[SHA256チェックサム検証]
    I --> J[ZIP展開]
    J --> K[LICENSE-MITダウンロード]
    K --> L[既存GYP/GN/READMEファイル保持]
    L --> M[deps/ada削除]
    M --> N[ワークスペースをdeps/adaへ移動]
    N --> O[バージョン情報更新]
    O --> P[バッチ終了]
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | ネットワークエラー | GitHub APIへの接続失敗 | ネットワーク接続を確認し、再実行 |
| 1 | API エラー | GitHub APIがエラーレスポンスを返却 | GITHUB_TOKENを設定してレート制限を回避 |
| 1 | チェックサムエラー | SHA256検証失敗 | ダウンロードを再試行、ファイル破損を確認 |
| 1 | ファイル操作エラー | ディレクトリ作成・削除失敗 | 権限を確認し、手動でクリーンアップ後に再実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 自動リトライなし |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

### 障害時対応

1. スクリプトは `set -e` によりエラー発生時に即座に終了する
2. 一時ワークスペースは `trap cleanup INT TERM EXIT` により自動クリーンアップされる
3. 失敗した場合は `deps/ada/` ディレクトリの状態を確認し、必要に応じてgit checkoutで復元する

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | N/A（ファイルシステム操作） |
| コミットタイミング | N/A |
| ロールバック条件 | スクリプト失敗時はgit checkoutで復元可能 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1回の実行で1ライブラリを更新 |
| 目標処理時間 | 数分以内（ネットワーク速度に依存） |
| メモリ使用量上限 | 特に制限なし |

## 排他制御

同一のdeps/adaディレクトリに対する同時実行は想定されていない。複数プロセスからの同時実行は避けること。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 進捗ログ | 各処理ステップ | "Making temporary workspace...", "Fetching ada source archive...", "Replacing existing ada" 等 |
| 比較ログ | バージョン比較時 | "Comparing {new_version} with {current_version}" |
| スキップログ | バージョン同一時 | "Skipped because ada is on the latest version." |
| 完了ログ | 処理完了時 | "All done!" および gitコマンド例 |
| バージョンログ | 最終行 | "NEW_VERSION={version}" |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 終了コード | 0以外 | CI/CDシステム通知 |
| GitHub APIレート制限 | 残り回数が少ない場合 | GITHUB_TOKEN設定を推奨 |

## 備考

- adaはWHATWG URL標準に準拠した高速URL解析ライブラリである
- シングルヘッダー形式（ada.h + ada.cpp）で配布されるため、展開処理が簡潔である
- GYP/GNビルドファイルおよびREADME.mdはNode.js固有のカスタマイズが含まれるため、更新時に保持される
- 更新後は `git add -A deps/ada && git commit -m "deps: update ada to {version}"` でコミットする
