# バッチ設計書 33-update-simdjson.sh

## 概要

本ドキュメントは、Node.jsプロジェクトの依存ライブラリであるsimdjson高速JSONパーサーを最新バージョンに更新するシェルスクリプト `update-simdjson.sh` の設計仕様を記述したものである。

### 本バッチの処理概要

本バッチは、SIMD命令を活用した超高速JSONパーサーライブラリであるsimdjsonをNode.jsの依存ライブラリとして最新バージョンに更新する処理を自動化するものである。

**業務上の目的・背景**：Node.jsはJSONパース処理において高いパフォーマンスが求められる場面でsimdjsonライブラリを活用している。セキュリティ脆弱性の修正、パフォーマンス改善、新しいCPUアーキテクチャへの対応などの理由から、定期的に最新バージョンへの追従が必要である。手動での更新作業は煩雑でエラーが発生しやすいため、自動化スクリプトによって一貫した更新プロセスを提供する。

**バッチの実行タイミング**：手動実行または依存関係更新のCI/CDパイプラインからトリガーされる。新バージョンがリリースされた際に開発者が必要に応じて実行する。

**主要な処理内容**：
1. GitHub APIを使用してsimdjsonの最新リリースバージョンを取得
2. 現在のバージョンと比較し、更新が必要か判定
3. 一時ワークスペースを作成してソースアーカイブをダウンロード
4. アーカイブを展開し、シングルヘッダー形式のファイルを抽出
5. simdjson.h、simdjson.cpp、LICENSEファイルを配置
6. maintaining-dependencies.mdのバージョン情報を更新

**前後の処理との関連**：本バッチは単独で実行可能であるが、更新後はNode.js全体のビルドとテストが必要となる。更新結果はGitコミットとして記録され、PRを通じてレビューされる。

**影響範囲**：`deps/simdjson/` ディレクトリ配下のファイル、`doc/contributing/maintaining/maintaining-dependencies.md` のバージョン記載に影響を与える。

## バッチ種別

依存関係更新 / ライブラリ管理

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（新バージョンリリース時） |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 手動 / CI/CDパイプライン |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Node.js実行環境 | `out/Release/node` または `node` コマンドが利用可能であること |
| ネットワーク接続 | GitHub API および GitHubリリースページへのアクセスが可能であること |
| utils.sh | `tools/dep_updaters/utils.sh` が存在すること |
| curl | curlコマンドが利用可能であること |
| unzip | unzipコマンドが利用可能であること |

### 実行可否判定

GitHub APIから取得した最新バージョンと `deps/simdjson/simdjson.h` に定義されている現在のバージョン（SIMDJSON_VERSION）を比較する。バージョンが同一の場合はスクリプトを終了し、異なる場合は更新処理を続行する。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| NODE | 環境変数 | No | `$BASE_DIR/out/Release/node` または `node` | 使用するNode.js実行ファイルのパス |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| GitHub API | JSON | `https://api.github.com/repos/simdjson/simdjson/releases/latest` から最新バージョン情報を取得 |
| deps/simdjson/simdjson.h | C Header | 現在のバージョン情報（SIMDJSON_VERSION）を取得 |
| GitHub Releases | ZIP | `https://github.com/simdjson/simdjson/archive/refs/tags/v{VERSION}.zip` からソースアーカイブを取得 |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| deps/simdjson/simdjson.h | C Header | simdjsonライブラリのシングルヘッダー |
| deps/simdjson/simdjson.cpp | C++ Source | simdjsonライブラリの実装ファイル |
| deps/simdjson/LICENSE | テキスト | Apache 2.0ライセンスファイル |
| 標準出力 | テキスト | `NEW_VERSION={version}` の形式で新バージョンを出力 |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | simdjson.h, simdjson.cpp, LICENSE |
| 出力先 | deps/simdjson/ |
| 文字コード | UTF-8 |
| 区切り文字 | N/A |

## 処理フロー

### 処理シーケンス

```
1. 初期化
   └─ BASE_DIR, DEPS_DIR, NODE変数の設定
   └─ utils.shの読み込み
2. 最新バージョン取得
   └─ GitHub APIでsimdjson/simdjsonの最新リリースを取得
   └─ tag_nameから'v'プレフィックスを除去してバージョン番号を取得
3. 現在バージョン取得
   └─ deps/simdjson/simdjson.hからSIMDJSON_VERSIONを抽出
4. バージョン比較
   └─ 直接比較（if文）でバージョンを比較
   └─ 同一の場合はスクリプト終了
5. 一時ワークスペース作成
   └─ mktemp -dで一時ディレクトリを作成
   └─ trapでクリーンアップ関数を設定
6. ソースアーカイブダウンロード
   └─ curlでZIPファイルをダウンロード
7. アーカイブ展開
   └─ unzipで展開
8. シングルヘッダーファイル配置
   └─ singleheader/simdjson.hをdeps/simdjsonへ移動
   └─ singleheader/simdjson.cppをdeps/simdjsonへ移動
   └─ LICENSEをdeps/simdjsonへ移動
9. バージョン情報更新
   └─ finalize_version_update関数でドキュメント更新
   └─ 新バージョンを標準出力
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[環境変数・utils.sh読み込み]
    B --> C[GitHub APIで最新バージョン取得]
    C --> D[現在バージョンをsimdjson.hから取得]
    D --> E{バージョン比較}
    E -->|同一| F[スキップして終了]
    E -->|異なる| G[一時ワークスペース作成]
    G --> H[ソースアーカイブダウンロード]
    H --> I[ZIP展開]
    I --> J[simdjson-VERSION ディレクトリに移動]
    J --> K[simdjson.h配置]
    K --> L[simdjson.cpp配置]
    L --> M[LICENSE配置]
    M --> N[バージョン情報更新]
    N --> O[バッチ終了]
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | ネットワークエラー | GitHub APIへの接続失敗 | ネットワーク接続を確認し、再実行 |
| 1 | API エラー | GitHub APIがエラーレスポンスを返却 | 時間をおいて再実行（レート制限の可能性） |
| 1 | ファイル操作エラー | ディレクトリ作成・ファイル移動失敗 | 権限を確認し、手動でクリーンアップ後に再実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 自動リトライなし |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

### 障害時対応

1. スクリプトは `set -e` によりエラー発生時に即座に終了する
2. 一時ワークスペースは `trap cleanup INT TERM EXIT` により自動クリーンアップされる
3. 失敗した場合は `deps/simdjson/` ディレクトリの状態を確認し、必要に応じてgit checkoutで復元する

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | N/A（ファイルシステム操作） |
| コミットタイミング | N/A |
| ロールバック条件 | スクリプト失敗時はgit checkoutで復元可能 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1回の実行で1ライブラリを更新 |
| 目標処理時間 | 数分以内（ネットワーク速度に依存） |
| メモリ使用量上限 | 特に制限なし |

## 排他制御

同一のdeps/simdjsonディレクトリに対する同時実行は想定されていない。複数プロセスからの同時実行は避けること。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 進捗ログ | 各処理ステップ | "Making temporary workspace...", "Fetching simdjson source archive..." 等 |
| スキップログ | バージョン同一時 | "Skipped because simdjson is on the latest version." |
| 完了ログ | 処理完了時 | "All done!" および gitコマンド例 |
| バージョンログ | 最終行 | "NEW_VERSION={version}" |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 終了コード | 0以外 | CI/CDシステム通知 |
| GitHub APIレート制限 | 残り回数が少ない場合 | 時間をおいて再実行 |

## 備考

- simdjsonはSIMD命令を活用した超高速JSONパーサーであり、従来のパーサーと比較して数倍から数十倍の高速化を実現する
- シングルヘッダー形式（simdjson.h + simdjson.cpp）で配布されるため、展開処理が簡潔である
- 本スクリプトはGITHUB_TOKENによる認証を使用していないため、GitHub APIのレート制限に注意が必要
- 更新後は `git add -A deps/simdjson && git commit -m "deps: update simdjson to {version}"` でコミットする
