# バッチ設計書 44-update-googletest.sh

## 概要

本ドキュメントは、Node.jsプロジェクトの依存関係であるGoogleTest（C++テストフレームワーク）を最新バージョンに更新するシェルスクリプト `update-googletest.sh` の設計書です。

### 本バッチの処理概要

このバッチは、Node.jsで使用されるC++テストフレームワークGoogleTestを、GitHubリポジトリの最新コミットに自動更新します。GoogleTestは「Abseil Live at Head」哲学に従い、タグやリリースをほとんど作成しないため、mainブランチの最新コミットを使用します。

**業務上の目的・背景**：GoogleTestはNode.jsのC++ネイティブコード（V8統合部分、libuvバインディング等）のユニットテストに使用される重要なテストフレームワークです。テストフレームワークの更新により、新しいテスト機能の利用、バグ修正、セキュリティ改善が可能になります。定期的な更新により、テストコードの品質と信頼性を維持します。

**バッチの実行タイミング**：週次（日曜日深夜UTC）の自動実行が推奨されます。最新コミットが2日以内の場合はスキップされる安全機構があります。

**主要な処理内容**：
1. GitHubリポジトリから最新のコミットSHA1を取得
2. 現在のバージョンとの差分をチェック
3. 最新コミットが2日以内かどうかを確認（安定性のため）
4. git worktreeで一時的なチェックアウトを作成
5. LICENSE、include、srcディレクトリをコピー
6. googletest.gypファイルを更新（ソースファイルリストを再生成）
7. `maintaining-dependencies.md`のバージョン情報を更新

**前後の処理との関連**：このバッチは単独で実行可能ですが、更新後はNode.jsのビルドとテスト実行が必要です。特にC++テストの実行を推奨します。

**影響範囲**：`deps/googletest`ディレクトリ全体、`doc/contributing/maintaining/maintaining-dependencies.md`、C++テストの実行

## バッチ種別

依存関係更新

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 週次 |
| 実行時刻 | 00:00 UTC |
| 実行曜日 | 日曜日 |
| 実行日 | 任意 |
| トリガー | cron / GitHub Actions / 手動 |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Git | リポジトリ操作、worktree作成に必要 |
| date | 日付計算に必要（GNU date推奨） |
| sed | ファイル編集に必要 |
| find | ソースファイル検索に必要 |

### 実行可否判定

1. git diff-treeで現在のdeps/googletestとupstreamの差分をチェック
2. 差分がない場合はスクリプトを終了
3. 最新コミットの変更日が2日以内の場合はスキップ（安定性確保のため）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| なし | - | - | - | コマンドライン引数なし |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| GitHub Git | SHA1 | `git ls-remote https://github.com/google/googletest.git HEAD` から最新コミットを取得 |
| deps/googletest | Directory | 現在のGoogleTestソースコード |
| GoogleTest upstream | Git | https://github.com/google/googletest.git から最新ソースを取得 |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| deps/googletest/ | Directory | 更新されたGoogleTestソースコード |
| deps/googletest/googletest.gyp | GYP | 更新されたビルドファイル（ソースリスト含む） |
| 標準出力 | Text | 処理進捗メッセージ、NEW_VERSION環境変数 |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | deps/googletest/配下の全ファイル |
| 出力先 | プロジェクトルート/deps/googletest/ |
| 文字コード | UTF-8 |
| 区切り文字 | N/A |

## 処理フロー

### 処理シーケンス

```
1. 環境変数・パス設定
   └─ BASE_DIR、DEPS_DIRを設定
2. 最新コミットSHA1取得
   └─ git ls-remoteでHEADのSHA1を取得
3. utils.shの読み込み
   └─ 共通ユーティリティ関数をロード
4. リモートリポジトリ追加
   └─ git remote addでgoogletest-upstreamを追加
5. upstreamフェッチ
   └─ 最新コミットをフェッチ
6. リモート削除
   └─ フェッチ後にリモートを削除
7. 差分チェック
   └─ git diff-treeでLICENSE、include、srcの差分を確認
8. 変更日チェック
   └─ 最新コミットが2日以内かを確認
9. 一時ワークツリー作成
   └─ mktemp -dとgit worktree addで作業環境を作成
10. クリーンアップ関数設定
    └─ trapでINT/TERM/EXIT時の後処理を登録
11. ファイルコピー
    └─ LICENSE、include、srcをdeps/googletestにコピー
12. googletest.gyp更新
    └─ sedとfindでソースファイルリストを再生成
13. バージョン情報更新
    └─ finalize_version_update関数でドキュメント更新
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[環境設定]
    B --> C[git ls-remoteで最新コミット取得]
    C --> D[リモート追加・フェッチ]
    D --> E{差分あり?}
    E -->|No| F[スキップして終了]
    E -->|Yes| G{最新コミットが2日以上前?}
    G -->|No| H[スキップ：最新すぎる]
    G -->|Yes| I[一時ワークツリー作成]
    I --> J[LICENSE/include/srcをコピー]
    J --> K[googletest.gypを更新]
    K --> L[バージョン情報更新]
    L --> M[ワークツリー削除]
    M --> N[バッチ終了]
    F --> N
    H --> N
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベース操作を行いません。

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| N/A | N/A | N/A | データベース操作なし |

### テーブル別操作詳細

N/A

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | Git Error | git ls-remote失敗 | ネットワーク接続確認、GitHubアクセス確認 |
| 1 | Fetch Error | git fetch失敗 | ネットワーク接続確認、リポジトリURL確認 |
| 1 | Worktree Error | git worktree作成失敗 | ディスク容量確認、Git設定確認 |
| 1 | Permission Error | ファイル操作権限不足 | 実行ユーザーの権限確認 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

### 障害時対応

1. `set -e`によりエラー発生時は即座にスクリプトが終了
2. `trap cleanup`によりworktreeと一時ディレクトリは自動的にクリーンアップされる
3. git worktree removeで作業ツリーを適切に削除
4. 障害発生時は既存の`deps/googletest`ディレクトリへの影響を最小限に抑制
5. 部分的に更新された場合は、gitで変更を破棄し再実行

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | ファイルシステム操作全体 |
| コミットタイミング | 処理完了時（gitコミットは手動） |
| ロールバック条件 | スクリプト途中でのエラー発生時 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1回の実行で1ライブラリを更新 |
| 目標処理時間 | 5分以内（ネットワーク速度に依存） |
| メモリ使用量上限 | 特になし |

## 排他制御

同一プロジェクトディレクトリに対する複数同時実行は非推奨です。git worktreeの競合が発生する可能性があります。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 比較ログ | 差分チェック時 | "Comparing {version} with current revision" |
| スキップログ | スキップ時 | "Skipped because googletest is on the latest version." または "Skipped because the latest version is too recent." |
| 進捗ログ | 各処理ステップ | "Creating temporary work tree"、"Copying LICENSE, include and src to deps/googletest"、"Updating googletest.gyp" |
| 完了ログ | 処理完了時 | "All done!"、gitコマンドの案内 |
| 環境変数出力 | 最終行 | "NEW_VERSION={sha1}" |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 処理時間 | 10分 | CI/CDログ |
| エラー発生 | 1件以上 | GitHub Actions通知 |

## 備考

- GoogleTestは「Abseil Live at Head」哲学に従い、タグやリリースをほとんど作成しません
- そのため、mainブランチの最新コミットを直接使用します
- 安定性確保のため、最新コミットが2日以内の場合は更新をスキップします（週末のプッシュを避けるため）
- 更新後は必ずC++テストを実行してください
- googletest.gypファイルはソースファイルのリストを含むため、自動的に再生成されます
- gtest_main.cc、gtest-all.cc、gtest_prod.hはビルドから除外されます
