# 機能設計書 11-buffer

## 概要

本ドキュメントは、Node.jsのBufferモジュール（バイナリデータ操作用Buffer API）の機能設計を記述したものである。

### 本機能の処理概要

Buffer機能は、Node.jsにおいてバイナリデータを効率的に操作するための基盤APIを提供する。JavaScriptのStringは UTF-16エンコーディングで文字列を扱うが、ファイルシステムやネットワーク通信、暗号処理などでは生のバイナリデータを扱う必要があり、Bufferがその役割を担う。

**業務上の目的・背景**：ネットワーク通信、ファイル入出力、暗号処理など、低レベルのバイナリデータ操作が必要な多くの場面でBufferが利用される。特にストリーム処理やプロトコル実装において不可欠な機能である。

**機能の利用シーン**：
- ファイルの読み書き（バイナリファイルの処理）
- ネットワークパケットの構築・解析
- 暗号化・ハッシュ計算の入出力データ
- Base64/Hexなどのエンコーディング変換
- 画像・音声などのメディアデータ処理

**主要な処理内容**：
1. Buffer生成（alloc、allocUnsafe、from、of、copyBytesFrom）
2. データの読み書き（write、read系メソッド）
3. エンコーディング変換（toString、各種エンコーディング対応）
4. バッファ操作（copy、slice、subarray、fill、concat）
5. 検索・比較（indexOf、lastIndexOf、includes、compare、equals）
6. バイトオーダー変換（swap16、swap32、swap64）
7. 文字列⇔バイナリ変換（btoa、atob）
8. エンコーディング検証（isUtf8、isAscii）

**関連システム・外部連携**：
- fs（ファイルシステム）：ファイル読み書きでBufferを使用
- stream：ストリームデータの内部表現としてBufferを使用
- crypto：暗号処理の入出力にBufferを使用
- net/http：ネットワーク通信でBufferを使用
- zlib：圧縮・解凍処理でBufferを使用

**権限による制御**：特になし。Bufferはユーザーランドで自由に使用可能。

## 関連画面

本機能はCLI/APIレベルの機能であり、直接関連する画面はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | - |

## 機能種別

データ操作 / バイナリ処理 / エンコーディング変換

## 入力仕様

### 入力パラメータ

#### Buffer.alloc(size[, fill[, encoding]])

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| size | number | Yes | 確保するバッファサイズ（バイト数） | 0 <= size <= kMaxLength |
| fill | string/Buffer/Uint8Array/integer | No | 初期化に使用する値 | - |
| encoding | string | No | fillが文字列の場合のエンコーディング | 有効なエンコーディング名 |

#### Buffer.from(value[, encodingOrOffset[, length]])

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| value | string/Array/Buffer/ArrayBuffer/Object | Yes | 変換元データ | 型に応じた検証 |
| encodingOrOffset | string/number | No | エンコーディングまたはオフセット | - |
| length | number | No | 長さ（ArrayBufferの場合） | - |

### 入力データソース

- アプリケーションコード（直接呼び出し）
- ファイルシステム（fs.readFileなどの戻り値）
- ネットワーク（ソケットからの受信データ）
- 他のBuffer/TypedArray

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Buffer | Buffer | Uint8Arrayを継承したバイナリデータオブジェクト |

### 出力先

- アプリケーションコード（戻り値として）
- ファイルシステム（fs.writeFileなどの引数として）
- ネットワーク（ソケットへの送信データとして）

## 処理フロー

### 処理シーケンス

```
1. Buffer生成リクエスト
   └─ サイズ/データの検証
2. メモリ確保
   └─ poolからの確保（小サイズ）または直接確保（大サイズ）
3. データ初期化
   └─ fill指定時は指定値で初期化、なければゼロ埋め（allocの場合）
4. Bufferオブジェクト返却
   └─ FastBuffer（Uint8Array派生）インスタンスを返却
```

### フローチャート

```mermaid
flowchart TD
    A[Buffer生成リクエスト] --> B{生成方法}
    B -->|alloc| C[サイズ検証]
    B -->|from| D[入力データ型判定]
    B -->|allocUnsafe| E[サイズ検証]

    C --> F{fill指定?}
    F -->|Yes| G[createUnsafeBuffer + fill処理]
    F -->|No| H[new FastBuffer]

    D -->|string| I[fromString]
    D -->|ArrayBuffer| J[fromArrayBuffer]
    D -->|Array-like| K[fromArrayLike]
    D -->|Buffer| L[copy処理]

    E --> M{poolSize/2未満?}
    M -->|Yes| N[poolから確保]
    M -->|No| O[createUnsafeBuffer]

    G --> P[Buffer返却]
    H --> P
    I --> P
    J --> P
    K --> P
    L --> P
    N --> P
    O --> P
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 最大サイズ制限 | BufferのサイズはkMaxLength（約2GB）を超えられない | alloc/allocUnsafe時 |
| BR-002 | Pool利用 | poolSize/2未満のBufferはプールから確保 | allocUnsafe時 |
| BR-003 | 非推奨警告 | Buffer()コンストラクタ直接呼び出しは非推奨警告（DEP0005） | Buffer()使用時 |
| BR-004 | エンコーディング対応 | utf8/ucs2/utf16le/latin1/ascii/base64/base64url/hex対応 | toString/from時 |

### 計算ロジック

#### base64バイト長計算
```javascript
// Base64エンコードされた文字列からバイト長を計算
function base64ByteLength(str, bytes) {
  // パディング処理
  if (StringPrototypeCharCodeAt(str, bytes - 1) === 0x3D) bytes--;
  if (bytes > 1 && StringPrototypeCharCodeAt(str, bytes - 1) === 0x3D) bytes--;
  // Base64比率: 3/4
  return (bytes * 3) >>> 2;
}
```

## データベース操作仕様

本機能はデータベースを直接操作しない。

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | - |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_BUFFER_OUT_OF_BOUNDS | RangeError | オフセットがバッファ範囲外 | オフセット値を修正 |
| ERR_INVALID_ARG_TYPE | TypeError | 引数の型が不正 | 正しい型の引数を渡す |
| ERR_INVALID_ARG_VALUE | TypeError | 引数の値が不正 | 正しい値の引数を渡す |
| ERR_INVALID_BUFFER_SIZE | RangeError | バッファサイズが不正（swap時） | サイズを調整 |
| ERR_OUT_OF_RANGE | RangeError | 値が許容範囲外 | 範囲内の値を使用 |
| ERR_UNKNOWN_ENCODING | TypeError | 不明なエンコーディング | 有効なエンコーディングを指定 |

### リトライ仕様

本機能にリトライ仕様はない（同期処理）。

## トランザクション仕様

本機能にトランザクション仕様はない（インメモリ操作）。

## パフォーマンス要件

- 小サイズBuffer（poolSize/2 = 4KB未満）はプールから高速に確保
- 大サイズBufferは直接メモリ確保
- swap16/32/64は128/192バイト未満ではJavaScript実装、それ以上はネイティブ実装を使用

## セキュリティ考慮事項

- `Buffer.allocUnsafe()`は未初期化メモリを返すため、機密データが残留する可能性がある
- `Buffer()`コンストラクタは非推奨（DEP0005）、セキュリティ上`Buffer.alloc()`または`Buffer.from()`を使用すべき
- `--zero-fill-buffers`フラグで全Bufferをゼロ初期化可能

## 備考

- BufferはUint8Arrayを継承している
- ES6のTypedArrayと互換性がある
- Node.js v6以降でBuffer()コンストラクタは非推奨

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Bufferの基本構造とFastBufferの関係を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | buffer.js | `lib/buffer.js` | Buffer定義とFastBufferの継承関係 |
| 1-2 | internal/buffer.js | `lib/internal/buffer.js` | FastBufferの実装詳細 |

**読解のコツ**: BufferはFastBuffer（Uint8Arrayのサブクラス）のprototypeを共有している（153-155行目）。

#### Step 2: エントリーポイントを理解する

Buffer生成メソッドの理解。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | buffer.js | `lib/buffer.js` | Buffer.alloc(), Buffer.from(), Buffer.allocUnsafe() |

**主要処理フロー**:
1. **302-312行目**: Buffer()コンストラクタ（非推奨）
2. **333-365行目**: Buffer.from()の実装、入力型に応じた分岐
3. **431-438行目**: Buffer.alloc()の実装
4. **445-448行目**: Buffer.allocUnsafe()の実装

#### Step 3: プール管理を理解する

メモリプール管理の仕組み。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | buffer.js | `lib/buffer.js` | createPool(), allocate()関数 |

**主要処理フロー**:
- **172行目**: poolSize = 8KB
- **175-181行目**: createPool() - プール作成
- **462-475行目**: allocate() - プールからの確保またはダイレクト確保

#### Step 4: エンコーディング処理を理解する

各種エンコーディングの変換処理。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | buffer.js | `lib/buffer.js` | encodingOps定義、toString()、write() |

**主要処理フロー**:
- **660-748行目**: encodingOps定義（各エンコーディングの処理関数群）
- **864-894行目**: toString()実装
- **1139-1177行目**: write()実装

### プログラム呼び出し階層図

```
Buffer (lib/buffer.js)
    │
    ├─ Buffer.from()
    │      ├─ fromString() ─── fromStringFast() / createFromString()
    │      ├─ fromArrayBuffer()
    │      └─ fromArrayLike()
    │
    ├─ Buffer.alloc()
    │      └─ createUnsafeBuffer() ─── _fill()
    │
    ├─ Buffer.allocUnsafe()
    │      └─ allocate()
    │             ├─ createPool() (プールサイズ未満)
    │             └─ createUnsafeBuffer() (プールサイズ以上)
    │
    ├─ Buffer.prototype.toString()
    │      └─ encodingOps[encoding].slice()
    │
    └─ Buffer.prototype.write()
           └─ encodingOps[encoding].write()

internalBinding('buffer') (C++実装)
    ├─ byteLengthUtf8
    ├─ compare / compareOffset
    ├─ copy
    ├─ fill
    ├─ indexOfBuffer / indexOfNumber / indexOfString
    ├─ swap16 / swap32 / swap64
    └─ 各エンコーディングのslice/write関数
```

### データフロー図

```
[入力]                    [処理]                         [出力]

文字列 ───────────▶ fromString()
                         │
                         ▼
ArrayBuffer ────────▶ fromArrayBuffer()  ───▶ FastBuffer
                         │                      (Uint8Array)
                         ▼
Array-like ─────────▶ fromArrayLike()
                         │
                         ▼
サイズ指定 ─────────▶ allocate() ─────────▶ プール/直接確保


Buffer ─────────────▶ toString()
                         │
                         ▼
                    encodingOps.slice() ────▶ 文字列
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| buffer.js | `lib/buffer.js` | ソース | Bufferモジュールのメインエントリポイント |
| internal/buffer.js | `lib/internal/buffer.js` | ソース | FastBuffer定義、内部実装 |
| internal/util.js | `lib/internal/util.js` | ソース | normalizeEncoding等のユーティリティ |
| internal/validators.js | `lib/internal/validators.js` | ソース | 入力検証関数群 |
| internal/errors.js | `lib/internal/errors.js` | ソース | エラーコード定義 |
| src/node_buffer.cc | `src/node_buffer.cc` | C++ソース | ネイティブ実装 |
