# 機能設計書 16-os

## 概要

本ドキュメントは、Node.jsのosモジュール（OS関連情報取得API）の機能設計を記述したものである。

### 本機能の処理概要

os機能は、オペレーティングシステムに関する情報を取得するAPIを提供する。CPU情報、メモリ使用量、ネットワークインターフェース、ホスト名、ユーザー情報など、システム管理や監視に必要な情報を取得できる。

**業務上の目的・背景**：アプリケーションの動作環境に応じた処理の分岐、システム監視、リソース管理、デバッグ情報の収集など、OSに依存した情報が必要な場面で使用される。

**機能の利用シーン**：
- システム監視・モニタリング
- 負荷分散の判断材料取得
- クロスプラットフォーム対応の分岐処理
- ログ出力時のシステム情報追加
- 一時ファイルパスの取得
- プロセス優先度の管理

**主要な処理内容**：
1. CPU情報取得（cpus、arch、availableParallelism）
2. メモリ情報取得（freemem、totalmem）
3. ネットワーク情報取得（networkInterfaces、hostname）
4. システム情報取得（platform、type、release、version、machine）
5. ユーザー情報取得（userInfo、homedir）
6. パス取得（tmpdir、devNull、EOL）
7. プロセス管理（getPriority、setPriority）
8. その他（uptime、loadavg、endianness）

**関連システム・外部連携**：
- process: プラットフォーム情報共有
- child_process: プロセス優先度継承

**権限による制御**：特になし（ただしsetPriorityは権限が必要な場合あり）。

## 関連画面

本機能はCLI/APIレベルの機能であり、直接関連する画面はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | - |

## 機能種別

情報取得 / システム管理

## 入力仕様

### 入力パラメータ

#### userInfo([options])

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| options.encoding | string | No | 文字列エンコーディング | 'buffer'で Buffer返却 |

#### getPriority([pid])

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pid | number | No | プロセスID | デフォルト0（現在のプロセス） |

#### setPriority([pid,] priority)

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pid | number | No | プロセスID | デフォルト0 |
| priority | number | Yes | 優先度 | -20〜19 |

### 入力データソース

- オペレーティングシステム（カーネル）
- 環境変数

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| cpus | Array | CPU情報配列（model, speed, times） |
| freemem | number | 空きメモリ（バイト） |
| totalmem | number | 総メモリ（バイト） |
| hostname | string | ホスト名 |
| networkInterfaces | Object | ネットワークインターフェース情報 |
| platform | string | プラットフォーム名 |
| type | string | OS種別 |
| release | string | OSリリースバージョン |
| userInfo | Object | ユーザー情報 |

### 出力先

- アプリケーションコード（戻り値）

## 処理フロー

### 処理シーケンス

```
1. 関数呼び出し
   └─ パラメータ検証（必要な場合）
2. ネイティブバインディング呼び出し
   └─ internalBinding('os')経由
3. 結果整形
   └─ JavaScriptオブジェクトに変換
4. 返却
   └─ エラー時は例外スロー
```

### フローチャート

```mermaid
flowchart TD
    A[関数呼び出し] --> B{パラメータあり?}
    B -->|Yes| C[パラメータ検証]
    B -->|No| D[ネイティブ呼び出し]
    C --> D
    D --> E{エラー?}
    E -->|Yes| F[ERR_SYSTEM_ERRORスロー]
    E -->|No| G[結果整形]
    G --> H[返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 優先度範囲 | setPriorityの優先度は-20〜19 | Unix系 |
| BR-002 | プラットフォーム依存 | loadavgはWindowsで常に[0,0,0] | Windows |
| BR-003 | tmpdir優先順位 | Windows: TEMP → TMP → SystemRoot\temp | Windows |
| BR-004 | EOL | Windowsは\r\n、それ以外は\n | プラットフォーム依存 |

### 計算ロジック

#### CIDR計算（networkInterfaces）
```javascript
function getCIDR(address, netmask, family) {
  // netmaskからCIDR表記を計算
  // 例: 255.255.255.0 → /24
}
```

## データベース操作仕様

本機能はデータベースを直接操作しない。

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | - |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_SYSTEM_ERROR | SystemError | OS API呼び出し失敗 | エラー内容を確認 |
| ERR_INVALID_ARG_TYPE | TypeError | 引数の型が不正 | 正しい型を使用 |

### リトライ仕様

本機能にリトライ仕様はない。

## トランザクション仕様

本機能にトランザクション仕様はない。

## パフォーマンス要件

- 多くの関数はネイティブAPIの薄いラッパー
- cpus()は全コア情報を取得するため、大量コア環境では注意
- networkInterfaces()はインターフェース数に比例

## セキュリティ考慮事項

- userInfo()でユーザー名やホームディレクトリが取得可能
- setPriority()は権限によっては失敗する可能性

## 備考

- 多くの関数はSymbolToPrimitiveを実装し、テンプレートリテラルで直接使用可能
- constants.signalsでシグナル定数を取得可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

エクスポートされる関数と定数の構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | os.js | `lib/os.js` | module.exports（311-332行目） |

**読解のコツ**: 多くの関数は直接ネイティブバインディングを呼び出すか、薄いラッパーである。

#### Step 2: ネイティブバインディングを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | os.js | `lib/os.js` | internalBinding('os')（46-61行目） |

**主要処理フロー**:
- **46-61行目**: ネイティブ関数インポート（getCPUs, getFreeMem等）

#### Step 3: ヘルパー関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | os.js | `lib/os.js` | getCheckedFunction（63-72行目） |

**主要処理フロー**:
- **63-72行目**: エラーチェック付きラッパー生成

#### Step 4: 主要関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | os.js | `lib/os.js` | cpus()、networkInterfaces()、userInfo()等 |

**主要処理フロー**:
- **141-160行目**: cpus() - CPU情報取得
- **217-245行目**: networkInterfaces() - ネットワーク情報取得
- **298-309行目**: userInfo() - ユーザー情報取得

### プログラム呼び出し階層図

```
os (lib/os.js)
    │
    ├─ CPU関連
    │      ├─ cpus() ─── getCPUs()
    │      ├─ arch() ─── process.arch
    │      └─ availableParallelism() ─── getAvailableParallelism()
    │
    ├─ メモリ関連
    │      ├─ freemem() ─── getFreeMem()
    │      └─ totalmem() ─── getTotalMem()
    │
    ├─ ネットワーク関連
    │      ├─ hostname() ─── getHostname() [checked]
    │      └─ networkInterfaces() ─── getInterfaceAddresses() [checked]
    │                                       └─ getCIDR()
    │
    ├─ システム関連
    │      ├─ platform() ─── process.platform
    │      ├─ type() ─── getOSType()
    │      ├─ release() ─── getOSRelease()
    │      ├─ version() ─── getOSVersion()
    │      ├─ machine() ─── getMachine()
    │      └─ uptime() ─── getUptime() [checked]
    │
    ├─ ユーザー関連
    │      ├─ userInfo() ─── getUserInfo()
    │      └─ homedir() ─── getHomeDirectory() [checked]
    │
    ├─ パス関連
    │      ├─ tmpdir() ─── getTempDir() / process.env
    │      ├─ EOL ─── '\r\n' or '\n'
    │      └─ devNull ─── '\\\\.\\nul' or '/dev/null'
    │
    └─ プロセス優先度
           ├─ getPriority() ─── _getPriority()
           └─ setPriority() ─── _setPriority()

internalBinding('os')
    ├─ getCPUs / getFreeMem / getTotalMem
    ├─ getHostname / getInterfaceAddresses
    ├─ getOSInformation (type/version/release/machine)
    ├─ getUptime / getLoadAvg
    ├─ getUserInfo / getHomeDirectory
    ├─ getPriority / setPriority
    └─ getAvailableParallelism / isBigEndian
```

### データフロー図

```
[OS カーネル]              [Node.js]                  [アプリケーション]

/proc/cpuinfo ─────▶ getCPUs() ─────▶ cpus() ─────▶ CPU情報配列
/proc/meminfo ─────▶ getFreeMem() ──▶ freemem() ──▶ 空きメモリ量
/etc/hostname ─────▶ getHostname() ─▶ hostname() ─▶ ホスト名
ifconfig ──────────▶ getInterfaces() ▶ networkInterfaces() ▶ NIC情報

環境変数 ──────────▶ tmpdir() ─────────────────────▶ 一時ディレクトリ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| os.js | `lib/os.js` | ソース | メインエントリポイント |
| internal/util.js | `lib/internal/util.js` | ソース | getCIDR |
| internal/validators.js | `lib/internal/validators.js` | ソース | validateInt32 |
| internal/errors.js | `lib/internal/errors.js` | ソース | ERR_SYSTEM_ERROR |
| src/node_os.cc | `src/node_os.cc` | C++ソース | ネイティブ実装 |
