# 機能設計書 17-process

## 概要

本ドキュメントは、Node.jsのprocessモジュール（プロセス情報・制御API）の機能設計を記述したものである。

### 本機能の処理概要

process機能は、現在実行中のNode.jsプロセスに関する情報の取得と制御を行うグローバルオブジェクトである。プロセスの終了、シグナル処理、環境変数アクセス、リソース使用状況の取得など、プロセスレベルの操作を提供する。

**業務上の目的・背景**：アプリケーションの実行環境情報取得、グレースフルシャットダウン、環境変数による設定管理、リソース監視など、プロセス制御に関する基本機能が必要である。

**機能の利用シーン**：
- アプリケーションの正常終了・異常終了処理
- シグナルハンドリング（SIGINT、SIGTERM等）
- 環境変数による設定管理
- CPU・メモリ使用状況の監視
- コマンドライン引数の取得
- 高精度タイマー（hrtime）
- プロセス間通信（IPC）

**主要な処理内容**：
1. プロセス制御（exit、kill、abort）
2. 環境変数アクセス（env、loadEnvFile）
3. リソース情報（cpuUsage、memoryUsage、resourceUsage）
4. 高精度時間計測（hrtime、hrtime.bigint）
5. シグナル処理（on('SIGINT')等）
6. イベント発行（exit、beforeExit、uncaughtException等）
7. コマンドライン（argv、execArgv、execPath）
8. 標準入出力（stdin、stdout、stderr）

**関連システム・外部連携**：
- os: システム情報共有
- child_process: 子プロセスの親プロセス
- cluster: クラスターのプライマリ/ワーカー
- worker_threads: メインスレッド/ワーカースレッド

**権限による制御**：特になし（ただしkillは対象プロセスへの権限が必要）。

## 関連画面

本機能はCLI/APIレベルの機能であり、直接関連する画面はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | - |

## 機能種別

プロセス制御 / システム情報取得 / イベント処理

## 入力仕様

### 入力パラメータ

#### exit([code])

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| code | number | No | 終了コード | デフォルト0 |

#### kill(pid[, signal])

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pid | number | Yes | プロセスID | 整数 |
| signal | string/number | No | シグナル | デフォルト'SIGTERM' |

#### cpuUsage([previousValue])

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| previousValue | Object | No | 前回の値（差分計算用） | {user, system} |

#### loadEnvFile([path])

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| path | string/URL | No | .envファイルパス | デフォルト.env |

### 入力データソース

- オペレーティングシステム（プロセス情報）
- 環境変数
- コマンドライン引数
- .envファイル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| cpuUsage | {user, system} | CPU使用時間（マイクロ秒） |
| memoryUsage | Object | メモリ使用状況 |
| resourceUsage | Object | リソース使用状況 |
| hrtime | [seconds, nanoseconds] | 高精度時間 |
| hrtime.bigint | bigint | 高精度時間（ナノ秒） |

### 出力先

- アプリケーションコード（戻り値、イベント）
- 標準出力/エラー出力
- プロセス終了コード

## 処理フロー

### 処理シーケンス

```
1. Node.js起動
   └─ processオブジェクト初期化
   └─ 環境変数・引数設定
2. イベントループ実行
   └─ イベントリスナー処理
   └─ シグナル受信処理
3. 終了処理
   └─ beforeExitイベント発火
   └─ exitイベント発火
   └─ プロセス終了
```

### フローチャート

```mermaid
flowchart TD
    A[Node.js起動] --> B[process初期化]
    B --> C[イベントループ開始]
    C --> D{イベント待機}
    D -->|シグナル| E[シグナルハンドラ実行]
    D -->|タイマー/IO| F[コールバック実行]
    D -->|終了要求| G[beforeExit発火]
    E --> D
    F --> D
    G --> H{新規タスク?}
    H -->|Yes| D
    H -->|No| I[exitイベント発火]
    I --> J[process.exit]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 終了コード | exit(code)でプロセス終了コードを設定 | exit呼び出し時 |
| BR-002 | シグナル定数 | kill()のsignalは名前または数値 | kill呼び出し時 |
| BR-003 | exitイベント | exitイベントハンドラは同期のみ | process.on('exit') |
| BR-004 | uncaughtException | 未捕捉例外時に発火、ハンドラなしで終了 | 例外発生時 |
| BR-005 | メインスレッド制限 | execveはメインスレッドのみ | process.execve() |

### 計算ロジック

#### hrtime差分計算
```javascript
function hrtime(time) {
  binding.hrtime();
  if (time !== undefined) {
    const sec = (hrValues[0] * 0x100000000 + hrValues[1]) - time[0];
    const nsec = hrValues[2] - time[1];
    const needsBorrow = nsec < 0;
    return [needsBorrow ? sec - 1 : sec, needsBorrow ? nsec + 1e9 : nsec];
  }
  return [hrValues[0] * 0x100000000 + hrValues[1], hrValues[2]];
}
```

## データベース操作仕様

本機能はデータベースを直接操作しない。

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | - |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_INVALID_ARG_TYPE | TypeError | 引数の型が不正 | 正しい型を使用 |
| ERR_INVALID_ARG_VALUE | TypeError | 引数の値が不正 | 正しい値を使用 |
| ERR_OUT_OF_RANGE | RangeError | 範囲外の値 | 範囲内の値を使用 |
| ERR_UNKNOWN_SIGNAL | TypeError | 不明なシグナル名 | 有効なシグナル名を使用 |
| ERR_WORKER_UNSUPPORTED_OPERATION | Error | ワーカーで非対応の操作 | メインスレッドで実行 |
| ErrnoException | Error | システムコール失敗 | エラー内容を確認 |

### リトライ仕様

本機能にリトライ仕様はない。

## トランザクション仕様

本機能にトランザクション仕様はない。

## パフォーマンス要件

- hrtime()はナノ秒精度
- cpuUsage()はマイクロ秒精度
- memoryUsage()は即時取得

## セキュリティ考慮事項

- process.envは環境変数への読み書きアクセスを提供
- kill()は対象プロセスへの権限が必要
- execve()は実験的機能、メインスレッド限定

## 備考

- processはグローバルオブジェクトであり、require不要
- EventEmitterを継承
- allowedNodeEnvironmentFlagsでNODE_OPTIONS許可フラグを確認可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

processオブジェクトの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | per_thread.js | `lib/internal/process/per_thread.js` | wrapProcessMethods |

**読解のコツ**: processはブートストラップ時に構築され、様々なモジュールからプロパティが追加される。

#### Step 2: 主要メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | per_thread.js | `lib/internal/process/per_thread.js` | exit, kill, cpuUsage等 |

**主要処理フロー**:
- **127-161行目**: cpuUsage() - CPU使用時間取得
- **216-225行目**: memoryUsage() - メモリ使用状況取得
- **229-252行目**: exit() - プロセス終了
- **254-280行目**: kill() - シグナル送信
- **329-350行目**: resourceUsage() - リソース使用状況

#### Step 3: hrtimeを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | per_thread.js | `lib/internal/process/per_thread.js` | hrtime, hrtimeBigInt（78-102行目） |

**主要処理フロー**:
- **78-97行目**: hrtime() - 高精度時間取得
- **99-102行目**: hrtimeBigInt() - BigInt版

#### Step 4: allowedNodeEnvironmentFlagsを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | per_thread.js | `lib/internal/process/per_thread.js` | buildAllowedFlags（385-511行目） |

### プログラム呼び出し階層図

```
process (グローバル)
    │
    ├─ プロセス制御
    │      ├─ exit(code) ─── reallyExit()
    │      ├─ kill(pid, signal) ─── _kill()
    │      ├─ abort() ─── ネイティブ
    │      └─ execve(path, args, env) ─── _execve() [実験的]
    │
    ├─ リソース情報
    │      ├─ cpuUsage() ─── _cpuUsage()
    │      ├─ threadCpuUsage() ─── _threadCpuUsage()
    │      ├─ memoryUsage() ─── _memoryUsage()
    │      │      └─ memoryUsage.rss()
    │      └─ resourceUsage() ─── _resourceUsage()
    │
    ├─ 時間計測
    │      ├─ hrtime(time) ─── binding.hrtime()
    │      └─ hrtime.bigint() ─── binding.hrtimeBigInt()
    │
    ├─ 環境・設定
    │      ├─ env ─── 環境変数オブジェクト
    │      ├─ loadEnvFile(path) ─── _loadEnvFile()
    │      └─ allowedNodeEnvironmentFlags ─── NodeEnvironmentFlagsSet
    │
    ├─ プロセス情報
    │      ├─ pid, ppid
    │      ├─ argv, execArgv, execPath
    │      ├─ arch, platform, version
    │      └─ title
    │
    ├─ 標準入出力
    │      ├─ stdin
    │      ├─ stdout
    │      └─ stderr
    │
    └─ イベント
           ├─ 'exit'
           ├─ 'beforeExit'
           ├─ 'uncaughtException'
           ├─ 'unhandledRejection'
           └─ シグナルイベント (SIGINT, SIGTERM等)
```

### データフロー図

```
[OS/カーネル]               [process]                [アプリケーション]

環境変数 ─────────────▶ process.env ─────────────▶ 設定値
コマンドライン ─────────▶ process.argv ───────────▶ 引数
CPU使用量 ────────────▶ cpuUsage() ──────────────▶ {user, system}
メモリ情報 ───────────▶ memoryUsage() ───────────▶ {rss, heapTotal, ...}
高精度時計 ───────────▶ hrtime() ────────────────▶ [seconds, nanoseconds]

シグナル受信 ──────────▶ 'SIGINT'イベント ────────▶ ハンドラ実行
                                │
終了要求 ─────────────▶ exit() ──────────────────▶ 'exit'イベント
                                │
                                ▼
                         プロセス終了
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| per_thread.js | `lib/internal/process/per_thread.js` | ソース | スレッド毎のprocess関数 |
| main_thread_only.js | `lib/internal/process/main_thread_only.js` | ソース | メインスレッド専用関数 |
| bootstrap/node.js | `lib/internal/bootstrap/node.js` | ソース | processの初期化 |
| internal/validators.js | `lib/internal/validators.js` | ソース | パラメータ検証 |
| internal/errors.js | `lib/internal/errors.js` | ソース | エラーコード定義 |
| src/node_process_methods.cc | `src/node_process_methods.cc` | C++ソース | ネイティブ実装 |
