# 機能設計書 21-util

## 概要

本ドキュメントは、Node.jsのutilモジュールの機能設計について記述する。utilモジュールは、開発者向けのユーティリティ関数群を提供し、デバッグ、型変換、非同期処理のラッピング、オブジェクト検査など多岐にわたる機能を提供する。

### 本機能の処理概要

**業務上の目的・背景**：utilモジュールは、Node.jsアプリケーション開発において頻繁に必要となる汎用的な処理を一元的に提供することで、開発効率の向上とコードの一貫性を保つことを目的としている。コールバック形式の関数をPromise形式に変換するpromisify、オブジェクトの詳細な検査を行うinspect、非推奨機能の警告を出すdeprecateなど、開発・デバッグに不可欠な機能を提供する。

**機能の利用シーン**：
- コールバックベースのAPIをPromiseベースに変換する際（promisify）
- オブジェクトの内容をデバッグ出力する際（inspect、format）
- プロトタイプ継承を設定する際（inherits）
- システムエラーの詳細情報を取得する際（getSystemErrorName、getSystemErrorMessage）
- 環境変数ファイル(.env)をパースする際（parseEnv）
- テキストにANSIスタイルを適用する際（styleText）

**主要な処理内容**：
1. `promisify` - コールバック形式の関数をPromise形式にラップ
2. `callbackify` - Promise形式の関数をコールバック形式に変換
3. `inspect` - オブジェクトを可読性のある文字列に変換
4. `format` / `formatWithOptions` - printf形式の文字列フォーマット
5. `deprecate` - 非推奨警告を付与した関数を生成
6. `inherits` - プロトタイプ継承の設定
7. `isDeepStrictEqual` - 厳密な深い比較
8. `getSystemErrorName` / `getSystemErrorMessage` - システムエラー情報取得
9. `parseEnv` - .envファイル形式の文字列をパース
10. `styleText` - ANSIエスケープシーケンスによるテキストスタイリング
11. `getCallSites` - コールスタック情報の取得
12. `TextEncoder` / `TextDecoder` - テキストエンコーディング
13. `MIMEType` / `MIMEParams` - MIMEタイプの解析

**関連システム・外部連携**：
- V8エンジンのネイティブバインディング（internalBinding('util')）
- ソースマップ機能との連携（getCallSitesのsourceMapオプション）
- internal/util/inspect - オブジェクト検査の内部実装

**権限による制御**：特になし。すべての機能は任意のユーザーコードから利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本モジュールはAPI機能であり、関連する画面はありません |

## 機能種別

ユーティリティ関数 / 型変換 / デバッグ支援

## 入力仕様

### 入力パラメータ

#### promisify

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| original | Function | Yes | コールバック形式の関数 | validateFunction |

#### callbackify

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| original | Function | Yes | Promise返却関数 | validateFunction |

#### inspect

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| object | any | Yes | 検査対象オブジェクト | なし |
| options | Object | No | 表示オプション | validateObject |

#### styleText

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| format | string \| string[] | Yes | スタイル指定 | validateOneOf |
| text | string | Yes | 対象テキスト | validateString |
| options | Object | No | オプション | validateObject |

#### getCallSites

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| frameCount | number | No | 取得するフレーム数（デフォルト10） | validateInteger(1-200) |
| options | Object | No | sourceMapオプション等 | validateObject |

#### parseEnv

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| content | string | Yes | .env形式の文字列 | validateString |

### 入力データソース

プログラムコードからの直接呼び出し。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| promisify戻り値 | Function | Promise返却関数 |
| callbackify戻り値 | Function | コールバック形式関数 |
| inspect戻り値 | string | オブジェクトの文字列表現 |
| styleText戻り値 | string | スタイル適用済みテキスト |
| getCallSites戻り値 | CallSite[] | コールサイト情報配列 |
| parseEnv戻り値 | Record<string, string> | キーと値のオブジェクト |

### 出力先

呼び出し元への戻り値として返却。

## 処理フロー

### 処理シーケンス

```
1. promisify処理
   └─ コールバック関数をPromiseでラップし、新しい関数を生成
2. callbackify処理
   └─ Promise関数を実行し、結果をprocess.nextTickでコールバックに渡す
3. inspect処理
   └─ internal/util/inspectモジュールに処理を委譲
4. styleText処理
   └─ ANSIエスケープコードを生成してテキストを装飾
5. getCallSites処理
   └─ V8バインディングからスタック情報を取得し、必要に応じてソースマップを適用
```

### フローチャート

```mermaid
flowchart TD
    A[util関数呼び出し] --> B{関数タイプ}
    B -->|promisify| C[コールバック関数をラップ]
    B -->|callbackify| D[Promise関数をラップ]
    B -->|inspect| E[オブジェクト検査]
    B -->|styleText| F[ANSIスタイル適用]
    B -->|getCallSites| G[スタック情報取得]
    C --> H[Promise返却関数を生成]
    D --> I[コールバック形式関数を生成]
    E --> J[文字列表現を返却]
    F --> K[スタイル適用テキストを返却]
    G --> L[CallSite配列を返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | promisifyカスタムシンボル | util.promisify.customシンボルでカスタム実装を指定可能 | カスタム実装がある場合 |
| BR-002 | コールバック規約 | コールバックは最後の引数として渡される(err, result)形式 | promisify使用時 |
| BR-003 | 非推奨警告 | deprecateでラップした関数は初回呼び出し時のみ警告 | deprecate使用時 |
| BR-004 | スタイル色検証 | styleTextはストリームがTTYかどうかで色出力を判断 | validateStream=true時 |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

本モジュールはデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_INVALID_ARG_TYPE | TypeError | 引数の型が不正 | 正しい型の引数を渡す |
| ERR_OUT_OF_RANGE | RangeError | getSystemErrorNameに正の数を渡した | 負の整数を渡す |
| ERR_FALSY_VALUE_REJECTION | Error | callbackifyでfalsyな値でrejectされた | エラーオブジェクトでrejectする |

### リトライ仕様

リトライ処理は行わない。

## トランザクション仕様

トランザクション処理は行わない。

## パフォーマンス要件

特に定義なし。軽量なユーティリティ関数として設計されている。

## セキュリティ考慮事項

- inspectはオブジェクトの内部構造を露出するため、機密情報を含むオブジェクトの出力に注意が必要
- parseEnvは信頼できない入力に対して使用する場合、インジェクション攻撃に注意

## 備考

- `util._extend`は非推奨（DEP0060）、代わりにObject.assign()を使用
- `util.isArray`は非推奨（DEP0044）、代わりにArray.isArray()を使用
- TextEncoder/TextDecoder、MIMEType/MIMEParamsは遅延ロードされる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

utilモジュールはシンプルな関数群で構成されており、複雑なデータ構造は少ない。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | util.js | `lib/util.js` | エクスポートされる関数一覧（470-515行目） |

**読解のコツ**: module.exportsで公開されている関数がパブリックAPIとなる。defineLazyPropertiesで遅延ロードされる機能も確認する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | util.js | `lib/util.js` | 主要関数の実装（24-546行目） |

**主要処理フロー**:
1. **24-43行目**: primordialsからの組み込みオブジェクト取得
2. **45-76行目**: 内部モジュールのインポート
3. **85行目**: ネイティブバインディングの取得
4. **124-204行目**: styleText関数の実装
5. **220-239行目**: inherits関数の実装
6. **278-312行目**: callbackify関数の実装
7. **367-369行目**: parseEnv関数（ネイティブ実装へ委譲）
8. **432-462行目**: getCallSites関数の実装

#### Step 3: promisify/callbackifyの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | internal/util.js | `lib/internal/util.js` | promisifyの内部実装 |

**主要処理フロー**:
- promisifyはinternal/utilモジュールから再エクスポートされる
- callbackifyは278-312行目で直接実装

#### Step 4: inspect機能を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | internal/util/inspect.js | `lib/internal/util/inspect.js` | オブジェクト検査の詳細実装 |

### プログラム呼び出し階層図

```
lib/util.js
    │
    ├─ internal/util/inspect.js (format, formatWithOptions, inspect)
    │      └─ オブジェクトの文字列化処理
    │
    ├─ internal/util.js (promisify, deprecate, getSystemErrorMap等)
    │      └─ コア機能の実装
    │
    ├─ internal/util/debuglog.js (debuglog)
    │      └─ デバッグログ機能
    │
    ├─ internal/validators.js (validateFunction, validateNumber等)
    │      └─ 引数バリデーション
    │
    ├─ internal/util/colors.js (shouldColorize)
    │      └─ 色出力判定
    │
    └─ internalBinding('util') (parseEnv, getCallSites)
           └─ ネイティブ実装
```

### データフロー図

```
[入力]               [処理]                    [出力]

コールバック関数 ───▶ promisify ───▶ Promise返却関数
Promise関数 ───▶ callbackify ───▶ コールバック関数
任意オブジェクト ───▶ inspect ───▶ 文字列表現
フォーマット+値 ───▶ format ───▶ フォーマット済み文字列
スタイル+テキスト ───▶ styleText ───▶ ANSI装飾テキスト
.env文字列 ───▶ parseEnv ───▶ キーバリューオブジェクト
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| util.js | `lib/util.js` | ソース | メインモジュール |
| internal/util.js | `lib/internal/util.js` | ソース | 内部ユーティリティ |
| internal/util/inspect.js | `lib/internal/util/inspect.js` | ソース | オブジェクト検査 |
| internal/util/debuglog.js | `lib/internal/util/debuglog.js` | ソース | デバッグログ |
| internal/util/types.js | `lib/internal/util/types.js` | ソース | 型判定ユーティリティ |
| internal/util/colors.js | `lib/internal/util/colors.js` | ソース | 色出力判定 |
| internal/util/comparisons.js | `lib/internal/util/comparisons.js` | ソース | 深い比較 |
| internal/validators.js | `lib/internal/validators.js` | ソース | バリデーション |
| internal/encoding.js | `lib/internal/encoding.js` | ソース | TextEncoder/TextDecoder |
| internal/mime.js | `lib/internal/mime.js` | ソース | MIMEType/MIMEParams |
