# 機能設計書 27-inspector

## 概要

本ドキュメントは、Node.jsのinspectorモジュールの機能設計について記述する。inspectorモジュールは、V8インスペクタープロトコル（Chrome DevTools Protocol）へのアクセスを提供し、JavaScriptコードのデバッグ、プロファイリング、コード検査を可能にする。

### 本機能の処理概要

**業務上の目的・背景**：inspectorモジュールは、Node.jsアプリケーションのデバッグとプロファイリングのための基盤機能を提供する。Chrome DevToolsやVS Code等のデバッガーと連携し、ブレークポイントの設定、変数の検査、パフォーマンスプロファイリングなどの開発者向け機能を実現する。

**機能の利用シーン**：
- Node.jsアプリケーションをデバッグする際
- Chrome DevToolsと接続してリモートデバッグする際
- パフォーマンスプロファイルを取得する際
- ヒープスナップショットを取得する際
- コードカバレッジを計測する際
- ネットワークリクエストを監視する際

**主要な処理内容**：
1. `inspector.open([port], [host], [wait])` - インスペクターを起動
2. `inspector.close()` - インスペクターを停止
3. `inspector.url()` - WebSocket接続URLを取得
4. `inspector.waitForDebugger()` - デバッガー接続を待機
5. `inspector.Session` - インスペクタープロトコルセッションクラス
6. `Session.connect()` - インスペクターバックエンドに接続
7. `Session.connectToMainThread()` - メインスレッドのインスペクターに接続
8. `Session.post(method, [params], [callback])` - プロトコルメッセージを送信
9. `Session.disconnect()` - セッションを切断
10. `inspector.console` - インスペクターコンソール
11. `inspector.Network` - ネットワークイベント送信
12. `inspector.DOMStorage` - DOMストレージイベント送信

**関連システム・外部連携**：
- internalBinding('inspector') - ネイティブインスペクター機能
- Chrome DevTools Protocol (CDP) - デバッグプロトコル
- internal/inspector/network_resources - ネットワークリソース管理

**権限による制御**：インスペクターを公開IPにバインドする場合、セキュリティ警告が発行される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | System Analyzer | 補助機能 | V8インスペクターデバッグ情報の可視化とトレース |
| 4 | Turbolizer | 補助機能 | コンパイラ最適化フェーズのデバッグ情報表示 |

## 機能種別

デバッグ支援 / プロファイリング / 開発者ツール連携

## 入力仕様

### 入力パラメータ

#### inspector.open

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| port | number | No | リッスンポート | validateInt32(0-65535) |
| host | string | No | バインドホスト | - |
| wait | boolean | No | デバッガー接続を待機 | - |

#### Session.post

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| method | string | Yes | プロトコルメソッド名 | validateString |
| params | Object | No | メソッドパラメータ | validateObject |
| callback | Function | No | レスポンスコールバック | validateFunction |

#### Network.requestWillBeSent / responseReceived等

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| params | Object | Yes | CDPイベントパラメータ | validateObject |

### 入力データソース

プログラムコードからの直接呼び出し、またはデバッグクライアントからの接続。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| inspector.url()戻り値 | string \| undefined | WebSocket URL（例：ws://127.0.0.1:9229/...） |
| inspector.open()戻り値 | Disposable | SymbolDispose対応オブジェクト |
| Session.post callback | (err, result) | エラーまたはレスポンス |

### 出力先

- WebSocket経由でデバッグクライアントに送信
- Session.onメソッドでプロトコルイベントを受信

## 処理フロー

### 処理シーケンス

```
1. インスペクター起動（inspector.open）
   └─ ポート・ホストでWebSocketサーバー起動
2. セッション作成（new Session）
   └─ EventEmitterを継承
3. 接続（Session.connect）
   └─ ネイティブConnectionに接続
4. メッセージ送信（Session.post）
   └─ CDPメッセージをJSON化して送信
5. レスポンス/イベント受信
   └─ コールバック実行またはEventEmitter emit
6. 切断（Session.disconnect）
   └─ 保留中コールバックにエラー通知
```

### フローチャート

```mermaid
flowchart TD
    A[inspector.open] --> B[WebSocketサーバー起動]
    B --> C[デバッガー接続待機]
    C --> D[Session.connect]
    D --> E[CDPメッセージ送受信]
    E --> F{レスポンス種別}
    F -->|result| G[コールバック実行]
    F -->|event| H[emit イベント]
    G --> E
    H --> E
    E --> I[Session.disconnect]
    I --> J[保留コールバックにエラー]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | セキュリティ警告 | 公開IPにバインド時にSecurityWarningを発行 | host != loopback |
| BR-002 | 既にアクティブ | 二重起動時にERR_INSPECTOR_ALREADY_ACTIVATED | isEnabled() = true |
| BR-003 | 二重接続禁止 | 既に接続済みのセッションに再接続不可 | this.#connection != null |
| BR-004 | ワーカー制限 | connectToMainThreadはワーカースレッドでのみ使用可能 | isMainThread = false |

### 計算ロジック

特になし。

## データベース操作仕様

### 操作別データベース影響一覧

本モジュールはデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_INSPECTOR_NOT_AVAILABLE | Error | インスペクターが利用不可 | --inspectフラグでNode.jsを起動 |
| ERR_INSPECTOR_ALREADY_ACTIVATED | Error | 二重起動 | 既存のインスペクターを使用 |
| ERR_INSPECTOR_ALREADY_CONNECTED | Error | セッション二重接続 | 既存セッションを使用 |
| ERR_INSPECTOR_NOT_CONNECTED | Error | 未接続でpost | 先にconnect()を呼ぶ |
| ERR_INSPECTOR_CLOSED | Error | 切断後のコールバック | セッションの再接続 |
| ERR_INSPECTOR_COMMAND | Error | CDPコマンドエラー | コマンドパラメータを確認 |
| ERR_INSPECTOR_NOT_ACTIVE | Error | インスペクターが非アクティブ | openを呼び出す |
| ERR_INSPECTOR_NOT_WORKER | Error | ワーカー以外でconnectToMainThread | ワーカースレッドで使用 |

### リトライ仕様

リトライ処理は行わない。

## トランザクション仕様

トランザクション処理は行わない。

## パフォーマンス要件

- インスペクターはオンデマンドで起動（常時有効化は不要）
- デバッグ時のみオーバーヘッドが発生

## セキュリティ考慮事項

- 公開IPへのバインドはリモートコード実行の脆弱性につながる
- 本番環境でのインスペクター有効化は非推奨
- 公式ドキュメント参照：https://nodejs.org/api/cli.html#--inspecthostport

## 備考

- Session#disconnectで保留中の全コールバックにERR_INSPECTOR_CLOSEDを通知
- inspector.consoleはV8インスペクターコンソールへの直接アクセス
- Network/DOMStorageはCDPイベントをフロントエンドにブロードキャスト

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | inspector.js | `lib/inspector.js` | Sessionクラスのプライベートフィールド（55-57行目） |

**読解のコツ**: Sessionクラスは#connection、#nextId、#messageCallbacksの3つのプライベートフィールドを持つ。#messageCallbacksはリクエストIDとコールバックのマップ。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | inspector.js | `lib/inspector.js` | module.exports（242-252行目） |

**主要処理フロー**:
1. **242行目**: open関数のエクスポート
2. **243行目**: closeのエクスポート（_debugEnd）
3. **246行目**: Sessionクラスのエクスポート
4. **247-250行目**: Network、NetworkResources、DOMStorageのエクスポート

#### Step 3: inspector.open実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | inspector.js | `lib/inspector.js` | inspectorOpen関数（169-196行目） |

**主要処理フロー**:
- **170-172行目**: 既にアクティブな場合のエラー
- **177-178行目**: ポートの範囲検証（0-65535）
- **180-189行目**: 非ループバックIPへのセキュリティ警告
- **191行目**: ネイティブopen関数の呼び出し
- **192-193行目**: waitオプション対応
- **195行目**: Disposableオブジェクトの返却

#### Step 4: Sessionクラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | inspector.js | `lib/inspector.js` | Sessionクラス（54-160行目） |

**主要処理フロー**:
- **63-67行目**: connect() - Connectionインスタンス生成
- **74-82行目**: connectToMainThread() - MainThreadConnection使用
- **84-106行目**: #onMessage() - レスポンス/イベント振り分け
- **115-140行目**: post() - CDPメッセージ送信
- **148-159行目**: disconnect() - クリーンアップと保留コールバック処理

#### Step 5: ネットワーク/DOMStorage機能を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | inspector.js | `lib/inspector.js` | Network/DOMStorageオブジェクト（215-240行目） |

**主要処理フロー**:
- **209-213行目**: broadcastToFrontend() - CDPイベントの送信
- **215-226行目**: Network - ネットワーク関連CDPイベント
- **232-240行目**: DOMStorage - ストレージ関連CDPイベント

### プログラム呼び出し階層図

```
lib/inspector.js
    │
    ├─ inspectorOpen() [169-196行目]
    │      ├─ isEnabled() (internalBinding)
    │      ├─ isLoopback() (internal/net)
    │      ├─ open() (internalBinding)
    │      └─ waitForDebugger() (internalBinding)
    │
    ├─ Session クラス [54-160行目]
    │      ├─ connect() [63-67行目]
    │      │      └─ new Connection() (internalBinding)
    │      ├─ connectToMainThread() [74-82行目]
    │      │      └─ new MainThreadConnection() (internalBinding)
    │      ├─ #onMessage() [84-106行目]
    │      ├─ post() [115-140行目]
    │      │      └─ #connection.dispatch()
    │      └─ disconnect() [148-159行目]
    │
    ├─ Network [215-226行目]
    │      └─ broadcastToFrontend() [209-213行目]
    │              └─ emitProtocolEvent() (internalBinding)
    │
    └─ DOMStorage [232-240行目]
           └─ broadcastToFrontend()
```

### データフロー図

```
[入力]               [処理]                    [出力]

port, host ───▶ inspector.open ───▶ WebSocket URL
                         │
                         ▼
               デバッグクライアント接続
                         │
                         ▼
CDPメソッド+パラメータ ───▶ Session.post ───▶ CDPレスポンス
                         │
                         ▼
               CDPイベント ───▶ Session.emit
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| inspector.js | `lib/inspector.js` | ソース | メインモジュール |
| internal/inspector/network_resources.js | `lib/internal/inspector/network_resources.js` | ソース | ネットワークリソース管理 |
| internal/net.js | `lib/internal/net.js` | ソース | isLoopback判定 |
| internal/process/task_queues.js | `lib/internal/process/task_queues.js` | ソース | queueMicrotask |
| events.js | `lib/events.js` | ソース | EventEmitter基底クラス |
