# 機能設計書 3-http

## 概要

本ドキュメントは、Node.jsのhttpモジュール（HTTPサーバー・クライアント機能）の機能設計を記載する。httpモジュールは、HTTP/1.1プロトコルに基づくWebサーバーの構築およびHTTPリクエストの送信機能を提供するコアモジュールである。

### 本機能の処理概要

httpモジュールは、Node.jsでHTTP通信を行うための基盤機能を提供する。サーバー側ではリクエストを受信してレスポンスを返却し、クライアント側では外部サーバーへのHTTPリクエストを実行する。

**業務上の目的・背景**：Webアプリケーション開発において、HTTPサーバーの構築とHTTPクライアント機能は最も基本的な要件である。httpモジュールは、Express.js、Fastify等の上位フレームワークの基盤として、また直接利用することでシンプルなHTTPサーバーを構築するために使用される。RESTful API、WebSocket（初期ハンドシェイク）、Server-Sent Events等の実装基盤となる。

**機能の利用シーン**：
- RESTful APIサーバーの構築
- 静的ファイルサーバーの実装
- プロキシサーバーの構築
- 外部APIへのHTTPリクエスト送信
- マイクロサービス間通信
- WebSocketアップグレード処理

**主要な処理内容**：
1. HTTPサーバーの作成と起動（createServer, Server.listen）
2. リクエスト受信とレスポンス送信（IncomingMessage, ServerResponse）
3. HTTPクライアントリクエスト（request, get, ClientRequest）
4. ヘッダー操作（validateHeaderName, validateHeaderValue）
5. コネクション管理（Agent, globalAgent）
6. プロキシサポート（setGlobalProxyFromEnv）
7. WebSocket対応（undici経由）

**関連システム・外部連携**：
- netモジュール（TCP接続）
- tlsモジュール（HTTPS時）
- dnsモジュール（ホスト名解決）
- undici（WebSocket、プロキシ）

**権限による制御**：特別な権限制御なし。ポート番号1024未満はOS権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 直接関連する画面なし |

## 機能種別

ネットワーク通信 / サーバー構築 / クライアント通信

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| url | string / URL | Yes（リクエスト時） | リクエスト先URL | URL形式チェック |
| options | HTTPRequestOptions | No | リクエストオプション | validateObject |
| opts | object | No | サーバーオプション | validateObject |
| requestListener | Function | No | リクエストハンドラ | validateFunction |
| port | number | No | リッスンポート | validatePort |
| host | string | No | リッスンホスト | - |

### 入力データソース

- サーバー：クライアントからのHTTPリクエスト
- クライアント：アプリケーションコードからのAPI呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| server | Server | HTTPサーバーインスタンス |
| request | ClientRequest | クライアントリクエストインスタンス |
| response | IncomingMessage | レスポンスメッセージ |
| statusCode | number | HTTPステータスコード |
| headers | object | HTTPヘッダー |
| body | Buffer / string | レスポンスボディ |

### 出力先

- サーバー：クライアントへのHTTPレスポンス
- クライアント：イベント（'response'）またはコールバック

## 処理フロー

### 処理シーケンス（サーバー）

```
1. createServer()でServerインスタンス生成
   └─ requestListenerをrequestイベントに登録
2. server.listen()でポートをバインド
   └─ netモジュールでTCP待ち受け開始
3. クライアント接続時
   └─ _connectionListenerでHTTP解析開始
4. リクエスト受信完了
   └─ requestイベント発火（req, res引数）
5. レスポンス送信
   └─ res.writeHead(), res.end()
```

### 処理シーケンス（クライアント）

```
1. request()またはget()でClientRequest生成
   └─ URL解析、オプション正規化
2. エージェントによるコネクション取得
   └─ 新規作成またはキープアライブ接続再利用
3. リクエスト送信
   └─ req.write()、req.end()
4. レスポンス受信
   └─ responseイベント発火
5. ボディ受信
   └─ dataイベントでチャンク受信
```

### フローチャート

```mermaid
flowchart TD
    A[createServer] --> B[Server インスタンス]
    B --> C[listen]
    C --> D[TCP待ち受け開始]
    D --> E{接続あり?}
    E -->|Yes| F[_connectionListener]
    F --> G[HTTP解析]
    G --> H[requestイベント]
    H --> I[ハンドラ実行]
    I --> J[レスポンス送信]
    J --> E
    E -->|close| K[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | HTTPメソッド | METHODSでサポートされるメソッドを定義 | すべてのリクエスト |
| BR-002 | ステータスコード | STATUS_CODESでコードと説明を定義 | レスポンス生成時 |
| BR-003 | ヘッダー検証 | validateHeaderName/Valueで不正ヘッダーを検出 | ヘッダー設定時 |
| BR-004 | エージェント管理 | globalAgentでデフォルト接続プールを管理 | コネクション時 |
| BR-005 | プロキシ対応 | 環境変数からプロキシ設定を読み込み | setGlobalProxyFromEnv使用時 |

### 計算ロジック

- maxHeaderSize: `--max-http-header-size`オプションで設定
- parsers.max: `setMaxIdleHTTPParsers()`で設定

## データベース操作仕様

本モジュールはデータベースを直接操作しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ECONNREFUSED | OSエラー | 接続先サーバーが拒否 | 接続先を確認 |
| ECONNRESET | OSエラー | 接続がリセットされた | リトライを検討 |
| ETIMEDOUT | OSエラー | 接続タイムアウト | タイムアウト値を調整 |
| ERR_PROXY_INVALID_CONFIG | プロキシエラー | プロキシURL形式不正 | プロキシ設定を確認 |
| HPE_* | パースエラー | HTTP解析エラー | リクエスト形式を確認 |

### リトライ仕様

クライアントリクエストのリトライはアプリケーション側で実装が必要。

## トランザクション仕様

HTTPはステートレスプロトコルであり、トランザクション管理はアプリケーション層で実装する。

## パフォーマンス要件

- Keep-Alive接続によるコネクション再利用
- HTTPパーサープールによるメモリ効率化
- ストリーミングによる大容量データ対応

## セキュリティ考慮事項

- ヘッダーインジェクション対策（validateHeaderName/Value）
- HTTPリクエストスマグリング対策
- レスポンス分割攻撃対策
- 大きなヘッダーによるDoS攻撃対策（maxHeaderSize）

## 備考

- HTTP/2はhttp2モジュールを使用
- HTTPS通信はhttpsモジュールを使用
- WebSocketはundici経由で提供

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、HTTPリクエスト・レスポンスのデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | _http_incoming.js | `lib/_http_incoming.js` | IncomingMessageクラス |
| 1-2 | _http_outgoing.js | `lib/_http_outgoing.js` | OutgoingMessageクラス |
| 1-3 | _http_server.js | `lib/_http_server.js` | Server, ServerResponseクラス |

**読解のコツ**: IncomingMessageは受信したHTTPメッセージを表し、ServerResponseはサーバーからの応答を構築するために使用される。

#### Step 2: エントリーポイントを理解する

処理の起点となるファイル・関数を特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | http.js | `lib/http.js` | 公開API定義 |

**主要処理フロー**:
1. **68-70行目**: createServer()関数 - Serverインスタンス生成
2. **107-109行目**: request()関数 - ClientRequest生成
3. **118-122行目**: get()関数 - GETリクエストのショートカット
4. **132-182行目**: setGlobalProxyFromEnv()関数 - プロキシ設定
5. **184-204行目**: モジュールエクスポート

#### Step 3: 内部実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | _http_agent.js | `lib/_http_agent.js` | Agent（コネクションプール）実装 |
| 3-2 | _http_client.js | `lib/_http_client.js` | ClientRequest実装 |
| 3-3 | _http_common.js | `lib/_http_common.js` | 共通ユーティリティ |

### プログラム呼び出し階層図

```
http.createServer(opts, requestListener)
    │
    └─ new Server(opts, requestListener)
           │
           ├─ storeHTTPOptions(opts)
           │
           ├─ net.Server.call(this)
           │      └─ TCP待ち受け準備
           │
           └─ this.addListener('request', requestListener)

http.request(url, options, cb)
    │
    └─ new ClientRequest(url, options, cb)
           │
           ├─ URL解析
           │
           ├─ Agent取得 (options.agent || globalAgent)
           │
           └─ socket取得 → リクエスト送信
```

### データフロー図

```
[クライアント]              [サーバー]                    [レスポンス]

HTTPリクエスト   ───▶  _connectionListener()
                           │
                      _http_common.parser   ───▶  パース結果
                           │
                      IncomingMessage       ───▶  req オブジェクト
                           │
                      ServerResponse        ───▶  res オブジェクト
                           │
                      requestListener(req, res)
                           │
                      res.writeHead()
                      res.end()             ───▶  HTTPレスポンス
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| http.js | `lib/http.js` | ソース | 公開API定義 |
| _http_agent.js | `lib/_http_agent.js` | ソース | Agent（コネクション管理） |
| _http_client.js | `lib/_http_client.js` | ソース | ClientRequest実装 |
| _http_common.js | `lib/_http_common.js` | ソース | 共通ユーティリティ |
| _http_incoming.js | `lib/_http_incoming.js` | ソース | IncomingMessage実装 |
| _http_outgoing.js | `lib/_http_outgoing.js` | ソース | OutgoingMessage実装 |
| _http_server.js | `lib/_http_server.js` | ソース | Server, ServerResponse実装 |
| internal/http.js | `lib/internal/http.js` | ソース | 内部ユーティリティ |
