# 機能設計書 31-diagnostics_channel

## 概要

本ドキュメントは、Node.js の `diagnostics_channel` モジュールの機能設計書である。このモジュールは、診断情報をパブリッシュ・サブスクライブ型のパターンでやり取りするための軽量なチャンネルAPIを提供する。

### 本機能の処理概要

diagnostics_channel モジュールは、Node.js アプリケーションにおける診断データの収集と配信を可能にする Pub/Sub メッセージングシステムを提供する。モジュール開発者が任意の名前付きチャンネルを作成し、サブスクライバーがそのチャンネルに登録することで、診断情報をリアルタイムに受信できる。

**業務上の目的・背景**：アプリケーションやライブラリの内部状態を監視・デバッグするために、一貫した診断データの収集メカニズムが必要とされる。このモジュールにより、ライブラリ開発者はユーザーのコードに侵入することなく診断ポイントを公開でき、APM（Application Performance Monitoring）ツールやロギングシステムがこれらのデータを収集できる。

**機能の利用シーン**：
- APMツール（DataDog、New Relicなど）がHTTPリクエストのライフサイクルをトレースする場合
- デバッグ目的でライブラリ内部の処理状況を監視する場合
- 分散トレーシングシステムでコンテキストを伝搬する場合
- カスタムロギングやメトリクス収集システムを構築する場合

**主要な処理内容**：
1. 名前付きチャンネルの作成・取得（`channel()` 関数）
2. チャンネルへのサブスクリプション登録・解除（`subscribe()` / `unsubscribe()`）
3. チャンネルへのメッセージパブリッシュ（`publish()`）
4. AsyncLocalStorageとの統合によるコンテキスト伝搬（`bindStore()` / `runStores()`）
5. トレーシング用の複合チャンネル管理（`TracingChannel` クラス）

**関連システム・外部連携**：
- `async_hooks` モジュール（AsyncLocalStorageによるコンテキスト管理）
- APMツール、分散トレーシングシステム、ロギングフレームワーク

**権限による制御**：特段の権限制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | CLIツールのため画面なし |

## 機能種別

イベント配信 / データ連携 / コンテキスト管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| name | string \| symbol | Yes | チャンネルの識別名 | 文字列またはシンボルであること |
| subscription | Function | Yes | メッセージを受信するコールバック関数 | 関数であること |
| data | any | No | パブリッシュするメッセージデータ | なし |
| store | AsyncLocalStorage | No | バインドするAsyncLocalStorageインスタンス | AsyncLocalStorageであること |
| transform | Function | No | データ変換関数 | 関数であること |

### 入力データソース

- アプリケーションコードからの直接呼び出し
- ライブラリ内部からの診断データ発行

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| channel | Channel | 指定した名前のチャンネルオブジェクト |
| hasSubscribers | boolean | チャンネルにサブスクライバーが存在するかどうか |
| unsubscribed | boolean | unsubscribeの成功/失敗 |

### 出力先

- サブスクライバーのコールバック関数への配信
- 戻り値としての各種オブジェクト・真偽値

## 処理フロー

### 処理シーケンス

```
1. チャンネル取得
   └─ channel(name) で既存チャンネルを取得、なければ新規作成
2. サブスクリプション登録
   └─ channel.subscribe(callback) でコールバックを登録
3. メッセージパブリッシュ
   └─ channel.publish(data) で全サブスクライバーに配信
4. ストアバインド（オプション）
   └─ channel.bindStore(store, transform) でAsyncLocalStorageを連携
5. トレーシング実行
   └─ TracingChannel.traceSync/tracePromise/traceCallback でライフサイクル追跡
```

### フローチャート

```mermaid
flowchart TD
    A[channel/name取得] --> B{チャンネル存在?}
    B -->|Yes| C[既存チャンネル返却]
    B -->|No| D[新規Channelインスタンス作成]
    D --> E[WeakRefMapに登録]
    E --> C
    C --> F{サブスクライブ?}
    F -->|Yes| G[ActiveChannelに昇格]
    G --> H[_subscribersに追加]
    F -->|No| I{パブリッシュ?}
    I -->|Yes| J{hasSubscribers?}
    J -->|Yes| K[全サブスクライバーに配信]
    J -->|No| L[何もしない]
    K --> M[完了]
    L --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | チャンネルの一意性 | 同一名のチャンネルは常に同一インスタンスを返す | channel()呼び出し時 |
| BR-02 | 遅延アクティブ化 | サブスクライバーがいない場合は軽量プロトタイプを使用 | パフォーマンス最適化 |
| BR-03 | エラー非伝搬 | サブスクライバーでの例外は他のサブスクライバーに影響しない | publish()実行時 |
| BR-04 | GC連動 | 参照がなくなったチャンネルはGCされる | WeakRefMap使用 |

### 計算ロジック

特段の計算ロジックはなし。

## データベース操作仕様

該当なし（インメモリのみ）

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_INVALID_ARG_TYPE | TypeError | name が string/symbol 以外 | 適切な型を渡す |
| ERR_INVALID_ARG_TYPE | TypeError | subscription が関数以外 | 関数を渡す |
| uncaughtException | Error | サブスクライバー内での例外 | process.nextTickで非同期発火 |

### リトライ仕様

リトライは行わない。サブスクライバー内のエラーは次のtickで uncaughtException として発火される。

## トランザクション仕様

該当なし

## パフォーマンス要件

- サブスクライバーがいない場合のpublish()は実質ノーオペレーション
- プロトタイプ切り替えによる最適化で、非アクティブチャンネルのオーバーヘッドを最小化

## セキュリティ考慮事項

- チャンネル名は文字列またはシンボルのみ許可
- 診断データには機密情報が含まれる可能性があるため、サブスクライバーの管理に注意が必要

## 備考

- `TracingChannel` クラスは、start/end/asyncStart/asyncEnd/error の5つの関連チャンネルをまとめて管理する
- Node.js内部のHTTP、DNS等の主要モジュールが診断チャンネルを提供している

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、チャンネルの基本構造と状態管理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | diagnostics_channel.js | `lib/diagnostics_channel.js` | Channel/ActiveChannelクラスの構造 |

**読解のコツ**: `Channel` クラスは非アクティブ状態、`ActiveChannel` クラスはサブスクライバーが存在する状態を表す。プロトタイプの切り替えによりパフォーマンスを最適化している。

#### Step 2: エントリーポイントを理解する

モジュールのエクスポートと主要関数の役割を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | diagnostics_channel.js | `lib/diagnostics_channel.js` | module.exports（437-444行目） |

**主要処理フロー**:
1. **227-236行目**: `channel()` 関数 - 名前からチャンネルを取得/作成
2. **238-244行目**: `subscribe()` / `unsubscribe()` - グローバルショートカット関数
3. **246-251行目**: `hasSubscribers()` - サブスクライバー存在確認

#### Step 3: Channelクラスの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | diagnostics_channel.js | `lib/diagnostics_channel.js` | Channelクラス（181-223行目） |
| 3-2 | diagnostics_channel.js | `lib/diagnostics_channel.js` | ActiveChannelクラス（105-179行目） |

**主要処理フロー**:
- **67-72行目**: `markActive()` - Channel を ActiveChannel に昇格
- **74-82行目**: `maybeMarkInactive()` - サブスクライバーがいなくなったら降格
- **106-111行目**: `ActiveChannel.subscribe()` - サブスクライバー登録
- **151-163行目**: `ActiveChannel.publish()` - メッセージ配信

#### Step 4: TracingChannelを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | diagnostics_channel.js | `lib/diagnostics_channel.js` | TracingChannelクラス（283-431行目） |

**主要処理フロー**:
- **283-292行目**: コンストラクタ - 5つのイベントチャンネルを作成
- **326-346行目**: `traceSync()` - 同期処理のトレース
- **348-388行目**: `tracePromise()` - Promise処理のトレース
- **390-430行目**: `traceCallback()` - コールバック処理のトレース

#### Step 5: WeakRefMapによるメモリ管理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | diagnostics_channel.js | `lib/diagnostics_channel.js` | WeakRefMapクラス（38-65行目） |

**主要処理フロー**:
- **38-43行目**: FinalizationRegistryによるGC連動クリーンアップ
- **45-48行目**: `set()` - WeakReferenceでの保存
- **58-64行目**: 参照カウント管理（incRef/decRef）

### プログラム呼び出し階層図

```
channel(name)
    │
    ├─ channels.get(name)
    │      └─ WeakRefMap.get() → WeakReference.get()
    │
    └─ new Channel(name)
           └─ channels.set(name, this)
                  └─ WeakRefMap.set() → new WeakReference(value)

Channel.subscribe(subscription)
    │
    ├─ markActive(this)
    │      ├─ ObjectSetPrototypeOf(channel, ActiveChannel.prototype)
    │      └─ _subscribers = [], _stores = new SafeMap()
    │
    └─ ActiveChannel.subscribe(subscription)
           ├─ validateFunction(subscription)
           ├─ ArrayPrototypePush(_subscribers, subscription)
           └─ channels.incRef(name)

ActiveChannel.publish(data)
    │
    └─ for each subscriber
           └─ subscriber(data, channelName)
                  └─ on error: process.nextTick(triggerUncaughtException)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

publish(data) ───────▶ ActiveChannel.publish() ───────▶ subscriber callbacks
                              │
                              ├─ for each _subscribers
                              │     └─ callback(data, name)
                              │
                              └─ runStores で AsyncLocalStorage 連携
                                    └─ store.run(context, fn)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| diagnostics_channel.js | `lib/diagnostics_channel.js` | ソース | メインモジュール実装 |
| internal/validators.js | `lib/internal/validators.js` | ソース | 入力バリデーション |
| internal/util.js | `lib/internal/util.js` | ソース | WeakReferenceクラス提供 |
| internal/errors.js | `lib/internal/errors.js` | ソース | エラーコード定義 |
