# 機能設計書 45-constants

## 概要

本ドキュメントは、Node.jsの`constants`モジュールの機能設計を記載する。このモジュールは、OS、ファイルシステム、暗号化関連の各種定数を一か所に集約して提供する。ただし、ドキュメント上では非推奨とされており、各モジュール固有の定数を使用することが推奨されている。

### 本機能の処理概要

constantsモジュールは、Node.js内部で使用される各種システム定数を一つのオブジェクトとしてエクスポートする。これらの定数は主にファイルシステム操作、シグナル処理、暗号化操作などで使用される低レベルのOS定数である。

**業務上の目的・背景**：Node.jsの初期バージョンでは、すべてのシステム定数が`constants`モジュールに集約されていた。しかし、定数の数が増加し、各モジュール（fs、os、crypto）が独自の定数を持つようになったため、この集約モジュールはドキュメント上で非推奨（DEP0008）となった。現在は、各モジュールの`.constants`プロパティ（例：`fs.constants`、`os.constants`、`crypto.constants`）を使用することが推奨されている。

**機能の利用シーン**：
- レガシーコードの互換性維持
- 複数モジュールの定数を一括インポート
- 低レベルシステム操作での定数参照

**主要な処理内容**：
1. OSの動的リンカー（dlopen）関連定数
2. OSエラー番号（errno）定数
3. プロセス優先度（priority）定数
4. シグナル（signals）定数
5. ファイルシステム（fs）定数
6. 暗号化（crypto）定数

**関連システム・外部連携**：
- `os`モジュール：OS関連定数
- `fs`モジュール：ファイルシステム定数
- `crypto`モジュール：暗号化定数

**権限による制御**：特になし（定数のみの提供）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに該当なし |

## 機能種別

定数定義 / ユーティリティ

## 入力仕様

### 入力パラメータ

該当なし（定数のみのモジュール）

### 入力データソース

C++バインディング（`internalBinding('constants')`）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| RTLD_LAZY | Number | dlopen: 遅延バインド |
| RTLD_NOW | Number | dlopen: 即時バインド |
| RTLD_GLOBAL | Number | dlopen: グローバルシンボル |
| RTLD_LOCAL | Number | dlopen: ローカルシンボル |
| ENOENT | Number | errno: ファイルなし |
| EACCES | Number | errno: アクセス拒否 |
| SIGINT | Number | signal: 割り込み |
| SIGTERM | Number | signal: 終了 |
| O_RDONLY | Number | fs: 読み取り専用 |
| O_WRONLY | Number | fs: 書き込み専用 |
| O_RDWR | Number | fs: 読み書き |
| SSL_OP_* | Number | crypto: SSL/TLSオプション |

### 出力先

呼び出し元への定数値

## 処理フロー

### 処理シーケンス

```
1. モジュール読み込み
   └─ require('constants')
2. C++バインディングから定数取得
   └─ internalBinding('constants')
3. 定数の集約
   └─ os.dlopen, os.errno, os.priority, os.signals
   └─ fs定数
   └─ crypto定数
4. オブジェクトのフリーズ
   └─ ObjectFreeze(exports)
5. エクスポート
   └─ フリーズされた定数オブジェクト
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[internalBinding 'constants']
    B --> C[constants.os.dlopen取得]
    C --> D[constants.os.errno取得]
    D --> E[constants.os.priority取得]
    E --> F[constants.os.signals取得]
    F --> G[constants.fs取得]
    G --> H[constants.crypto取得]
    H --> I[ObjectAssign exports]
    I --> J[ObjectFreeze exports]
    J --> K[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 非推奨 | ドキュメント上で非推奨（DEP0008） | 常時 |
| BR-02 | 読み取り専用 | エクスポートされたオブジェクトはフリーズされている | 常時 |
| BR-03 | フラット構造 | すべての定数がトップレベルにフラット化 | 常時 |
| BR-04 | プラットフォーム依存 | 定数の値はOSによって異なる可能性 | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

該当なし（定数のみのモジュールで、エラーは発生しない）

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- モジュールロード時に一度だけ定数が集約される
- フリーズされたオブジェクトへのアクセスは高速

## セキュリティ考慮事項

- 定数オブジェクトはObjectFreezeで保護されている
- 変更不可能なため、定数の改ざんはできない

## 備考

- ドキュメント非推奨：DEP0008
- 代替として各モジュールの`.constants`プロパティを使用:
  - `require('fs').constants`
  - `require('os').constants`
  - `require('crypto').constants`

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

モジュールの構造とC++バインディングからの定数取得を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | constants.js | `lib/constants.js` | 1-20行目のライセンスコメント |
| 1-2 | constants.js | `lib/constants.js` | 22-27行目のprimordials import |

**読解のコツ**: このモジュールはライセンスコメントが大部分を占める非常にシンプルな構造。`ObjectAssign`と`ObjectFreeze`のみがprimordialsからインポートされている。

#### Step 2: エントリーポイントを理解する

定数の集約とエクスポートを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | constants.js | `lib/constants.js` | 29-32行目のコメント |
| 2-2 | constants.js | `lib/constants.js` | 33行目のinternalBinding |
| 2-3 | constants.js | `lib/constants.js` | 34-40行目のObjectAssign |
| 2-4 | constants.js | `lib/constants.js` | 41行目のObjectFreeze |

**主要処理フロー**:
1. **29-32行目**: 非推奨に関するコメント（DEP0008）
2. **33行目**: `internalBinding('constants')`でC++からの定数取得
3. **34-40行目**: `ObjectAssign`で複数のソースからexportsに定数を集約
   - `constants.os.dlopen`: 動的リンカー関連
   - `constants.os.errno`: エラー番号
   - `constants.os.priority`: プロセス優先度
   - `constants.os.signals`: シグナル
   - `constants.fs`: ファイルシステム
   - `constants.crypto`: 暗号化
4. **41行目**: `ObjectFreeze`でオブジェクトを凍結

### プログラム呼び出し階層図

```
require('constants')
    │
    └─ lib/constants.js
           │
           ├─ internalBinding('constants')
           │      │
           │      └─ C++ バインディング
           │             │
           │             ├─ constants.os
           │             │      ├─ dlopen (RTLD_*)
           │             │      ├─ errno (E*)
           │             │      ├─ priority (PRIORITY_*)
           │             │      └─ signals (SIG*)
           │             │
           │             ├─ constants.fs (O_*, S_*, F_*)
           │             │
           │             └─ constants.crypto (SSL_OP_*, etc.)
           │
           ├─ ObjectAssign(exports, ...)
           │      └─ 全定数をフラット化してexportsにマージ
           │
           └─ ObjectFreeze(exports)
                  └─ exportsオブジェクトを凍結（変更不可）
```

### データフロー図

```
[入力: C++バインディング]              [処理]                    [出力]

constants.os.dlopen   ───┐
  RTLD_LAZY              │
  RTLD_NOW               │
  RTLD_GLOBAL            │
  RTLD_LOCAL             │
                         │
constants.os.errno    ───┤      ┌──────────────────┐
  ENOENT                 ├─────▶│  ObjectAssign    │
  EACCES                 │      │  (フラット化)     │
  EEXIST                 │      └────────┬─────────┘
  ...                    │               │
                         │               ▼
constants.os.signals  ───┤      ┌──────────────────┐
  SIGINT                 │      │  ObjectFreeze    │───▶  exports
  SIGTERM                │      │  (凍結)           │      (読み取り専用)
  SIGHUP                 │      └──────────────────┘
  ...                    │
                         │
constants.fs          ───┤
  O_RDONLY               │
  O_WRONLY               │
  O_RDWR                 │
  S_IRUSR                │
  ...                    │
                         │
constants.crypto      ───┘
  SSL_OP_*
  ...
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| constants.js | `lib/constants.js` | ソース | 定数集約モジュールのエントリーポイント |
| os.js | `lib/os.js` | ソース | os.constants（推奨代替）の提供元 |
| fs.js | `lib/fs.js` | ソース | fs.constants（推奨代替）の提供元 |
| crypto.js | `lib/crypto.js` | ソース | crypto.constants（推奨代替）の提供元 |
