# 機能設計書 5-http2

## 概要

本ドキュメントは、Node.jsのhttp2モジュール（HTTP/2プロトコルサポート）の機能設計を記載する。http2モジュールは、HTTP/2プロトコルに基づく高性能なWebサーバー・クライアント機能を提供するコアモジュールである。

### 本機能の処理概要

http2モジュールは、HTTP/2プロトコルの完全な実装を提供する。マルチプレキシング、サーバープッシュ、ヘッダー圧縮（HPACK）、ストリーム優先度などのHTTP/2特有の機能をサポートし、HTTP/1.1と比較して大幅なパフォーマンス向上を実現する。

**業務上の目的・背景**：HTTP/2は単一TCP接続上で複数のリクエスト/レスポンスを並列処理できるため、Webアプリケーションのパフォーマンスを大幅に向上させる。特に、多数のリソースを読み込むSPAやAPIサーバーで効果を発揮する。また、サーバープッシュにより、クライアントがリクエストする前にリソースを送信できる。

**機能の利用シーン**：
- 高パフォーマンスWebサーバーの構築
- リアルタイムAPI（gRPCの基盤）
- サーバープッシュによるリソース配信最適化
- マイクロサービス間の効率的な通信
- ストリーミングAPI

**主要な処理内容**：
1. HTTP/2サーバーの作成（createServer, createSecureServer）
2. HTTP/2クライアント接続（connect）
3. ストリーム管理（Http2Stream）
4. セッション管理（Http2Session）
5. サーバープッシュ（pushStream）
6. 設定管理（getDefaultSettings, getPackedSettings）
7. ALPN（Application-Layer Protocol Negotiation）
8. ヘッダー圧縮（HPACK）

**関連システム・外部連携**：
- nghttp2ライブラリ（HTTP/2実装）
- tlsモジュール（暗号化）
- netモジュール（TCP接続）
- httpモジュール（互換性レイヤー）

**権限による制御**：特別な権限制御なし。暗号化が必須の環境ではTLS証明書が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 直接関連する画面なし |

## 機能種別

ネットワーク通信 / ストリーム処理 / サーバー構築

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| authority | string / URL | Yes（connect時） | 接続先のauthority | validateString |
| options | object | No | サーバー/クライアントオプション | validateObject |
| key | string / Buffer | Yes（SecureServer） | TLS秘密鍵 | - |
| cert | string / Buffer | Yes（SecureServer） | TLS証明書 | - |
| settings | object | No | HTTP/2設定 | assertIsObject |
| onRequestHandler | Function | No | リクエストハンドラ | validateFunction |

### 入力データソース

- 設定：アプリケーションコード
- 証明書：ファイルシステムまたは設定から読み込み
- ストリームデータ：クライアントからのリクエスト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| server | Http2Server / Http2SecureServer | HTTP/2サーバーインスタンス |
| session | ClientHttp2Session | クライアントセッション |
| stream | Http2Stream | HTTP/2ストリーム |
| headers | object | HTTP/2ヘッダー |
| settings | object | HTTP/2設定値 |

### 出力先

- サーバー：クライアントへのHTTP/2レスポンス
- クライアント：イベント（'response', 'stream'等）

## 処理フロー

### 処理シーケンス（サーバー）

```
1. createServer/createSecureServerでサーバー生成
   └─ 設定の検証と初期化
2. server.listen()でポートバインド
3. クライアント接続
   └─ TLSハンドシェイク（SecureServer）
   └─ ALPN: h2ネゴシエーション
4. HTTP/2セッション確立
   └─ SETTINGS フレーム交換
5. ストリーム受信
   └─ onSessionHeaders()でヘッダー処理
6. リクエスト処理
   └─ Http2ServerRequest/Http2ServerResponse
7. レスポンス送信
   └─ ストリームでデータ送信
```

### 処理シーケンス（クライアント）

```
1. connect()でセッション生成
   └─ 接続先authorityの検証
2. TCP/TLS接続確立
3. SETTINGS フレーム交換
4. request()でストリーム作成
   └─ HEADERSフレーム送信
5. レスポンス受信
   └─ 'response'イベント発火
6. データ受信
   └─ 'data'イベントでチャンク受信
```

### フローチャート

```mermaid
flowchart TD
    A[createSecureServer] --> B[TLSサーバー初期化]
    B --> C[listen]
    C --> D{接続あり?}
    D -->|Yes| E[TLSハンドシェイク]
    E --> F[ALPN: h2確認]
    F --> G[Http2Session生成]
    G --> H[SETTINGS交換]
    H --> I{ストリームあり?}
    I -->|Yes| J[onSessionHeaders]
    J --> K[Http2Stream生成]
    K --> L[リクエストハンドラ]
    L --> M[レスポンス送信]
    M --> I
    D -->|close| N[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 最大フレームサイズ | kMaxFrameSize = (2^24) - 1 | すべてのフレーム |
| BR-002 | 最大ストリーム数 | kMaxStreams = (2^32) - 1 | セッションあたり |
| BR-003 | 最大初期ウィンドウサイズ | kMaxInitialWindowSize = (2^31) - 1 | フロー制御 |
| BR-004 | センシティブヘッダー | kSensitiveHeadersでマーク | ヘッダー圧縮時 |
| BR-005 | ステータス101禁止 | HTTP_STATUS_SWITCHING_PROTOCOLS不可 | HTTP/2仕様 |

### 計算ロジック

- kMaxFrameSize: (2^24) - 1 = 16,777,215 バイト
- kMaxInitialWindowSize: (2^31) - 1 = 2,147,483,647 バイト
- NGHTTP2_DEFAULT_WEIGHT: デフォルトストリーム優先度重み

## データベース操作仕様

本モジュールはデータベースを直接操作しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERR_HTTP2_INVALID_SESSION | セッションエラー | 無効なセッション | セッション状態確認 |
| ERR_HTTP2_INVALID_STREAM | ストリームエラー | 無効なストリーム | ストリームID確認 |
| ERR_HTTP2_HEADERS_SENT | ヘッダーエラー | ヘッダー送信済み | 送信順序確認 |
| ERR_HTTP2_OUT_OF_STREAMS | リソースエラー | ストリームID枯渇 | 新規セッション作成 |
| ERR_HTTP2_GOAWAY_SESSION | セッション終了 | GOAWAY受信 | 新規セッション作成 |
| ERR_HTTP2_STATUS_101 | プロトコルエラー | 101ステータス使用 | ステータスコード変更 |

### リトライ仕様

GOAWAYやストリームエラー発生時は新規セッション/ストリームの作成を検討。

## トランザクション仕様

HTTP/2はステートレスプロトコル。ストリームレベルでの状態管理をサポート。

## パフォーマンス要件

- マルチプレキシングによる並列処理
- HPACKによるヘッダー圧縮
- フロー制御による輻輳回避
- 単一TCP接続での効率的なリソース利用

## セキュリティ考慮事項

- TLS 1.2以上の使用推奨（h2プロトコル）
- ALPN によるプロトコルネゴシエーション
- HPACK爆弾攻撃への対策
- ストリームID枯渇攻撃への対策

## 備考

- 平文HTTP/2（h2c）は createServer() で作成
- 暗号化HTTP/2（h2）は createSecureServer() で作成
- gRPCの基盤として広く利用される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | http2.js | `lib/http2.js` | エクスポートされるAPI一覧 |
| 1-2 | core.js | `lib/internal/http2/core.js` | 主要クラスと定数定義 |

**読解のコツ**: http2.jsはシンプルなラッパーで、実際の実装はinternal/http2/core.jsにある。Symbol定義（kSession, kState等）が内部状態管理の鍵。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | http2.js | `lib/http2.js` | 公開API定義 |

**主要処理フロー**:
1. **1-15行目**: internal/http2/coreからの再エクスポート
2. **17-29行目**: モジュールエクスポート

#### Step 3: コア実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | core.js | `lib/internal/http2/core.js` | Http2Session, Http2Streamクラス |

**主要処理フロー（core.js）**:
- **231-235行目**: 定数定義（kMaxFrameSize等）
- **245-266行目**: Symbol定義（kID, kState等）
- **268-279行目**: nghttp2バインディング定数
- **281-326行目**: HTTP/2定数（メソッド、ステータス等）
- **328-339行目**: ストリーム/セッションフラグ定義
- **351-467行目**: onSessionHeaders() - ヘッダー受信処理

### プログラム呼び出し階層図

```
http2.createSecureServer(options, onRequest)
    │
    └─ internal/http2/core.createSecureServer()
           │
           ├─ new Http2SecureServer(options)
           │      └─ tls.Server継承
           │
           └─ onRequest登録

http2.connect(authority, options)
    │
    └─ internal/http2/core.connect()
           │
           ├─ authority解析
           │
           ├─ TLS/TCP接続
           │      └─ tls.connect() / net.connect()
           │
           └─ new ClientHttp2Session()
                  │
                  ├─ SETTINGS送信
                  │
                  └─ 'connect'イベント
```

### データフロー図

```
[クライアント]              [サーバー]

connect()         ───▶   accept
    │                        │
SETTINGS          ◀───▶   SETTINGS
    │                        │
HEADERS (request) ───▶   onSessionHeaders()
    │                        │
DATA              ───▶   stream.on('data')
    │                        │
                  ◀───   HEADERS (response)
                  ◀───   DATA
    │                        │
END_STREAM        ◀───▶   END_STREAM
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| http2.js | `lib/http2.js` | ソース | 公開API定義 |
| core.js | `lib/internal/http2/core.js` | ソース | コア実装（セッション、ストリーム） |
| compat.js | `lib/internal/http2/compat.js` | ソース | HTTP/1.1互換レイヤー |
| util.js | `lib/internal/http2/util.js` | ソース | ユーティリティ関数 |
| tls.js | `lib/tls.js` | ソース | TLS基盤 |
| net.js | `lib/net.js` | ソース | TCP基盤 |
