# API設計書

## 概要

本ドキュメントは、OpenSearch の REST API エンドポイントの設計仕様を定義する。OpenSearch は分散型の検索・分析エンジンであり、HTTP REST API を通じてインデックス管理、ドキュメント操作、検索、クラスタ管理等の機能を提供する。本書では、REST API仕様定義（`rest-api-spec/`）およびJava実装コード（`server/src/main/java/org/opensearch/rest/action/`）に基づき、全166エンドポイントをカテゴリ別に記載する。

## 共通仕様

### ベースURL

```
http://{host}:9200
```

デフォルトポートは `9200`。HTTPS の場合はセキュリティプラグインの設定に依存する。

### 認証方式

OpenSearch のコアモジュールには組み込みの認証機構は存在しない。認証・認可は **プラグイン方式** で提供される。`ActionPlugin.getRestHandlerWrapper(ThreadContext)` を通じて、すべてのRESTハンドラにインターセプタを適用するアーキテクチャとなっている。

セキュリティプラグイン利用時は、以下の認証方式が一般的に使用される:

- HTTP Basic認証（`Authorization: Basic <credentials>`）
- Bearer Token認証
- API Key認証
- クライアント証明書認証（mutual TLS）

### 共通ヘッダー

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| Content-Type | ○（ボディ送信時） | `application/json`（NDJSON形式のAPIは `application/x-ndjson`） |
| Accept | - | `application/json`（デフォルト） |
| Authorization | -（セキュリティプラグイン使用時は○） | 認証情報 |

### 共通クエリパラメータ

すべてのAPIエンドポイントで以下の共通パラメータが使用可能（`_common.json` で定義）。

| パラメータ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| pretty | boolean | false | レスポンスJSONを整形して返す |
| human | boolean | true | 人間が読みやすい形式で統計値を返す |
| error_trace | boolean | false | エラー時にスタックトレースを含める |
| source | string | - | URLエンコードされたリクエスト定義（ボディの代替） |
| filter_path | list | - | レスポンスのフィルタリング（カンマ区切り） |

### 共通エラーレスポンス

| ステータスコード | 説明 |
| --- | --- |
| 400 | Bad Request - リクエストパラメータ不正 |
| 401 | Unauthorized - 認証エラー（セキュリティプラグイン使用時） |
| 403 | Forbidden - 権限エラー（セキュリティプラグイン使用時） |
| 404 | Not Found - リソース未検出 |
| 405 | Method Not Allowed - 許可されていないHTTPメソッド |
| 406 | Not Acceptable - サポートされていないメディアタイプ |
| 409 | Conflict - バージョン競合 |
| 429 | Too Many Requests - スロットリング |
| 500 | Internal Server Error - サーバーエラー |

エラーレスポンス形式:

```json
{
  "error": {
    "root_cause": [
      {
        "type": "resource_not_found_exception",
        "reason": "..."
      }
    ],
    "type": "resource_not_found_exception",
    "reason": "..."
  },
  "status": 404
}
```

## API一覧

### 基本情報

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| 基本情報 | `/` | GET | クラスタの基本情報を返す |
| 基本情報 | `/_cluster/health` | GET | クラスタのヘルスステータスを返す |
| 基本情報 | `/_cluster/state` | GET | クラスタの状態情報を返す |
| 基本情報 | `/_cluster/stats` | GET | クラスタの統計情報を返す |
| 基本情報 | `/_cluster/settings` | GET | クラスタ設定を取得 |
| 基本情報 | `/_cluster/settings` | PUT | クラスタ設定を更新 |

### ドキュメント操作

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| ドキュメント | `/{index}/_doc/{id}` | PUT, POST | ドキュメントを作成・更新 |
| ドキュメント | `/{index}/_doc` | POST | IDを自動生成してドキュメントを作成 |
| ドキュメント | `/{index}/_create/{id}` | PUT, POST | ドキュメントを新規作成（存在時エラー） |
| ドキュメント | `/{index}/_doc/{id}` | GET | ドキュメントを取得 |
| ドキュメント | `/{index}/_source/{id}` | GET | ドキュメントのソースのみ取得 |
| ドキュメント | `/{index}/_doc/{id}` | DELETE | ドキュメントを削除 |
| ドキュメント | `/{index}/_update/{id}` | POST | ドキュメントを部分更新 |
| ドキュメント | `/_bulk` | POST, PUT | 一括操作（index/update/delete） |
| ドキュメント | `/_mget` | GET, POST | 複数ドキュメントを一括取得 |
| ドキュメント | `/{index}/_delete_by_query` | POST | クエリに一致するドキュメントを削除 |
| ドキュメント | `/{index}/_update_by_query` | POST | クエリに一致するドキュメントを更新 |
| ドキュメント | `/_reindex` | POST | インデックス間でドキュメントをコピー |

### 検索

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| 検索 | `/_search` | GET, POST | ドキュメントを検索 |
| 検索 | `/{index}/_search` | GET, POST | 特定インデックスでドキュメントを検索 |
| 検索 | `/_msearch` | GET, POST | 複数の検索を一括実行 |
| 検索 | `/_count` | GET, POST | クエリに一致するドキュメント数を返す |
| 検索 | `/_search/scroll` | GET, POST | スクロール検索の続きを取得 |
| 検索 | `/_search/scroll` | DELETE | スクロールコンテキストをクリア |
| 検索 | `/_search_shards` | GET, POST | 検索対象のシャード情報を返す |
| 検索 | `/_search/template` | GET, POST | テンプレートを使用して検索 |
| 検索 | `/_msearch/template` | GET, POST | テンプレートを使用した一括検索 |
| 検索 | `/_render/template` | GET, POST | 検索テンプレートをレンダリング |
| 検索 | `/_field_caps` | GET, POST | フィールドの機能情報を返す |
| 検索 | `/{index}/_validate/query` | GET, POST | クエリの妥当性を検証 |
| 検索 | `/{index}/_explain/{id}` | GET, POST | ドキュメントのスコア計算を説明 |
| 検索 | `/_rank_eval` | GET, POST | 検索ランキングを評価 |
| 検索 | `/{index}/_termvectors/{id}` | GET, POST | タームベクトルを取得 |
| 検索 | `/_mtermvectors` | GET, POST | 複数のタームベクトルを取得 |

### Point in Time (PIT)

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| PIT | `/{index}/_search/point_in_time` | POST | PIT を作成 |
| PIT | `/_search/point_in_time` | DELETE | PIT を削除 |
| PIT | `/_search/point_in_time/_all` | DELETE | すべての PIT を削除 |
| PIT | `/_search/point_in_time/_all` | GET | すべての PIT を取得 |

### インデックス管理

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| インデックス | `/{index}` | PUT | インデックスを作成 |
| インデックス | `/{index}` | GET | インデックス情報を取得 |
| インデックス | `/{index}` | DELETE | インデックスを削除 |
| インデックス | `/{index}/_close` | POST | インデックスを閉じる |
| インデックス | `/{index}/_open` | POST | インデックスを開く |
| インデックス | `/{index}/_mapping` | PUT, POST | マッピングを更新 |
| インデックス | `/{index}/_mapping` | GET | マッピングを取得 |
| インデックス | `/{index}/_settings` | PUT | インデックス設定を更新 |
| インデックス | `/{index}/_settings` | GET | インデックス設定を取得 |
| インデックス | `/{index}/_alias/{name}` | PUT, POST | エイリアスを作成・更新 |
| インデックス | `/{index}/_alias/{name}` | GET | エイリアスを取得 |
| インデックス | `/{index}/_alias/{name}` | DELETE | エイリアスを削除 |
| インデックス | `/_aliases` | POST | 一括エイリアス操作 |
| インデックス | `/{index}/_refresh` | POST | インデックスをリフレッシュ |
| インデックス | `/{index}/_flush` | POST | インデックスをフラッシュ |
| インデックス | `/{index}/_forcemerge` | POST | セグメントを強制マージ |
| インデックス | `/{index}/_cache/clear` | POST | キャッシュをクリア |
| インデックス | `/{index}/_clone/{target}` | PUT, POST | インデックスをクローン |
| インデックス | `/{index}/_shrink/{target}` | PUT, POST | インデックスを縮小 |
| インデックス | `/{index}/_split/{target}` | PUT, POST | インデックスを分割 |
| インデックス | `/{index}/_rollover` | POST | インデックスをロールオーバー |
| インデックス | `/{index}/_block/{block}` | PUT | インデックスにブロックを追加 |
| インデックス | `/_resolve/index/{name}` | GET | インデックス名を解決 |
| インデックス | `/{index}/_stats` | GET | インデックス統計を取得 |
| インデックス | `/{index}/_segments` | GET | セグメント情報を取得 |
| インデックス | `/{index}/_shard_stores` | GET | シャードストア情報を取得 |
| インデックス | `/{index}/_recovery` | GET | リカバリ情報を取得 |
| インデックス | `/{index}/_upgrade` | GET | アップグレード状態を取得 |
| インデックス | `/{index}/_upgrade` | POST | インデックスをアップグレード |

### インデックステンプレート

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| テンプレート | `/_index_template/{name}` | PUT | インデックステンプレートを作成・更新 |
| テンプレート | `/_index_template/{name}` | GET | インデックステンプレートを取得 |
| テンプレート | `/_index_template/{name}` | DELETE | インデックステンプレートを削除 |
| テンプレート | `/_template/{name}` | PUT | レガシーテンプレートを作成・更新 |
| テンプレート | `/_template/{name}` | GET | レガシーテンプレートを取得 |
| テンプレート | `/_template/{name}` | DELETE | レガシーテンプレートを削除 |
| テンプレート | `/_component_template/{name}` | PUT | コンポーネントテンプレートを作成・更新 |
| テンプレート | `/_component_template/{name}` | GET | コンポーネントテンプレートを取得 |
| テンプレート | `/_component_template/{name}` | DELETE | コンポーネントテンプレートを削除 |
| テンプレート | `/_index_template/_simulate_index/{name}` | POST | インデックステンプレートのシミュレーション |
| テンプレート | `/_index_template/_simulate` | POST | テンプレートのシミュレーション |

### データストリーム

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| データストリーム | `/_data_stream/{name}` | PUT | データストリームを作成 |
| データストリーム | `/_data_stream/{name}` | GET | データストリーム情報を取得 |
| データストリーム | `/_data_stream/{name}` | DELETE | データストリームを削除 |
| データストリーム | `/_data_stream/_stats` | GET | データストリーム統計を取得 |

### クラスタ管理

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| クラスタ | `/_cluster/health` | GET | クラスタヘルスを取得 |
| クラスタ | `/_cluster/state` | GET | クラスタ状態を取得 |
| クラスタ | `/_cluster/stats` | GET | クラスタ統計を取得 |
| クラスタ | `/_cluster/settings` | GET | クラスタ設定を取得 |
| クラスタ | `/_cluster/settings` | PUT | クラスタ設定を更新 |
| クラスタ | `/_cluster/reroute` | POST | シャードの再配置 |
| クラスタ | `/_cluster/allocation/explain` | GET, POST | シャード割り当ての説明 |
| クラスタ | `/_cluster/pending_tasks` | GET | 保留中のタスクを取得 |
| クラスタ | `/_remote/info` | GET | リモートクラスタ情報を取得 |
| クラスタ | `/_cluster/voting_config_exclusions` | POST | 投票設定除外を追加 |
| クラスタ | `/_cluster/voting_config_exclusions` | DELETE | 投票設定除外を削除 |
| クラスタ | `/_cluster/decommission/awareness/{awareness_attribute_name}/{awareness_attribute_value}` | PUT | デコミッション設定 |
| クラスタ | `/_cluster/decommission/awareness` | GET | デコミッション状態を取得 |
| クラスタ | `/_cluster/decommission/awareness` | DELETE | デコミッション設定を削除 |
| クラスタ | `/_cluster/routing/awareness/{attribute}/weights` | GET | ルーティング重み設定を取得 |
| クラスタ | `/_cluster/routing/awareness/{attribute}/weights` | PUT | ルーティング重み設定を更新 |
| クラスタ | `/_cluster/routing/awareness/weights` | DELETE | ルーティング重み設定を削除 |

### ノード管理

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| ノード | `/_nodes` | GET | ノード情報を取得 |
| ノード | `/_nodes/{node_id}` | GET | 特定ノードの情報を取得 |
| ノード | `/_nodes/stats` | GET | ノード統計を取得 |
| ノード | `/_nodes/{node_id}/stats` | GET | 特定ノードの統計を取得 |
| ノード | `/_nodes/hot_threads` | GET | ホットスレッド情報を取得 |
| ノード | `/_nodes/usage` | GET | ノード使用状況を取得 |
| ノード | `/_nodes/reload_secure_settings` | POST | セキュア設定をリロード |

### CAT API

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| CAT | `/_cat/aliases` | GET | エイリアス一覧 |
| CAT | `/_cat/allocation` | GET | シャード割り当て情報 |
| CAT | `/_cat/cluster_manager` | GET | クラスタマネージャノード情報 |
| CAT | `/_cat/count` | GET | ドキュメント数 |
| CAT | `/_cat/fielddata` | GET | フィールドデータ情報 |
| CAT | `/_cat/health` | GET | クラスタヘルス情報 |
| CAT | `/_cat/help` | GET | 利用可能なCAT APIの一覧 |
| CAT | `/_cat/indices` | GET | インデックス情報 |
| CAT | `/_cat/nodeattrs` | GET | ノード属性情報 |
| CAT | `/_cat/nodes` | GET | ノード一覧 |
| CAT | `/_cat/pending_tasks` | GET | 保留タスク一覧 |
| CAT | `/_cat/plugins` | GET | プラグイン一覧 |
| CAT | `/_cat/recovery` | GET | リカバリ情報 |
| CAT | `/_cat/repositories` | GET | リポジトリ一覧 |
| CAT | `/_cat/segment_replication` | GET | セグメントレプリケーション情報 |
| CAT | `/_cat/segments` | GET | セグメント情報 |
| CAT | `/_cat/shards` | GET | シャード情報 |
| CAT | `/_cat/snapshots` | GET | スナップショット一覧 |
| CAT | `/_cat/tasks` | GET | タスク一覧 |
| CAT | `/_cat/templates` | GET | テンプレート一覧 |
| CAT | `/_cat/thread_pool` | GET | スレッドプール情報 |

### Ingest パイプライン

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Ingest | `/_ingest/pipeline/{id}` | PUT | パイプラインを作成・更新 |
| Ingest | `/_ingest/pipeline/{id}` | GET | パイプラインを取得 |
| Ingest | `/_ingest/pipeline/{id}` | DELETE | パイプラインを削除 |
| Ingest | `/_ingest/pipeline/_simulate` | GET, POST | パイプラインをシミュレーション |
| Ingest | `/_ingest/processor/grok` | GET | Grokパターンを取得 |

### スナップショット・リストア

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| スナップショット | `/_snapshot/{repository}` | PUT, POST | リポジトリを作成 |
| スナップショット | `/_snapshot/{repository}` | GET | リポジトリ情報を取得 |
| スナップショット | `/_snapshot/{repository}` | DELETE | リポジトリを削除 |
| スナップショット | `/_snapshot/{repository}/_verify` | POST | リポジトリを検証 |
| スナップショット | `/_snapshot/{repository}/_cleanup` | POST | リポジトリをクリーンアップ |
| スナップショット | `/_snapshot/{repository}/{snapshot}` | PUT, POST | スナップショットを作成 |
| スナップショット | `/_snapshot/{repository}/{snapshot}` | GET | スナップショット情報を取得 |
| スナップショット | `/_snapshot/{repository}/{snapshot}` | DELETE | スナップショットを削除 |
| スナップショット | `/_snapshot/{repository}/{snapshot}/_restore` | POST | スナップショットをリストア |
| スナップショット | `/_snapshot/{repository}/{snapshot}/_clone/{target_snapshot}` | PUT | スナップショットをクローン |
| スナップショット | `/_snapshot/_status` | GET | スナップショットのステータスを取得 |

### スクリプト

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| スクリプト | `/_scripts/{id}` | PUT, POST | ストアドスクリプトを保存 |
| スクリプト | `/_scripts/{id}` | GET | ストアドスクリプトを取得 |
| スクリプト | `/_scripts/{id}` | DELETE | ストアドスクリプトを削除 |
| スクリプト | `/_script_context` | GET | スクリプトコンテキストを取得 |
| スクリプト | `/_script_language` | GET | スクリプト言語を取得 |
| スクリプト | `/_scripts/painless/_execute` | GET, POST | Painlessスクリプトを実行 |

### タスク管理

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| タスク | `/_tasks` | GET | タスク一覧を取得 |
| タスク | `/_tasks/{task_id}` | GET | 特定タスクの情報を取得 |
| タスク | `/_tasks/_cancel` | POST | タスクをキャンセル |

### Dangling インデックス

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Dangling | `/_dangling` | GET | ダングリングインデックス一覧 |
| Dangling | `/_dangling/{index_uuid}` | POST | ダングリングインデックスをインポート |
| Dangling | `/_dangling/{index_uuid}` | DELETE | ダングリングインデックスを削除 |

### Search Pipeline

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Search Pipeline | `/_search/pipeline/{id}` | PUT | 検索パイプラインを作成・更新 |
| Search Pipeline | `/_search/pipeline/{id}` | GET | 検索パイプラインを取得 |
| Search Pipeline | `/_search/pipeline/{id}` | DELETE | 検索パイプラインを削除 |

### Remote Store

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Remote Store | `/_remotestore/_restore` | POST | リモートストアからリストア |
| Remote Store | `/_remotestore/stats` | GET | リモートストア統計を取得 |

### その他

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| その他 | `/` | GET | クラスタ基本情報 |
| その他 | `/` | HEAD | ノードの稼働確認（ping） |
| その他 | `/_delete_by_query/{task_id}/_rethrottle` | POST | Delete by query のスロットル変更 |
| その他 | `/_reindex/{task_id}/_rethrottle` | POST | Reindex のスロットル変更 |
| その他 | `/_update_by_query/{task_id}/_rethrottle` | POST | Update by query のスロットル変更 |

---

## 各APIエンドポイント定義

### 基本情報

#### 1. クラスタ情報取得

クラスタの基本情報（名前、バージョン等）を返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | なし |

**クエリパラメータ**

共通パラメータのみ。

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "name": "node-1",
  "cluster_name": "opensearch-cluster",
  "cluster_uuid": "xxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx",
  "version": {
    "distribution": "opensearch",
    "number": "2.x.x",
    "build_type": "tar",
    "build_hash": "xxxxxxx",
    "build_date": "2024-01-01T00:00:00.000000Z",
    "build_snapshot": false,
    "lucene_version": "9.x.x",
    "minimum_wire_compatibility_version": "7.10.0",
    "minimum_index_compatibility_version": "7.0.0"
  },
  "tagline": "The OpenSearch Project: https://opensearch.org/"
}
```

---

#### 2. クラスタヘルス

クラスタのヘルスステータス情報を返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /_cluster/health` |
| エンドポイント（インデックス指定） | `GET /_cluster/health/{index}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | cluster:monitor/health |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | list | - | 対象インデックス（カンマ区切り） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| level | enum | - | cluster | 詳細レベル（cluster, indices, shards, awareness_attributes） |
| local | boolean | - | false | ローカルノード情報のみ返す |
| timeout | time | - | - | 操作タイムアウト |
| wait_for_status | enum | - | - | 指定ステータスまで待機（green, yellow, red） |
| wait_for_nodes | string | - | - | 指定ノード数まで待機 |
| wait_for_active_shards | string | - | - | 指定数のアクティブシャードまで待機 |
| wait_for_no_relocating_shards | boolean | - | - | 移動中シャードが0になるまで待機 |
| wait_for_no_initializing_shards | boolean | - | - | 初期化中シャードが0になるまで待機 |
| wait_for_events | enum | - | - | 指定優先度のイベント処理完了まで待機 |
| expand_wildcards | enum | - | all | ワイルドカード展開の対象 |
| awareness_attribute | string | - | - | ヘルス確認対象のアウェアネス属性 |
| cluster_manager_timeout | time | - | - | クラスタマネージャ接続タイムアウト |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "cluster_name": "opensearch-cluster",
  "status": "green",
  "timed_out": false,
  "number_of_nodes": 3,
  "number_of_data_nodes": 3,
  "discovered_cluster_manager": true,
  "active_primary_shards": 5,
  "active_shards": 10,
  "relocating_shards": 0,
  "initializing_shards": 0,
  "unassigned_shards": 0,
  "delayed_unassigned_shards": 0,
  "number_of_pending_tasks": 0,
  "number_of_in_flight_fetch": 0,
  "task_max_waiting_in_queue_millis": 0,
  "active_shards_percent_as_number": 100.0
}
```

---

### ドキュメント操作

#### 3. ドキュメント作成・更新（Index）

ドキュメントをインデックスに作成または更新する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /{index}/_doc/{id}` / `POST /{index}/_doc/{id}` |
| エンドポイント（ID自動生成） | `POST /{index}/_doc` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/write/index |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | string | ○ | インデックス名 |
| id | string | - | ドキュメントID（省略時は自動生成） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| routing | string | - | - | ルーティング値 |
| timeout | time | - | - | 操作タイムアウト |
| version | number | - | - | 楽観的排他制御のバージョン番号 |
| version_type | enum | - | - | バージョンタイプ（internal, external, external_gte） |
| op_type | enum | - | - | 操作タイプ（index, create） |
| refresh | enum | - | false | リフレッシュポリシー（true, false, wait_for） |
| wait_for_active_shards | string | - | 1 | アクティブシャード待機数 |
| pipeline | string | - | - | Ingestパイプライン名 |
| if_seq_no | number | - | - | 楽観的排他制御のシーケンス番号 |
| if_primary_term | number | - | - | 楽観的排他制御のプライマリターム |
| require_alias | boolean | - | false | エイリアスのみを許可 |

**リクエストボディ**

```json
{
  "field1": "value1",
  "field2": 123,
  "field3": {
    "nested_field": "nested_value"
  }
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| （任意フィールド） | any | ○ | インデックスのマッピングに従ったドキュメントフィールド |

**レスポンス（成功時）**

ステータスコード: `201 Created`（新規） / `200 OK`（更新）

```json
{
  "_index": "my-index",
  "_id": "1",
  "_version": 1,
  "result": "created",
  "_shards": {
    "total": 2,
    "successful": 1,
    "failed": 0
  },
  "_seq_no": 0,
  "_primary_term": 1
}
```

---

#### 4. ドキュメント取得（Get）

指定したドキュメントを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /{index}/_doc/{id}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/read/get |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | string | ○ | インデックス名 |
| id | string | ○ | ドキュメントID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| stored_fields | list | - | - | 返却するストアドフィールド |
| preference | string | - | random | ノード/シャード指定 |
| realtime | boolean | - | - | リアルタイムモード |
| refresh | boolean | - | - | 取得前にリフレッシュ |
| routing | string | - | - | ルーティング値 |
| _source | list | - | - | _sourceフィールドの制御 |
| _source_excludes | list | - | - | _sourceから除外するフィールド |
| _source_includes | list | - | - | _sourceに含めるフィールド |
| version | number | - | - | バージョン指定 |
| version_type | enum | - | - | バージョンタイプ |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "_index": "my-index",
  "_id": "1",
  "_version": 1,
  "_seq_no": 0,
  "_primary_term": 1,
  "found": true,
  "_source": {
    "field1": "value1",
    "field2": 123
  }
}
```

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

```json
{
  "_index": "my-index",
  "_id": "999",
  "found": false
}
```

---

#### 5. ドキュメント削除（Delete）

指定したドキュメントを削除する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /{index}/_doc/{id}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/write/delete |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | string | ○ | インデックス名 |
| id | string | ○ | ドキュメントID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| routing | string | - | - | ルーティング値 |
| timeout | time | - | - | 操作タイムアウト |
| version | number | - | - | 楽観的排他制御のバージョン番号 |
| version_type | enum | - | - | バージョンタイプ（internal, external, external_gte, force） |
| refresh | enum | - | false | リフレッシュポリシー（true, false, wait_for） |
| wait_for_active_shards | string | - | 1 | アクティブシャード待機数 |
| if_seq_no | number | - | - | 楽観的排他制御のシーケンス番号 |
| if_primary_term | number | - | - | 楽観的排他制御のプライマリターム |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "_index": "my-index",
  "_id": "1",
  "_version": 2,
  "result": "deleted",
  "_shards": {
    "total": 2,
    "successful": 1,
    "failed": 0
  },
  "_seq_no": 1,
  "_primary_term": 1
}
```

---

#### 6. ドキュメント更新（Update）

既存ドキュメントをスクリプトまたは部分ドキュメントで更新する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /{index}/_update/{id}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/write/update |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | string | ○ | インデックス名 |
| id | string | ○ | ドキュメントID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| routing | string | - | - | ルーティング値 |
| timeout | time | - | - | 操作タイムアウト |
| refresh | enum | - | false | リフレッシュポリシー（true, false, wait_for） |
| wait_for_active_shards | string | - | 1 | アクティブシャード待機数 |
| retry_on_conflict | number | - | 0 | 競合発生時のリトライ回数 |
| _source | list | - | - | _sourceフィールドの制御 |
| _source_excludes | list | - | - | _sourceから除外するフィールド |
| _source_includes | list | - | - | _sourceに含めるフィールド |
| lang | string | - | painless | スクリプト言語 |
| if_seq_no | number | - | - | 楽観的排他制御のシーケンス番号 |
| if_primary_term | number | - | - | 楽観的排他制御のプライマリターム |
| require_alias | boolean | - | false | エイリアスのみを許可 |

**リクエストボディ**

部分更新の場合:

```json
{
  "doc": {
    "field1": "new_value"
  }
}
```

スクリプト更新の場合:

```json
{
  "script": {
    "source": "ctx._source.counter += params.count",
    "lang": "painless",
    "params": {
      "count": 4
    }
  }
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| doc | object | -（`script`と排他） | 部分ドキュメント |
| script | object | -（`doc`と排他） | 更新スクリプト |
| doc_as_upsert | boolean | - | doc が存在しない場合に upsert するか |
| upsert | object | - | ドキュメント不存在時の初期値 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "_index": "my-index",
  "_id": "1",
  "_version": 2,
  "result": "updated",
  "_shards": {
    "total": 2,
    "successful": 1,
    "failed": 0
  },
  "_seq_no": 1,
  "_primary_term": 1
}
```

---

#### 7. 一括操作（Bulk）

複数のindex/update/delete操作を1リクエストで実行する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /_bulk` / `POST /{index}/_bulk` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/write/bulk |
| Content-Type | `application/x-ndjson` |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | string | - | デフォルトインデックス名 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| routing | string | - | - | ルーティング値 |
| timeout | time | - | - | 操作タイムアウト |
| refresh | enum | - | false | リフレッシュポリシー |
| wait_for_active_shards | string | - | 1 | アクティブシャード待機数 |
| pipeline | string | - | - | Ingestパイプライン名 |
| _source | list | - | - | _sourceフィールドの制御 |
| _source_excludes | list | - | - | _sourceから除外するフィールド |
| _source_includes | list | - | - | _sourceに含めるフィールド |
| require_alias | boolean | - | false | エイリアスのみを許可 |
| batch_size | int | - | - | バッチサイズ |

**リクエストボディ**

NDJSON形式（各行が改行で区切られる）:

```
{"index":{"_index":"my-index","_id":"1"}}
{"field1":"value1"}
{"delete":{"_index":"my-index","_id":"2"}}
{"update":{"_index":"my-index","_id":"3"}}
{"doc":{"field1":"updated_value"}}
```

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "took": 30,
  "errors": false,
  "items": [
    {
      "index": {
        "_index": "my-index",
        "_id": "1",
        "_version": 1,
        "result": "created",
        "status": 201
      }
    },
    {
      "delete": {
        "_index": "my-index",
        "_id": "2",
        "_version": 2,
        "result": "deleted",
        "status": 200
      }
    }
  ]
}
```

---

#### 8. 複数ドキュメント取得（Multi Get）

複数のドキュメントを一度に取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /_mget` / `POST /_mget` |
| エンドポイント（インデックス指定） | `GET /{index}/_mget` / `POST /{index}/_mget` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/read/mget |

**リクエストボディ**

```json
{
  "docs": [
    {
      "_index": "my-index",
      "_id": "1"
    },
    {
      "_index": "my-index",
      "_id": "2"
    }
  ]
}
```

インデックスがURLで指定されている場合:

```json
{
  "ids": ["1", "2", "3"]
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "docs": [
    {
      "_index": "my-index",
      "_id": "1",
      "_version": 1,
      "found": true,
      "_source": {
        "field1": "value1"
      }
    },
    {
      "_index": "my-index",
      "_id": "2",
      "found": false
    }
  ]
}
```

---

#### 9. クエリによる削除（Delete by Query）

クエリに一致するすべてのドキュメントを削除する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /{index}/_delete_by_query` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/write/delete/byquery |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | list | ○ | 対象インデックス（カンマ区切り、`_all` で全インデックス） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| conflicts | enum | - | abort | 競合時の動作（abort, proceed） |
| refresh | boolean | - | - | 操作後にリフレッシュ |
| timeout | time | - | 1m | バルクリクエストごとのタイムアウト |
| wait_for_completion | boolean | - | true | 完了まで待機 |
| requests_per_second | number | - | 0 | スロットリング（-1で無制限） |
| slices | number/string | - | 1 | タスク分割数（`auto`可） |
| max_docs | number | - | - | 処理最大ドキュメント数 |
| scroll | time | - | - | スクロールコンテキスト保持時間 |
| scroll_size | number | - | 100 | スクロールサイズ |
| search_type | enum | - | - | 検索タイプ |

**リクエストボディ**

```json
{
  "query": {
    "match": {
      "field1": "value1"
    }
  }
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| query | object | ○ | Query DSL による検索条件 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "took": 147,
  "timed_out": false,
  "total": 5,
  "deleted": 5,
  "batches": 1,
  "version_conflicts": 0,
  "noops": 0,
  "retries": {
    "bulk": 0,
    "search": 0
  }
}
```

---

#### 10. リインデックス（Reindex）

インデックス間でドキュメントをコピーする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /_reindex` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/write/reindex |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| refresh | boolean | - | - | 操作後にリフレッシュ |
| timeout | time | - | 1m | タイムアウト |
| wait_for_completion | boolean | - | true | 完了まで待機 |
| wait_for_active_shards | string | - | 1 | アクティブシャード待機数 |
| requests_per_second | number | - | 0 | スロットリング |
| slices | number/string | - | 1 | タスク分割数 |
| scroll | time | - | 5m | スクロールコンテキスト保持時間 |
| max_docs | number | - | - | 処理最大ドキュメント数 |

**リクエストボディ**

```json
{
  "source": {
    "index": "source-index",
    "query": {
      "match_all": {}
    }
  },
  "dest": {
    "index": "dest-index"
  }
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| source | object | ○ | ソースインデックス定義 |
| source.index | string | ○ | ソースインデックス名 |
| source.query | object | - | フィルタリング用クエリ |
| dest | object | ○ | デスティネーションインデックス定義 |
| dest.index | string | ○ | デスティネーションインデックス名 |
| dest.pipeline | string | - | Ingestパイプライン名 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "took": 639,
  "timed_out": false,
  "total": 1000,
  "updated": 0,
  "created": 1000,
  "deleted": 0,
  "batches": 1,
  "version_conflicts": 0,
  "noops": 0,
  "retries": {
    "bulk": 0,
    "search": 0
  }
}
```

---

### 検索

#### 11. 検索（Search）

Query DSLを使用してドキュメントを検索する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /_search` / `POST /_search` |
| エンドポイント（インデックス指定） | `GET /{index}/_search` / `POST /{index}/_search` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/read/search |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | list | - | 対象インデックス（カンマ区切り、`_all` で全インデックス） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| q | string | - | - | Lucene形式のクエリ文字列 |
| analyzer | string | - | - | クエリ文字列のアナライザ |
| analyze_wildcard | boolean | - | false | ワイルドカード・プレフィックスクエリをアナライズ |
| default_operator | enum | - | OR | クエリ文字列のデフォルト演算子（AND, OR） |
| df | string | - | - | デフォルトフィールド |
| from | number | - | 0 | 開始オフセット |
| size | number | - | 10 | 返却件数 |
| sort | list | - | - | ソート条件（`field:direction` のカンマ区切り） |
| _source | list | - | - | _sourceフィールドの制御 |
| _source_excludes | list | - | - | _sourceから除外するフィールド |
| _source_includes | list | - | - | _sourceに含めるフィールド |
| stored_fields | list | - | - | 返却するストアドフィールド |
| docvalue_fields | list | - | - | docvalue形式で返すフィールド |
| explain | boolean | - | - | スコア計算の詳細情報 |
| search_type | enum | - | - | 検索タイプ（query_then_fetch, dfs_query_then_fetch） |
| scroll | time | - | - | スクロール検索コンテキスト保持時間 |
| timeout | time | - | - | 操作タイムアウト |
| terminate_after | number | - | - | シャードごとの最大収集ドキュメント数 |
| track_scores | boolean | - | - | スコア計算の強制 |
| track_total_hits | boolean | - | - | ヒット総数の正確な追跡 |
| allow_partial_search_results | boolean | - | true | 部分的な検索結果を許可 |
| preference | string | - | random | ノード/シャード指定 |
| routing | list | - | - | ルーティング値 |
| request_cache | boolean | - | - | リクエストキャッシュの使用 |
| batched_reduce_size | number | - | 512 | コーディネータノードでの一括リデュースサイズ |
| max_concurrent_shard_requests | number | - | 5 | ノードあたりの並行シャードリクエスト数 |
| search_pipeline | string | - | - | 使用する検索パイプライン |
| ccs_minimize_roundtrips | boolean | - | true | CCSのラウンドトリップ最小化 |

**リクエストボディ**

```json
{
  "query": {
    "bool": {
      "must": [
        {
          "match": {
            "title": "opensearch"
          }
        }
      ],
      "filter": [
        {
          "range": {
            "date": {
              "gte": "2024-01-01"
            }
          }
        }
      ]
    }
  },
  "from": 0,
  "size": 10,
  "sort": [
    {"date": {"order": "desc"}}
  ],
  "_source": ["title", "date", "author"],
  "aggs": {
    "by_author": {
      "terms": {
        "field": "author.keyword"
      }
    }
  }
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "took": 5,
  "timed_out": false,
  "_shards": {
    "total": 5,
    "successful": 5,
    "skipped": 0,
    "failed": 0
  },
  "hits": {
    "total": {
      "value": 100,
      "relation": "eq"
    },
    "max_score": 1.0,
    "hits": [
      {
        "_index": "my-index",
        "_id": "1",
        "_score": 1.0,
        "_source": {
          "title": "opensearch guide",
          "date": "2024-01-15",
          "author": "John"
        }
      }
    ]
  },
  "aggregations": {
    "by_author": {
      "buckets": [
        {
          "key": "John",
          "doc_count": 50
        }
      ]
    }
  }
}
```

---

#### 12. 複数検索（Multi Search）

複数の検索リクエストを一度に実行する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /_msearch` / `POST /_msearch` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/read/msearch |
| Content-Type | `application/x-ndjson` |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| search_type | enum | - | - | 検索タイプ |
| max_concurrent_searches | number | - | - | 最大並行検索数 |
| max_concurrent_shard_requests | number | - | 5 | ノードあたりの並行シャードリクエスト数 |

**リクエストボディ**

NDJSON形式:

```
{"index": "my-index-1"}
{"query": {"match": {"title": "opensearch"}}}
{"index": "my-index-2"}
{"query": {"match_all": {}}}
```

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "took": 10,
  "responses": [
    {
      "took": 5,
      "timed_out": false,
      "hits": {
        "total": {"value": 10, "relation": "eq"},
        "hits": [...]
      },
      "status": 200
    },
    {
      "took": 3,
      "timed_out": false,
      "hits": {
        "total": {"value": 100, "relation": "eq"},
        "hits": [...]
      },
      "status": 200
    }
  ]
}
```

---

#### 13. カウント（Count）

クエリに一致するドキュメント数を返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /_count` / `POST /_count` |
| エンドポイント（インデックス指定） | `GET /{index}/_count` / `POST /{index}/_count` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:data/read/search |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| q | string | - | - | Lucene形式のクエリ文字列 |
| analyzer | string | - | - | クエリ文字列のアナライザ |
| default_operator | enum | - | OR | デフォルト演算子 |
| df | string | - | - | デフォルトフィールド |
| min_score | number | - | - | 最小スコア |
| routing | list | - | - | ルーティング値 |
| terminate_after | number | - | - | シャードごとの最大収集ドキュメント数 |
| preference | string | - | random | ノード/シャード指定 |

**リクエストボディ**

```json
{
  "query": {
    "match": {
      "title": "opensearch"
    }
  }
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "count": 100,
  "_shards": {
    "total": 5,
    "successful": 5,
    "skipped": 0,
    "failed": 0
  }
}
```

---

### インデックス管理

#### 14. インデックス作成

オプションの設定とマッピングを含むインデックスを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /{index}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:admin/create |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | string | ○ | インデックス名 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| wait_for_active_shards | string | - | - | アクティブシャード待機数 |
| timeout | time | - | - | 操作タイムアウト |
| cluster_manager_timeout | time | - | - | クラスタマネージャ接続タイムアウト |

**リクエストボディ**

```json
{
  "settings": {
    "number_of_shards": 3,
    "number_of_replicas": 1
  },
  "mappings": {
    "properties": {
      "title": {
        "type": "text",
        "analyzer": "standard"
      },
      "date": {
        "type": "date"
      },
      "count": {
        "type": "integer"
      }
    }
  },
  "aliases": {
    "my-alias": {}
  }
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| settings | object | - | インデックス設定（シャード数、レプリカ数等） |
| mappings | object | - | フィールドマッピング定義 |
| aliases | object | - | エイリアス定義 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "acknowledged": true,
  "shards_acknowledged": true,
  "index": "my-index"
}
```

---

#### 15. インデックス取得

インデックスの情報（設定、マッピング、エイリアス）を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /{index}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:admin/get |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | list | ○ | インデックス名（カンマ区切り） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| local | boolean | - | false | ローカルノード情報のみ |
| ignore_unavailable | boolean | - | false | 利用不可インデックスを無視 |
| allow_no_indices | boolean | - | false | インデックスなしの場合エラーにしない |
| expand_wildcards | enum | - | open | ワイルドカード展開対象 |
| flat_settings | boolean | - | false | フラット形式で設定を返す |
| include_defaults | boolean | - | false | デフォルト設定を含める |
| cluster_manager_timeout | time | - | - | クラスタマネージャ接続タイムアウト |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "my-index": {
    "aliases": {},
    "mappings": {
      "properties": {
        "title": {"type": "text"},
        "date": {"type": "date"}
      }
    },
    "settings": {
      "index": {
        "number_of_shards": "3",
        "number_of_replicas": "1",
        "creation_date": "1700000000000",
        "uuid": "xxxxxx",
        "version": {"created": "..."}
      }
    }
  }
}
```

---

#### 16. インデックス削除

指定されたインデックスを削除する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /{index}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:admin/delete |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | list | ○ | インデックス名（カンマ区切り、`_all`/`*` で全インデックス） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| timeout | time | - | - | 操作タイムアウト |
| master_timeout | time | - | - | マスターノード接続タイムアウト |
| ignore_unavailable | boolean | - | false | 利用不可インデックスを無視 |
| allow_no_indices | boolean | - | false | インデックスなしの場合エラーにしない |
| expand_wildcards | enum | - | open | ワイルドカード展開対象 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "acknowledged": true
}
```

---

#### 17. マッピング更新

インデックスのマッピングを更新する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /{index}/_mapping` / `POST /{index}/_mapping` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:admin/mapping/put |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| index | list | ○ | インデックス名（カンマ区切り、ワイルドカード対応） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| timeout | time | - | - | 操作タイムアウト |
| cluster_manager_timeout | time | - | - | クラスタマネージャ接続タイムアウト |
| ignore_unavailable | boolean | - | - | 利用不可インデックスを無視 |
| allow_no_indices | boolean | - | - | インデックスなしの場合エラーにしない |
| expand_wildcards | enum | - | open | ワイルドカード展開対象 |
| write_index_only | boolean | - | false | 書き込みインデックスのみに適用 |

**リクエストボディ**

```json
{
  "properties": {
    "new_field": {
      "type": "keyword"
    },
    "another_field": {
      "type": "text",
      "analyzer": "standard"
    }
  }
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| properties | object | ○ | フィールドマッピング定義 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "acknowledged": true
}
```

---

### ノード管理

#### 18. ノード情報取得

クラスタ内のノード情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /_nodes` |
| エンドポイント（ノード指定） | `GET /_nodes/{node_id}` |
| エンドポイント（メトリクス指定） | `GET /_nodes/{node_id}/{metric}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | cluster:monitor/nodes/info |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| node_id | list | - | ノードID/名前（カンマ区切り、`_local` で接続ノード） |
| metric | list | - | メトリクス（settings, os, process, jvm, thread_pool, transport, http, plugins, ingest） |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| flat_settings | boolean | - | false | フラット形式で設定を返す |
| timeout | time | - | - | 操作タイムアウト |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "_nodes": {
    "total": 3,
    "successful": 3,
    "failed": 0
  },
  "cluster_name": "opensearch-cluster",
  "nodes": {
    "node-id-1": {
      "name": "node-1",
      "transport_address": "10.0.0.1:9300",
      "host": "10.0.0.1",
      "ip": "10.0.0.1",
      "version": "2.x.x",
      "roles": ["cluster_manager", "data", "ingest"],
      "os": {
        "name": "Linux",
        "arch": "amd64",
        "available_processors": 8
      },
      "jvm": {
        "version": "17.0.x",
        "vm_name": "OpenJDK 64-Bit Server VM",
        "mem": {
          "heap_init_in_bytes": 1073741824,
          "heap_max_in_bytes": 1073741824
        }
      }
    }
  }
}
```

---

### Ingest パイプライン

#### 19. パイプライン作成・更新

Ingestパイプラインを作成または更新する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /_ingest/pipeline/{id}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | cluster:admin/ingest/pipeline/put |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | string | ○ | パイプラインID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| cluster_manager_timeout | time | - | - | クラスタマネージャ接続タイムアウト |
| timeout | time | - | - | 操作タイムアウト |

**リクエストボディ**

```json
{
  "description": "My ingest pipeline",
  "processors": [
    {
      "set": {
        "field": "processed",
        "value": true
      }
    },
    {
      "rename": {
        "field": "old_field",
        "target_field": "new_field"
      }
    }
  ]
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| description | string | - | パイプラインの説明 |
| processors | array | ○ | プロセッサの配列 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "acknowledged": true
}
```

---

### スナップショット

#### 20. スナップショット作成

リポジトリにスナップショットを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /_snapshot/{repository}/{snapshot}` / `POST /_snapshot/{repository}/{snapshot}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | cluster:admin/snapshot/create |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| repository | string | ○ | リポジトリ名 |
| snapshot | string | ○ | スナップショット名 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| cluster_manager_timeout | time | - | - | クラスタマネージャ接続タイムアウト |
| wait_for_completion | boolean | - | false | 完了まで待機 |

**リクエストボディ**

```json
{
  "indices": "index-1,index-2",
  "ignore_unavailable": true,
  "include_global_state": false,
  "partial": false
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| indices | string | - | 対象インデックス（カンマ区切り） |
| ignore_unavailable | boolean | - | 利用不可インデックスを無視 |
| include_global_state | boolean | - | グローバル状態を含める |
| partial | boolean | - | 部分的なスナップショットを許可 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "accepted": true
}
```

wait_for_completion=true の場合:

```json
{
  "snapshot": {
    "snapshot": "my-snapshot",
    "uuid": "xxxxxx",
    "version_id": 2000099,
    "version": "2.x.x",
    "indices": ["index-1", "index-2"],
    "data_streams": [],
    "include_global_state": false,
    "state": "SUCCESS",
    "start_time": "2024-01-01T00:00:00.000Z",
    "end_time": "2024-01-01T00:01:00.000Z",
    "duration_in_millis": 60000,
    "shards": {
      "total": 10,
      "failed": 0,
      "successful": 10
    }
  }
}
```

---

### CAT API

#### 21. CAT Indices

インデックスの一覧と統計情報をテーブル形式で返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /_cat/indices` |
| エンドポイント（インデックス指定） | `GET /_cat/indices/{index}` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | indices:monitor/stats, cluster:monitor/health |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| format | string | - | - | レスポンス形式（json, yaml, text等） |
| bytes | enum | - | - | バイト値の表示単位（b, k, kb, m, mb, g, gb, t, tb, p, pb） |
| h | list | - | - | 表示するカラム名（カンマ区切り） |
| health | enum | - | - | ヘルスステータスでフィルタ（green, yellow, red） |
| help | boolean | - | false | ヘルプ情報を表示 |
| pri | boolean | - | false | プライマリシャードのみの統計 |
| s | list | - | - | ソートカラム（カンマ区切り） |
| v | boolean | - | false | カラムヘッダーを表示 |
| expand_wildcards | enum | - | all | ワイルドカード展開対象 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

`format=json` の場合:

```json
[
  {
    "health": "green",
    "status": "open",
    "index": "my-index",
    "uuid": "xxxxxx",
    "pri": "3",
    "rep": "1",
    "docs.count": "1000",
    "docs.deleted": "10",
    "store.size": "50mb",
    "pri.store.size": "25mb"
  }
]
```

---

### タスク管理

#### 22. タスク一覧

クラスタ内のタスク一覧を返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /_tasks` |
| 認証 | セキュリティプラグイン使用時は必要 |
| 権限 | cluster:monitor/tasks/list |
| 安定性 | experimental |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| nodes | list | - | - | ノードIDでフィルタ |
| actions | list | - | - | アクション名でフィルタ |
| detailed | boolean | - | false | 詳細情報を含める |
| parent_task_id | string | - | - | 親タスクIDでフィルタ |
| wait_for_completion | boolean | - | false | 完了まで待機 |
| group_by | enum | - | nodes | グループ化方法（nodes, parents, none） |
| timeout | time | - | - | 操作タイムアウト |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "nodes": {
    "node-id-1": {
      "name": "node-1",
      "tasks": {
        "node-id-1:12345": {
          "node": "node-id-1",
          "id": 12345,
          "type": "transport",
          "action": "indices:data/write/reindex",
          "start_time_in_millis": 1700000000000,
          "running_time_in_nanos": 1000000000,
          "cancellable": true
        }
      }
    }
  }
}
```

---

## 備考

### アーキテクチャ概要

OpenSearch のREST APIレイヤは以下の多層構造で構成されている:

1. **HTTPレイヤ**: Netty4ベースの `HttpServerTransport` がHTTPリクエストを受信
2. **RESTレイヤ**: `RestController` がURLパスとHTTPメソッドに基づいてリクエストをルーティング
3. **RESTハンドラ**: `BaseRestHandler` を継承した各ハンドラがリクエストを解析し、アクションリクエストに変換
4. **Transportアクション**: `TransportAction` が実際のビジネスロジックを実行
5. **レスポンス**: 結果が `RestChannel` を通じてHTTPレスポンスとしてシリアライズ

### プラグインによるAPI拡張

プラグインは `ActionPlugin` インタフェースの `getRestHandlers()` メソッドを実装することで、カスタムREST APIエンドポイントを追加できる。登録は `ActionModule` で行われる。

### API仕様の管理

すべてのREST API仕様は `rest-api-spec/src/main/resources/rest-api-spec/api/` ディレクトリに166個のJSONファイルとして管理されており、テスト自動化やクライアントライブラリ生成に使用されている。

### 非推奨パラメータ

`master_timeout` パラメータは OpenSearch 2.0.0 で非推奨となり、`cluster_manager_timeout` に置き換えられた。これはインクルーシブな言語の推進に基づく変更である。
