# セキュリティ設計書

## 概要

本ドキュメントは、OpenSearchプロジェクトにおけるセキュリティ設計を記述する。OpenSearchのセキュリティアーキテクチャは**プラグインベース**の拡張モデルを採用しており、コア（本リポジトリ）はセキュリティのフレームワークとインターフェースを提供し、具体的な認証・認可の実装はセキュリティプラグインに委譲する設計となっている。本書ではコアに実装されているセキュリティ機構を中心に記載する。

## 認証設計

### 認証方式

OpenSearchコアはプラグイン可能なアイデンティティフレームワークを提供する。

| 認証方式 | 実装クラス | 説明 |
| --- | --- | --- |
| HTTP Basic認証 | `BasicAuthToken` (`server/src/main/java/org/opensearch/identity/tokens/BasicAuthToken.java`) | Base64エンコードされた `username:password` 形式の認証。Authorizationヘッダーから `Basic` プレフィクスで識別 |
| Bearer Token（JWT） | `BearerAuthToken` (`server/src/main/java/org/opensearch/identity/tokens/BearerAuthToken.java`) | JWT形式のBearer Token認証。`header.payload.signature` の3部構成をパースし検証 |
| On-Behalf-Of Token | `TokenManager.issueOnBehalfOfToken()` | 委任トークン。デフォルト有効期限300秒。`OnBehalfOfClaims` でaudience・subject・expirationを管理 |
| Service Account Token | `TokenManager.issueServiceAccountToken()` | サービスアカウント用トークン |

**アイデンティティサービスの構成:**

- `IdentityService` (`server/src/main/java/org/opensearch/identity/IdentityService.java`) — 認証プラグインを管理する中央サービス
- `IdentityPlugin` (`server/src/main/java/org/opensearch/identity/IdentityPlugin.java`) — 認証プロバイダのプラグインインターフェース
- `Subject` (`server/src/main/java/org/opensearch/identity/Subject.java`) — 認証済みエンティティ（ユーザー/プロセス/デバイス）を表現
- `NoopIdentityPlugin` (`server/src/main/java/org/opensearch/identity/noop/NoopIdentityPlugin.java`) — セキュリティプラグイン未導入時のデフォルトNo-Op実装

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション方式 | トークンベース（ステートレス） | HTTPリクエストごとにAuthorizationヘッダーで認証情報を送信 |
| On-Behalf-Of Tokenデフォルト有効期限 | 300秒（5分） | `OnBehalfOfClaims` で設定可能 |
| セッション固定化対策 | トークンベースのため該当なし | サーバー側セッション状態を持たないステートレス設計 |
| セキュリティコンテキスト伝播 | `ThreadContext` | リクエストに紐づく認証情報をスレッド間で自動伝播 |

## 認可設計

### 権限体系

OpenSearchコアはアクションフィルターフレームワークを通じて認可の拡張ポイントを提供する。具体的なRBAC（ロールベースアクセス制御）の実装はセキュリティプラグインに委譲される。

| ロール | 権限 | 説明 |
| --- | --- | --- |
| （プラグイン定義） | （プラグイン定義） | コアにはビルトインのロール定義は存在しない。セキュリティプラグインがRBAC、ドキュメントレベルセキュリティ、フィールドレベルセキュリティを提供 |

### アクセス制御

**アクションフィルター方式:**

コアは `ActionFilter` インターフェース (`server/src/main/java/org/opensearch/action/support/ActionFilter.java`) を通じて、すべてのトランスポートアクションに対するインターセプト機構を提供する。

- `ActionFilter.apply()` — リクエスト処理前にフィルターを適用
- `ActionFilter.order()` — フィルターの実行順序を制御
- `ActionFilters` — 複数のフィルターをチェーン形式で管理
- `ActionFilterChain` — フィルターの連鎖実行を制御

**リクエストバリデーション:**

- `RequestValidators` (`server/src/main/java/org/opensearch/action/RequestValidators.java`) — リクエスト実行前のバリデーションフレームワーク。ClusterStateとインデックス情報に基づいてリクエストの妥当性を検証
- `ActionPlugin.getRestHandlerWrapper()` — RESTハンドラーのラッピングによる認証・認可チェックの挿入ポイント

**Javaセキュリティマネージャー:**

- `Security` (`server/src/main/java/org/opensearch/bootstrap/Security.java`) — JVMレベルのセキュリティポリシーを初期化
- ファイルシステム権限、ソケット権限、プラグイン固有の権限をコードベースに基づいて制御
- `SpecialPermission` — 特権コードセクションのマーキング
- プラグインごとに `plugin-security.policy` でサンドボックス化された権限を定義

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | `SecureHttpTransportSettingsProvider` インターフェースでTLS/SSL対応。プラグインがSSLEngineを提供し、HTTPレイヤーの暗号化を実現 |
| Transport Layer TLS | `SecureTransportSettingsProvider` (`server/src/main/java/org/opensearch/plugins/SecureTransportSettingsProvider.java`) でノード間通信のTLS/SSLを提供。`KeyManagerFactory`・`TrustManagerFactory`・プロトコルバージョン・暗号スイートを設定可能 |
| ホスト名検証 | `NetworkModule.TRANSPORT_SSL_ENFORCE_HOSTNAME_VERIFICATION` — デフォルト `true`。TLS接続時にホスト名検証を強制 |
| デュアルモード | `NetworkModule.TRANSPORT_SSL_DUAL_MODE_ENABLED` — デフォルト `false`。有効化するとTLSと非TLSの両方の接続を許可（移行期間用） |
| クライアント認証 | `SecureTransportSettingsProvider` で相互TLS（mTLS）のクライアント認証レベルを設定可能 |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信（HTTP） | TLS/SSL（プラグイン提供のSSLEngine経由） |
| 通信（ノード間） | TLS/SSL（Transport Layer Security、プロトコル・暗号スイート設定可能） |
| 保存データ（Keystore） | AES-GCM暗号化。鍵導出にPBKDF2WithHmacSHA512（10,000イテレーション）を使用 (`KeyStoreWrapper`) |
| 保存データ（KMS連携） | AWS KMS統合による鍵管理 (`plugins/crypto-kms/CryptoKmsPlugin.java`) |

### 機密情報管理

OpenSearchは暗号化キーストア (`opensearch.keystore`) を用いて機密設定値を管理する。

- `KeyStoreWrapper` (`server/src/main/java/org/opensearch/common/settings/KeyStoreWrapper.java`) — AES-GCM暗号化によるキーストア実装
  - パスワード保護（PBKDF2WithHmacSHA512、10,000イテレーション）
  - 資格情報、APIキー等の機密値を格納
  - SHA-256ダイジェストによる検証機能
- `SecureSettings` / `SecureSetting` — 暗号化された設定値へのアクセスインターフェース
  - `getString()` — セキュアな文字列の取得
  - `getFile()` — セキュアなファイルの取得
  - `getSHA256Digest()` — 検証用ダイジェストの取得
- `NodesReloadSecureSettingsAction` — ノード再起動なしでのセキュア設定のホットリロード対応

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | コアでの明示的なXSSフィルターは実装されていない。REST API提供のためWebフロントエンドを直接持たないアーキテクチャ。XSS対策はフロントエンド（OpenSearch Dashboards）およびセキュリティプラグインに委譲 |
| SQLインジェクション | OpenSearchはSQLデータベースを使用しないため直接の該当なし。クエリDSL（JSON）ベースのアクセスモデルを採用。リクエストバリデーションは `RequestValidators` フレームワークおよび各RESTハンドラーで実施 |
| CSRF | コアにはCSRFトークン機構は未実装。セキュリティプラグインに委譲。REST APIのステートレス設計によりCSRFリスクは限定的 |
| リクエストID検証 | `RequestUtils.validateRequestId()` — X-Request-Idが32文字の16進数であることを検証 |
| Content-Type強制 | `SETTING_HTTP_CONTENT_TYPE_REQUIRED` — Content-Typeヘッダーの必須化設定 |
| ペイロードサイズ制限 | `SETTING_HTTP_MAX_CONTENT_LENGTH` — デフォルト100MB |
| ヘッダーサイズ制限 | `SETTING_HTTP_MAX_HEADER_SIZE` — デフォルト16KB |
| リクエストライン長制限 | `SETTING_HTTP_MAX_INITIAL_LINE_LENGTH` — デフォルト4KB |

**CORS設定** (`server/src/main/java/org/opensearch/http/CorsHandler.java`):

| 設定項目 | デフォルト値 | 説明 |
| --- | --- | --- |
| `http.cors.enabled` | `false` | CORS有効化 |
| `http.cors.allow-origin` | `""` | 許可オリジン（ワイルドカード・正規表現対応） |
| `http.cors.allow-methods` | `OPTIONS,HEAD,GET,POST,PUT,DELETE` | 許可HTTPメソッド |
| `http.cors.allow-headers` | `X-Requested-With,Content-Type,Content-Length` | 許可ヘッダー |
| `http.cors.allow-credentials` | `false` | 資格情報の送信許可 |
| `http.cors.max-age` | `1728000` | プリフライトキャッシュ秒数 |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| アプリケーションログ | OpenSearch動作ログ（Log4j2経由） | 設定依存 |
| セキュリティ例外ログ | `OpenSearchSecurityException` によるセキュリティ違反の記録 | 設定依存 |
| 監査ログ（詳細） | コアには未実装。セキュリティプラグインが認証イベント、アクセスイベント、コンプライアンスログ等の詳細な監査ログ機能を提供 | プラグイン設定依存 |

## 備考

- OpenSearchのセキュリティアーキテクチャは**プラグインベースの拡張モデル**を採用している。コアはフレームワーク（`IdentityPlugin`、`ActionFilter`、`SecureTransportSettingsProvider`等のインターフェース）を提供し、具体的なセキュリティ機能の実装はセキュリティプラグインに委譲する設計である
- 本リポジトリには**OpenSearch Security Plugin**は含まれていない。同プラグインが提供する以下の機能はコアの範囲外となる:
  - LDAP/SAML/OpenID Connect等の外部認証プロバイダ連携
  - 完全なロールベースアクセス制御（RBAC）
  - ドキュメントレベルセキュリティ / フィールドレベルセキュリティ
  - 詳細な監査ログ
  - マルチテナンシー
- Javaセキュリティマネージャーによるプラグインのサンドボックス化が実装されており、各プラグインは `plugin-security.policy` で宣言された権限の範囲内でのみ動作する
- セキュリティ設定のホットリロード（`NodesReloadSecureSettingsAction`）により、ノード再起動なしでの設定更新が可能
