# バッチ設計書 9-AsyncShardFlushTask

## 概要

本ドキュメントは、OpenSearchのIndexShardにおけるシャード単位の定期フラッシュを実行するバッチタスクの設計を記載する。

### 本バッチの処理概要

AsyncShardFlushTaskは、シャード単位で定期的にフラッシュ（Luceneコミット）を実行するバッチタスクである。特にPull型インジェスションインデックスにおいて、トランスログを持たない環境でオフセットの定期的なコミットを行うために使用される。

**業務上の目的・背景**：Pull型インジェスション（Kafka等の外部データソースからのデータ取り込み）を行うインデックスでは、トランスログが使用されない場合がある。このような環境では、フラッシュ（Luceneコミット）を通じてインジェスションオフセットを定期的にコミットしないと、ノード障害時に大量のデータを再取り込みする必要が生じる。本バッチはこの問題を解決するために、定期的にフラッシュを実行する。

**バッチの実行タイミング**：`index.periodic_flush_interval`で設定可能な間隔で定期実行される。インターバルが0以上かつタスクが未起動の場合にのみ開始される。

**主要な処理内容**：
1. 同時実行チェック（他のflush/rollが実行中でないことを確認）
2. `flushOrRollRunning`フラグを取得（CAS操作）
3. `shard.flush(new FlushRequest())`を実行
4. periodicFlushMetricをインクリメント
5. `flushOrRollRunning`フラグを解放

**前後の処理との関連**：Pull型インジェスションのインデックスでは、通常のトランスログベースの耐久性の代わりに本タスクによる定期フラッシュがオフセットのコミットを保証する。AsyncTranslogFSync（No.1）とは異なるメカニズムでデータ耐久性を提供する。

**影響範囲**：当該シャードのLuceneインデックスおよびインジェスションオフセットに影響する。フラッシュの頻度はディスクI/OとCPU使用率に影響する。

## バッチ種別

データ同期処理（シャードレベルの定期フラッシュ・オフセットコミット）

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 定期実行（index.periodic_flush_intervalで設定） |
| 実行時刻 | シャードのライフサイクルに依存（常時） |
| 実行曜日 | 該当なし（常時稼働） |
| 実行日 | 該当なし（常時稼働） |
| トリガー | タイマーベースの定期スケジュール（AbstractAsyncTask） |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| シャードがCLOSED以外 | `indexShard.state != IndexShardState.CLOSED`であること |
| インデックスがOPEN状態 | `IndexMetadata.State.OPEN`であること |
| インターバルが0より大きい | `getInterval().millis() > 0`であること |
| フラッシュ/ロール未実行 | `flushOrRollRunning`がfalseであること |

### 実行可否判定

`mustReschedule()`でシャードがCLOSED状態でないこと、インデックスがOPEN状態であること、インターバルが0msより大きいことを確認する。`runInternal()`では`flushOrRollRunning.compareAndSet(false, true)`によるCAS操作で排他制御を行い、他のフラッシュ/ロール処理と同時実行されないようにする。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| index.periodic_flush_interval | TimeValue | No | -1（無効） | 定期フラッシュの実行間隔 |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| IndexShard | インメモリ | 対象シャードのエンジン |
| Lucene IndexWriter | インメモリ/ディスク | シャードのLuceneインデックスライター |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| Luceneコミットポイント | ディスクファイル | フラッシュによりLuceneコミットが作成される |
| インジェスションオフセット | ディスクファイル | コミットポイントにオフセット情報が含まれる |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | segments_N（Luceneコミットファイル） |
| 出力先 | シャードのインデックスディレクトリ |
| 文字コード | バイナリ形式 |
| 区切り文字 | 該当なし |

## 処理フロー

### 処理シーケンス

```
1. タイマーによる定期起動
   └─ AbstractAsyncTaskにより指定間隔で起動
2. 排他制御チェック
   └─ flushOrRollRunning.compareAndSet(false, true)でCAS取得
3. フラッシュ実行
   └─ indexShard.flush(new FlushRequest())を呼び出し
4. メトリクス更新
   └─ periodicFlushMetricをインクリメント
5. フラグ解放
   └─ flushOrRollRunning.set(false)
6. 次回スケジュール
   └─ mustReschedule()がtrueの場合、再スケジュール
```

### フローチャート

```mermaid
flowchart TD
    A[タイマー起動] --> B{flushOrRollRunning CAS取得?}
    B -->|No| F[スキップ・再スケジュール]
    B -->|Yes| C[flush 実行]
    C --> D[periodicFlushMetric++]
    D --> E[flushOrRollRunning 解放]
    E --> F
```

## データベース操作仕様

### 操作別データベース影響一覧

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| Luceneフラッシュ | Luceneセグメント | WRITE/COMMIT | インメモリデータをディスクにコミット |

### テーブル別操作詳細

#### Luceneセグメント

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| COMMIT | segments_Nファイル | 新しいコミットポイントの作成 | インジェスションオフセットを含む |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 各種Exception | フラッシュ実行時のエラー | AbstractAsyncTaskのエラーハンドリングにより処理。finally句でフラグ解放を保証。 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 明示的なリトライなし（次回スケジュール実行で再試行） |
| リトライ間隔 | periodic_flush_interval |
| リトライ対象エラー | 次回実行時に自動的に再試行 |

### 障害時対応

フラッシュに失敗した場合、finally句で`flushOrRollRunning`フラグが確実に解放される。次回スケジュール実行時に再度フラッシュが試行される。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | シャード単位 |
| コミットタイミング | フラッシュ実行時（Luceneコミット） |
| ロールバック条件 | フラッシュ失敗時は前のコミットポイントが有効のまま |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1シャード |
| 目標処理時間 | periodic_flush_interval内に完了すること |
| メモリ使用量上限 | フラッシュ中のバッファメモリ |

## 排他制御

`flushOrRollRunning`のAtomicBooleanによるCAS操作で排他制御を行う。他のフラッシュ/ロール処理が実行中の場合はスキップされる。各シャードに対してAsyncShardFlushTaskは1つのみ存在する。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 情報ログ | タスク開始時 | "Started periodic flush task for shard [shardId] with interval [interval]" |
| エラーログ | フラッシュ失敗時 | AbstractAsyncTaskによるエラーログ |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| periodicFlushMetric | フラッシュ実行回数 | OpenSearchメトリクス |

## 備考

- 本タスクはFLUSHスレッドプールで実行される
- Pull型インジェスションインデックス向けの機能であり、通常のインデックスでは使用されない
- `flushOrRollRunning`のAtomicBooleanによる排他制御により、同時フラッシュを防止する
- `periodicFlushMetric`によりフラッシュ実行回数のメトリクスが記録される
- IndexShard内の内部クラスとして実装されている
- ソースコード: `server/src/main/java/org/opensearch/index/shard/IndexShard.java` (AsyncShardFlushTask内部クラス)
