# 機能設計書 126-自動タグ付け共通基盤

## 概要

本ドキュメントは、OpenSearchにおけるリクエストに対する自動タグ付け機能の共通基盤の設計を記載する。

### 本機能の処理概要

本機能は、クエリやリクエストに対してルールベースで自動的にタグを付与するための共通基盤（フレームワーク）を提供するモジュールである。

**業務上の目的・背景**：ワークロード管理やクエリ分類など、リクエストに対して自動的にメタデータ（タグ）を付与する必要がある。本モジュールは、ルールの定義、永続化、マッチング、属性管理といった共通機能を提供し、各種タグ付け機能の基盤として利用される。

**機能の利用シーン**：ワークロードグループへのクエリ自動分類、インデックスパターンに基づくリクエストのルーティング、リソース管理のためのクエリ分類時に利用される。

**主要な処理内容**：
1. FeatureTypeの登録と管理（AutoTaggingRegistry）
2. Ruleの定義とバリデーション
3. Attributeの管理と値の抽出
4. ルールの永続化（IndexStoredRulePersistenceService）
5. 属性値のストア管理（AttributeValueStore）

**関連システム・外部連携**：ワークロード管理プラグイン（workload-management）がこのモジュールの主要な利用者。システムインデックスを用いたルールの永続化。

**権限による制御**：ルールのCRUD操作に対するアクセス制御はプラグイン側で管理される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は共通基盤であり、直接的な画面は定義されていない |

## 機能種別

フレームワーク（共通基盤）/ CRUD操作

## 入力仕様

### 入力パラメータ

Ruleの作成時:

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| description | String | Yes | ルールの説明 | 空でないこと |
| attributeMap | Map<Attribute, Set<String>> | Yes | 属性名と値のマッピング | 空でないこと |
| featureType | FeatureType | Yes | 機能種別（例: workload_group） | 登録済みであること |
| featureValue | String | Yes | 機能の値（例: ワークロードグループID） | 空でないこと |
| updatedAt | String | Yes | 更新日時 | 空でないこと |

### 入力データソース

REST APIリクエスト、システムインデックスからの読み出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| id | String | ルールのユニークID（ハッシュベース） |
| description | String | ルールの説明 |
| attributeMap | Map | 属性と値のマッピング |
| featureType | String | 機能種別名 |
| featureValue | String | 機能の値 |
| updatedAt | String | 更新日時 |

### 出力先

REST APIレスポンス（JSON形式）、システムインデックス

## 処理フロー

### 処理シーケンス

```
【FeatureType登録】
1. プラグイン起動時にAutoTaggingRegistry.registerFeatureType()
   └─ FeatureTypeのバリデーション
   └─ 重複チェック
   └─ featureTypesRegistryMapに登録

【ルール作成】
1. CreateRuleRequest受信
2. RuleValidatorによるバリデーション
3. RuleUtils.computeRuleHash()でID生成
4. IndexStoredRulePersistenceServiceでシステムインデックスに永続化
5. AttributeValueStoreに属性値を登録

【ルールマッチング】
1. リクエストからAttributeExtractorで属性値を抽出
2. AttributeValueStoreで一致するルールを検索
3. ルールに紐づくfeatureValueを返却
```

### フローチャート

```mermaid
flowchart TD
    A[プラグイン起動] --> B[FeatureType登録]
    B --> C[AutoTaggingRegistry]
    D[ルール作成リクエスト] --> E[RuleValidator]
    E --> F[ルールID生成]
    F --> G[システムインデックスに永続化]
    G --> H[AttributeValueStore更新]
    I[リクエスト受信] --> J[AttributeExtractor]
    J --> K[AttributeValueStore検索]
    K --> L[マッチしたルールのfeatureValue返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | FeatureType一意性 | 同名のFeatureTypeは一度だけ登録可能 | 起動時 |
| BR-002 | FeatureType名制限 | FeatureType名は最大30文字 | 登録時 |
| BR-003 | ルールIDハッシュ | ルールIDはdescription, featureType, attributeMap, featureValueから計算されるハッシュ | ルール作成時 |
| BR-004 | FeatureTypeシングルトン | FeatureTypeはシングルトンとして扱われる | 常時 |

### 計算ロジック

- ルールID = RuleUtils.computeRuleHash(description, featureType, attributeMap, featureValue)

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ルール作成 | システムインデックス | INSERT | ルール情報の永続化 |
| ルール取得 | システムインデックス | SELECT | ルール情報の読み出し |
| ルール更新 | システムインデックス | UPDATE | ルール情報の更新 |
| ルール削除 | システムインデックス | DELETE | ルール情報の削除 |

### テーブル別操作詳細

ルール情報はOpenSearchのシステムインデックスにJSON形式で永続化される。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| IllegalStateException | 状態不正 | FeatureTypeの重複登録、nullフィールド | プラグイン設定確認 |
| ResourceNotFoundException | 未検出 | 未登録のFeatureType参照 | FeatureType名確認 |
| XContentParseException | パース失敗 | ルールJSON解析エラー | 入力形式確認 |

### リトライ仕様

システムインデックスへの永続化失敗時はActionListenerで通知。自動リトライは呼び出し側の責務。

## トランザクション仕様

システムインデックスのドキュメント単位の操作。OpenSearchのドキュメント操作のトランザクション保証に依存。

## パフォーマンス要件

ルールマッチングはリクエスト処理のクリティカルパスに含まれるため、AttributeValueStoreによるインメモリ検索で高速化されている。

## セキュリティ考慮事項

システムインデックスへのアクセスは内部的に行われる。プラグイン側でのアクセス制御が前提。

## 備考

本モジュールはexperimentalとして位置づけられている。ワークロード管理プラグインが主要な利用者。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: コアデータ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Rule.java | `modules/autotagging-commons/common/src/main/java/org/opensearch/rule/autotagging/Rule.java` | ルールの定義。id, description, attributeMap, featureType, featureValue |
| 1-2 | FeatureType.java | 同ディレクトリ | 機能種別インターフェース。属性とバリデーターを定義 |
| 1-3 | Attribute.java | 同ディレクトリ | 属性インターフェース。名前とXContentパース機能 |

**読解のコツ**: Rule.java内のBuilderパターンに注目。fromXContent()でJSONからのパース、toXContent()でJSONへのシリアライズを行う。

**主要処理フロー（Rule.java）**:
- **71-87行目**: コンストラクタでRuleValidatorによるバリデーション実行
- **122-124行目**: fromXContent()で静的ファクトリメソッドパターン
- **174-186行目**: toXContent()でJSON出力

#### Step 2: レジストリを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AutoTaggingRegistry.java | 同ディレクトリ | FeatureTypeのシングルトンレジストリ |

**主要処理フロー**:
- **28行目**: featureTypesRegistryMapはHashMap（起動時に登録、実行時は読み取りのみ）
- **43-50行目**: registerFeatureType()でバリデーション後に登録
- **79-87行目**: getFeatureType()で名前から取得

#### Step 3: 永続化層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | RulePersistenceService.java | `modules/autotagging-commons/common/src/main/java/org/opensearch/rule/` | 永続化サービスインターフェース |
| 3-2 | AttributeValueStore.java | `modules/autotagging-commons/common/src/main/java/org/opensearch/rule/storage/` | インメモリ属性値ストア |
| 3-3 | AttributeValueStoreFactory.java | 同ディレクトリ | ストアファクトリ |

### プログラム呼び出し階層図

```
プラグイン起動時
    |
    +-- AutoTaggingRegistry.registerFeatureType()
            +-- validateFeatureType()

ルール操作時
    |
    +-- RulePersistenceService
    |       +-- createRule()
    |       +-- getRule()
    |       +-- updateRule()
    |       +-- deleteRule()
    |
    +-- AttributeValueStore
            +-- put(attribute, value, ruleId)
            +-- get(attribute, value)
            +-- remove(attribute, value)

リクエスト処理時
    |
    +-- AttributeExtractor.extract(request)
    +-- AttributeValueStore.get(attribute, value)
    +-- Rule -> featureValue
```

### データフロー図

```
[入力]                    [処理]                           [出力]

プラグイン設定    -->  AutoTaggingRegistry       -->  FeatureTypeマップ
ルール作成Req    -->  RulePersistenceService     -->  システムインデックス
                      RuleValidator                   AttributeValueStore
リクエスト       -->  AttributeExtractor         -->  featureValue
                      AttributeValueStore
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Rule.java | `modules/autotagging-commons/common/src/main/java/org/opensearch/rule/autotagging/` | ソース | ルールモデル |
| FeatureType.java | 同上 | ソース | 機能種別インターフェース |
| Attribute.java | 同上 | ソース | 属性インターフェース |
| AutoTaggingRegistry.java | 同上 | ソース | FeatureTypeレジストリ |
| RuleValidator.java | 同上 | ソース | ルールバリデータ |
| FeatureValueValidator.java | 同上 | ソース | 機能値バリデータ |
| RulePersistenceService.java | `modules/autotagging-commons/common/src/main/java/org/opensearch/rule/` | ソース | 永続化サービスIF |
| RuleEntityParser.java | 同上 | ソース | ルールパーサーIF |
| RuleUtils.java | 同上 | ソース | ユーティリティ（ハッシュ生成等） |
| MatchLabel.java | 同上 | ソース | マッチラベル |
| AttributeValueStore.java | `modules/autotagging-commons/common/src/main/java/org/opensearch/rule/storage/` | ソース | 属性値ストア |
| AttributeValueStoreFactory.java | 同上 | ソース | ストアファクトリ |
| XContentRuleParser.java | 同上 | ソース | XContentルールパーサー |
| AttributeExtractor.java | `modules/autotagging-commons/common/src/main/java/org/opensearch/rule/attribute_extractor/` | ソース | 属性抽出IF |
