# QUASTO コードリーディングガイドライン

## はじめに

このガイドラインは、QUASTO（Quality Assurance Tool for Oracle databases）のコードベースを効率的に理解するための手引きです。
PL/SQLやOracle APEXに精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、PL/SQLの基本構文と概念を解説します。

### 1.1 プログラム構造

QUASTOは主にPL/SQL（Procedural Language/SQL）で開発されています。PL/SQLはOracleデータベースの手続き型プログラミング言語で、SQLを拡張した構文を持ちます。

基本的なプログラム構造は以下の通りです：

```sql
-- ファイル: src/plsql/pkg/qa_api_pkg.sql:47-151
create or replace package body qa_api_pkg as

  function tf_run_rule
  (
    pi_qaru_rule_number in qa_rules.qaru_rule_number%type
   ,pi_qaru_client_name in qa_rules.qaru_client_name%type
   ,pi_target_scheme    in varchar2 default user
  ) return qa_rules_t is

    -- ローカル定数の宣言
    c_unit constant varchar2(32767) := $$plsql_unit || '.tf_run_rule';

    -- ローカル変数の宣言
    l_param_list qa_logger_pkg.tab_param;
    l_rule_active boolean;
    l_qa_rules    qa_rules_t;

  begin
    -- 処理本体
    -- ...
    return l_qa_rules;
  exception
    when no_data_found then
      -- 例外処理
      qa_logger_pkg.p_qa_log(p_text => 'No Data found...');
    when others then
      qa_logger_pkg.p_qa_log(p_text => 'There has been an error...');
      raise;
  end tf_run_rule;

end qa_api_pkg;
/
```

### 1.2 データ型と変数

PL/SQLで使用される主要なデータ型：

| データ型 | 説明 | 例 |
|---------|------|-----|
| `VARCHAR2(n)` | 可変長文字列 | `l_name varchar2(100)` |
| `NUMBER` | 数値型 | `l_count number` |
| `DATE` | 日付型 | `l_date date` |
| `BOOLEAN` | 論理型 | `l_flag boolean` |
| `CLOB` | 大容量文字データ | `l_sql clob` |

**%TYPE属性**：テーブルカラムと同じデータ型を宣言

```sql
-- ファイル: src/plsql/pkg/qa_main_pkg.sql:21-25
function f_get_rule
(
  pi_qaru_rule_number in qa_rules.qaru_rule_number%type  -- qa_rulesテーブルのqaru_rule_numberカラムと同じ型
 ,pi_qaru_client_name in qa_rules.qaru_client_name%type
) return qa_rule_t;
```

### 1.3 制御構造

**条件分岐（IF-THEN-ELSE）**：

```sql
-- ファイル: src/plsql/pkg/qa_api_pkg.sql:74-80
if pi_qaru_rule_number is null or
   pi_qaru_client_name is null or
   pi_target_scheme is null
then
  raise_application_error(-20001, 'Missing input parameter value...');
end if;
```

**ループ処理（FOR LOOP）**：

```sql
-- ファイル: src/plsql/pkg/qa_api_pkg.sql:216-260
for i in 1 .. l_qaru_rule_numbers.count
loop
  -- 処理内容
end loop;
```

**カーソルループ**：

```sql
-- ファイル: src/plsql/pkg/qa_main_pkg.sql:853-889
for i in pi_qa_rules.first .. pi_qa_rules.last
loop
  -- 処理内容
end loop;
```

### 1.4 関数/メソッド定義

**関数（FUNCTION）**：値を返す処理単位

```sql
-- ファイル: src/plsql/pkg/qa_main_pkg.sql:213-256
function f_get_rule
(
  pi_qaru_rule_number in qa_rules.qaru_rule_number%type
 ,pi_qaru_client_name in qa_rules.qaru_client_name%type
) return qa_rule_t is
  c_unit constant varchar2(32767) := $$plsql_unit || '.f_get_rule';
  l_qa_rule qa_rule_t;
begin
  select qa_rule_t(...)
  into l_qa_rule
  from qa_rules q
  where q.qaru_rule_number = pi_qaru_rule_number;

  return l_qa_rule;
end f_get_rule;
```

**プロシージャ（PROCEDURE）**：値を返さない処理単位

```sql
-- ファイル: src/plsql/pkg/qa_main_pkg.sql:684-746
procedure p_test_rule
(
  pi_qaru_client_name in qa_rules.qaru_client_name%type
 ,pi_qaru_rule_number in qa_rules.qaru_rule_number%type
 ,pi_scheme_names     in varchar2_tab_t
 ,po_result           out number  -- OUTパラメータで結果を返す
 ,po_scheme_objects   out qa_scheme_object_amounts_t
 ,po_invalid_objects  out qa_rules_t
) is
begin
  -- 処理内容
end p_test_rule;
```

### 1.5 モジュール/インポート

PL/SQLではモジュールは「パッケージ（PACKAGE）」として定義されます。

```sql
-- ファイル: src/plsql/pkg/qa_api_pkg.sql:1-46
create or replace package qa_api_pkg
  authid current_user  -- 呼び出しユーザーの権限で実行
as
  -- パッケージ仕様部（公開インターフェース）
  function tf_run_rule(...) return qa_rules_t;
  function tf_run_rules(...) return qa_rules_t;
end qa_api_pkg;
/

-- パッケージ本体部（実装）
create or replace package body qa_api_pkg as
  -- 実装コード
end qa_api_pkg;
/
```

他のパッケージの呼び出し：

```sql
-- ファイル: src/plsql/pkg/qa_api_pkg.sql:93-94
l_qa_rule := qa_main_pkg.f_get_rule(pi_qaru_rule_number => pi_qaru_rule_number
                                   ,pi_qaru_client_name => pi_qaru_client_name);
```

---

## 2. プロジェクト固有の概念

> このセクションでは、QUASTO特有の概念を解説します。

### 2.1 フレームワーク固有の概念

#### Oracle APEXとの連携

QUASTOはOracle APEXアプリケーションとして動作します。APEXはOracleデータベース上でWebアプリケーションを構築するローコード開発プラットフォームです。

```sql
-- ファイル: src/plsql/pkg/qa_apex_app_pkg.sql:59-115
function get_faceted_search_dashboard_data
(
  pi_page_id          in number
 ,pi_region_static_id in varchar2
) return qa_test_results_table_t pipelined  -- パイプライン関数でAPEXレポートにデータ供給
is
  l_region_id number;
  l_context   apex_exec.t_context;  -- APEX実行コンテキスト
begin
  -- APEX APIを使用したクエリ実行
  l_context := apex_region.open_query_context(p_page_id   => pi_page_id
                                             ,p_region_id => l_region_id);
  -- ...
end;
```

#### utPLSQLとの連携

QUASTOはutPLSQL（PL/SQLユニットテストフレームワーク）と統合されています。

```sql
-- ファイル: src/plsql/pkg/qa_unit_tests_pkg.sql:1090
ut.expect(pi_result).to_equal(1);  -- utPLSQLのアサーション
```

### 2.2 プロジェクト独自のパターン

#### ルールベースの品質チェック

QUASTOの核心は「ルール」です。各ルールはSQLクエリとして定義され、品質基準に違反するオブジェクトを検出します。

```sql
-- ファイル: README.md:119-152（ルール定義例）
-- 主キーがないテーブルを検出するルール
with param as
 (select :1 scheme
        ,:2 qaru_id
        ,:3 qaru_category
        ,:4 qaru_error_level
        ,:5 qaru_object_types
        ,:6 qaru_error_message
        ,:7 qaru_sql
  from dual)
select qa_rule_t(...)
from param p
cross join all_tables at
where not exists (select null
       from all_constraints cons
       where cons.constraint_type = 'P')
```

#### オブジェクト型（TYPE）によるデータ構造

```sql
-- ファイル: src/plsql/typ/qa_rule_t.sql:1-103
create or replace type qa_rule_t force as
  object
  (
    qaru_id number
   ,qaru_category varchar2(10)
   ,qaru_error_level number
   ,qaru_error_message varchar2(4000)
   ,scheme_name varchar2(100)
   ,object_name varchar2(1000)
   -- ...

  -- コンストラクタ関数
  constructor function qa_rule_t(...) return self as result,

  member function to_string return varchar2
  )
;
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `qa_*_pkg.sql` | パッケージファイル | `qa_api_pkg.sql`, `qa_main_pkg.sql` |
| `qa_*_t.sql` | オブジェクト型定義 | `qa_rule_t.sql`, `qa_rules_t.sql` |
| `qa_*_v.sql` | ビュー定義 | `qa_scheme_names_for_testing_v.sql` |
| `qa_*_trg.sql` | トリガー定義 | `qaru_iu_trg.sql` |
| `*_seq.sql` | シーケンス定義 | `qaru_seq.sql` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `f_` | 関数（Function） | `f_get_rule`, `f_is_rule_active` |
| `p_` | プロシージャ（Procedure） | `p_test_rule`, `p_exclude_objects` |
| `tf_` | テーブル関数（Table Function） | `tf_run_rule`, `tf_run_rules` |
| `pi_` | 入力パラメータ（Parameter In） | `pi_qaru_rule_number` |
| `po_` | 出力パラメータ（Parameter Out） | `po_result` |
| `l_` | ローカル変数（Local） | `l_qa_rule`, `l_count` |
| `c_` | 定数（Constant） | `c_unit` |
| `gc_` | グローバル定数（Global Constant） | `gc_apex_flag` |
| `_t` | 型（Type） | `qa_rule_t`, `qa_rules_t` |
| `_v` | ビュー（View） | `qa_scheme_names_for_testing_v` |
| `_trg` | トリガー（Trigger） | `qaru_iu_trg` |
| `_pkg` | パッケージ（Package） | `qa_api_pkg` |
| `_seq` | シーケンス（Sequence） | `qaru_seq` |

### 3.3 テーブル/カラム命名

| プレフィックス | 意味 | 例 |
|--------------|------|-----|
| `qa_` | QUASTOテーブル | `qa_rules`, `qa_test_runs` |
| `qaru_` | qa_rulesテーブルのカラム | `qaru_id`, `qaru_rule_number` |
| `qatr_` | qa_test_runsテーブルのカラム | `qatr_id`, `qatr_scheme_name` |
| `qato_` | qa_test_run_invalid_objectsテーブルのカラム | `qato_object_name` |
| `qaif_` | qa_import_filesテーブルのカラム | `qaif_id` |

### 3.3 プログラム分類一覧

| 分類 | プレフィックス | 説明 |
|-----|--------------|------|
| API層 | `qa_api_` | 外部インターフェース（ルール実行API） |
| メイン層 | `qa_main_` | コアビジネスロジック |
| ユーティリティ | `qa_utils_` | 汎用ヘルパー関数 |
| APEX連携 | `qa_apex_` | Oracle APEX固有機能 |
| ユニットテスト | `qa_unit_tests_` | utPLSQL連携機能 |
| ロガー | `qa_logger_` | ログ機能 |
| エクスポート/インポート | `qa_export_import_` | ルールのエクスポート・インポート |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
quasto-dev/
├── install.sql                 # メインインストールスクリプト
├── uninstall.sql              # アンインストールスクリプト
├── README.md                  # プロジェクト概要
├── license.txt                # ライセンス情報
├── src/                       # ソースコードディレクトリ
│   ├── install_objects.sql    # オブジェクトインストール制御
│   ├── install_quasto_objects.sql    # QUASTOコアインストール
│   ├── install_utplsql_objects.sql   # utPLSQL連携インストール
│   ├── install_apex_objects.sql      # APEX連携インストール
│   ├── uninstall_all_objects.sql     # 全オブジェクト削除
│   ├── apex/                  # Oracle APEXアプリケーション
│   │   ├── f141/             # APEXアプリケーションID:141
│   │   │   ├── install.sql
│   │   │   ├── application/  # アプリケーション定義
│   │   │   └── embedded_code/ # 埋め込みコード
│   │   └── plugin/           # APEXプラグイン
│   ├── ddl/                   # データ定義言語（DDL）
│   │   ├── tab/              # テーブル定義
│   │   ├── cons/             # 制約定義
│   │   ├── ind/              # インデックス定義
│   │   └── seq/              # シーケンス定義
│   ├── plsql/                 # PL/SQLコード
│   │   ├── pkg/              # パッケージ
│   │   ├── typ/              # 型定義
│   │   ├── vw/               # ビュー
│   │   └── trg/              # トリガー
│   └── scripts/               # ユーティリティスクリプト
│       ├── export_rules_to_file.sql
│       ├── import_file_to_rules.js
│       └── qa_rules_mt_ag.json  # サンプルルール
└── docs/                      # ドキュメント
    └── code-to-docs/          # 自動生成ドキュメント
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/` | ソースコード全体 | `install_objects.sql` |
| `src/apex/` | APEXアプリケーション | `f141.sql` |
| `src/ddl/tab/` | テーブル定義 | `qa_rules.sql` |
| `src/ddl/cons/` | 制約定義 | `qa_rules.sql` |
| `src/ddl/ind/` | インデックス定義 | `qa_rules.sql` |
| `src/ddl/seq/` | シーケンス定義 | `qaru_seq.sql` |
| `src/plsql/pkg/` | パッケージ（仕様/本体） | `qa_api_pkg.sql` |
| `src/plsql/typ/` | オブジェクト型 | `qa_rule_t.sql` |
| `src/plsql/vw/` | ビュー | `qa_scheme_names_for_testing_v.sql` |
| `src/plsql/trg/` | トリガー | `qaru_iu_trg.sql` |
| `src/scripts/` | ユーティリティ | `export_rules_to_file.sql` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

QUASTOは3層アーキテクチャを採用しています：

```mermaid
graph TB
    subgraph "プレゼンテーション層"
        APEX[Oracle APEX アプリケーション]
        API[QA_API_PKG]
    end

    subgraph "ビジネスロジック層"
        MAIN[QA_MAIN_PKG]
        UNIT[QA_UNIT_TESTS_PKG]
        UTILS[QA_UTILS_PKG]
    end

    subgraph "データアクセス層"
        RULES[QA_RULES テーブル]
        RESULTS[QA_TEST_RESULTS テーブル]
        RUNS[QA_TEST_RUNS テーブル]
    end

    APEX --> API
    API --> MAIN
    MAIN --> UNIT
    MAIN --> UTILS
    MAIN --> RULES
    UNIT --> RESULTS
    UNIT --> RUNS
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| API層 | 外部インターフェース提供 | `qa_api_pkg.sql` |
| ビジネスロジック層 | ルール処理・テスト実行 | `qa_main_pkg.sql`, `qa_unit_tests_pkg.sql` |
| データアクセス層 | データ永続化 | DDLファイル群、トリガー |
| ユーティリティ層 | 共通機能提供 | `qa_utils_pkg.sql`, `qa_logger_pkg.sql` |
| APEX層 | UI連携 | `qa_apex_app_pkg.sql`, `qa_apex_api_pkg.sql` |

### 5.3 データフロー

1. **ルール定義フロー**
   - ユーザーがAPEX UIまたはSQLで`QA_RULES`テーブルにルールを登録
   - トリガー`QARU_IU_TRG`が監査情報を自動設定

2. **ルール実行フロー**
   - `qa_api_pkg.tf_run_rule()`でルールを実行
   - `qa_main_pkg.f_get_rule()`でルール定義を取得
   - 動的SQLでルールを評価
   - 違反オブジェクトを`qa_rule_t`コレクションで返却

3. **テスト結果保存フロー**
   - `qa_unit_tests_pkg`でutPLSQLテストを実行
   - 結果を`QA_TEST_RUNS`、`QA_TEST_RESULTS`テーブルに保存
   - XML形式でエクスポート可能

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

**ルール実行のエントリーポイント**：

```sql
-- ファイル: src/plsql/pkg/qa_api_pkg.sql:24-29
-- 単一ルールの実行
function tf_run_rule
(
  pi_qaru_rule_number in qa_rules.qaru_rule_number%type
 ,pi_qaru_client_name in qa_rules.qaru_client_name%type
 ,pi_target_scheme    in varchar2 default user
) return qa_rules_t;

-- ファイル: src/plsql/pkg/qa_api_pkg.sql:39-43
-- 全ルールの実行
function tf_run_rules
(
  pi_qaru_client_name in qa_rules.qaru_client_name%type
 ,pi_target_scheme    in varchar2 default user
) return qa_rules_t;
```

使用例：

```sql
-- 単一ルールの実行
select *
from qa_api_pkg.tf_run_rule(pi_qaru_rule_number => '23.1'
                           ,pi_qaru_client_name => 'MT AG'
                           ,pi_target_scheme    => 'QUASTO')

-- 全ルールの実行
select *
from qa_api_pkg.tf_run_rules(pi_qaru_client_name => 'MT AG'
                            ,pi_target_scheme    => 'QUASTO')
```

### 6.2 ビジネスロジック

**コアパッケージ `qa_main_pkg`**：

```sql
-- ファイル: src/plsql/pkg/qa_main_pkg.sql:1-202（仕様部）
-- 主要機能：
-- - f_get_rule: ルール情報の取得
-- - f_is_rule_active: ルール有効性チェック
-- - p_test_rule: ルールテストの実行
-- - f_insert_rule: ルールの登録
-- - p_exclude_objects: 除外オブジェクトの処理
-- - f_check_for_loop: 先行ルールのループ検出
```

### 6.3 データアクセス

**主要テーブル `QA_RULES`**：

```sql
-- ファイル: src/ddl/tab/qa_rules.sql:4-27
create table QA_RULES
(
  qaru_id              NUMBER not null,           -- 主キー
  qaru_rule_number     VARCHAR2(20 CHAR) not null,-- ルール番号
  qaru_client_name     VARCHAR2(100 CHAR) not null,-- クライアント名
  qaru_name            VARCHAR2(100 CHAR) not null,-- ルール名
  qaru_category        VARCHAR2(100 CHAR) not null,-- カテゴリ（APEX/DDL/DATA）
  qaru_object_types    VARCHAR2(4000 CHAR) not null,-- 対象オブジェクト型
  qaru_error_message   VARCHAR2(4000 CHAR) not null,-- エラーメッセージ
  qaru_error_level     NUMBER not null,           -- エラーレベル（1=Error,2=Warning,4=Info）
  qaru_is_active       NUMBER default 1 not null, -- 有効フラグ
  qaru_sql             CLOB not null,             -- ルールSQL
  qaru_predecessor_ids VARCHAR2(4000 CHAR),       -- 先行ルールID
  qaru_layer           VARCHAR2(100 CHAR) not null -- レイヤー（PAGE/APPLICATION/DATABASE）
)
```

### 6.4 ユーティリティ/共通機能

**ユーティリティパッケージ `qa_utils_pkg`**：

```sql
-- ファイル: src/plsql/pkg/qa_utils_pkg.sql:19-66
-- 文字列⇔テーブル変換
function f_get_table_as_string(pi_table in varchar2_tab_t, pi_separator in varchar2) return varchar2;
function f_get_string_as_table(pi_string in varchar2, pi_separator in varchar2) return varchar2_tab_t;

-- 文字列の統一化（命名規則準拠）
function f_get_unified_string(pi_string in varchar2, ...) return varchar2;

-- DBMS_OUTPUT出力
procedure p_print_to_dbms_output(pi_string in varchar2);
procedure p_print_to_dbms_output(pi_string in clob);
```

**ロガーパッケージ `qa_logger_pkg`**：

```sql
-- ファイル: src/plsql/pkg/qa_logger_pkg.sql（参照）
-- 全パッケージで使用されるログ機能
qa_logger_pkg.append_param(p_params => l_param_list
                          ,p_name_01 => 'pi_qaru_rule_number'
                          ,p_val_01  => pi_qaru_rule_number);
qa_logger_pkg.p_qa_log(p_text => 'エラーメッセージ'
                      ,p_scope => c_unit
                      ,p_extra => sqlerrm
                      ,p_params => l_param_list);
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| パラメータログ | 関数開始時のパラメータ記録 | 高 | 全パッケージ |
| 例外ハンドリング | 統一的な例外処理 | 高 | 全パッケージ |
| BULK COLLECT | 効率的なデータ取得 | 中 | `qa_main_pkg.sql` |
| パイプライン関数 | APEXレポート向けデータ供給 | 中 | `qa_apex_app_pkg.sql` |
| 条件付きコンパイル | 機能フラグによるコード切替 | 中 | `qa_api_pkg.sql` |

### 各パターンの詳細

#### パターン1: パラメータログ

**目的:** デバッグとトレーサビリティのため、関数実行時のパラメータを記録する

**実装例:**
```sql
-- ファイル: src/plsql/pkg/qa_api_pkg.sql:66-72
function tf_run_rule(...) is
  c_unit constant varchar2(32767) := $$plsql_unit || '.tf_run_rule';
  l_param_list qa_logger_pkg.tab_param;
begin
  qa_logger_pkg.append_param(p_params  => l_param_list
                            ,p_name_01 => 'pi_qaru_rule_number'
                            ,p_val_01  => pi_qaru_rule_number
                            ,p_name_02 => 'pi_qaru_client_name'
                            ,p_val_02  => pi_qaru_client_name
                            ,p_name_03 => 'pi_target_scheme'
                            ,p_val_03  => pi_target_scheme);
  -- 処理本体
end;
```

**解説:** `$$plsql_unit`はパッケージ名を取得するPL/SQL予約語です。これにより、ログ出力時に呼び出し元を特定できます。

#### パターン2: 例外ハンドリング

**目的:** 統一的なエラー処理とログ記録を行う

**実装例:**
```sql
-- ファイル: src/plsql/pkg/qa_api_pkg.sql:139-151
exception
  when no_data_found then
    qa_logger_pkg.p_qa_log(p_text   => 'No Data found while selecting from qa_rules'
                          ,p_scope  => c_unit
                          ,p_extra  => sqlerrm
                          ,p_params => l_param_list);
  when others then
    qa_logger_pkg.p_qa_log(p_text   => 'There has been an error while trying to select from qa_rules!'
                          ,p_scope  => c_unit
                          ,p_extra  => sqlerrm
                          ,p_params => l_param_list);
    raise;  -- 例外を再スロー
```

**解説:** `no_data_found`と`others`を分離し、データなしの場合は静かに処理、その他のエラーは再スローしています。

#### パターン3: BULK COLLECT

**目的:** 複数行を効率的に一括取得する

**実装例:**
```sql
-- ファイル: src/plsql/pkg/qa_api_pkg.sql:99-108
execute immediate l_qa_rule.qaru_sql bulk collect
  into l_qa_rules
  using pi_target_scheme, l_qa_rule.qaru_id, l_qa_rule.qaru_category, ...;
```

**解説:** `BULK COLLECT INTO`により、動的SQLの結果をコレクションに一括格納します。行単位のフェッチより高速です。

#### パターン4: 条件付きコンパイル

**目的:** コンパイル時に機能の有効/無効を切り替える

**実装例:**
```sql
-- ファイル: src/plsql/pkg/qa_api_pkg.sql:124-132
$IF qa_constant_pkg.gc_apex_flag = 1
$THEN
  if l_qa_rule.qaru_category = 'APEX'
  then
    qa_apex_api_pkg.p_exclude_not_whitelisted_apex_entries(pi_qa_rules_t => l_qa_rules);
  end if;
$ELSE
  null;
$END
```

**解説:** `$IF`〜`$THEN`〜`$ELSE`〜`$END`はコンパイル時に評価され、不要なコードは除外されます。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: 単一ルール実行フロー

**概要:** ルール番号を指定して品質チェックを実行し、違反オブジェクトを取得する

**処理フロー:**
```
qa_api_pkg.tf_run_rule → qa_main_pkg.f_is_rule_active → qa_main_pkg.f_get_rule → EXECUTE IMMEDIATE → qa_main_pkg.p_exclude_objects → 結果返却
```

**詳細な追跡:**

1. **API呼び出し** (`src/plsql/pkg/qa_api_pkg.sql:49-54`)
   ```sql
   function tf_run_rule
   (
     pi_qaru_rule_number in qa_rules.qaru_rule_number%type
    ,pi_qaru_client_name in qa_rules.qaru_client_name%type
    ,pi_target_scheme    in varchar2 default user
   ) return qa_rules_t is
   ```

2. **ブラックリストチェック** (`src/plsql/pkg/qa_api_pkg.sql:82-84`)
   ```sql
   if qa_main_pkg.f_is_owner_black_listed(pi_user_name => pi_target_scheme) = false
   then
   ```

3. **ルール有効性確認** (`src/plsql/pkg/qa_api_pkg.sql:85-91`)
   ```sql
   l_rule_active := qa_main_pkg.f_is_rule_active(pi_qaru_rule_number => pi_qaru_rule_number
                                                ,pi_qaru_client_name => pi_qaru_client_name);
   if l_rule_active = false
   then
     raise_application_error(-20001, 'Rule is not set to active...');
   end if;
   ```

4. **ルール情報取得** (`src/plsql/pkg/qa_api_pkg.sql:93-94`)
   ```sql
   l_qa_rule := qa_main_pkg.f_get_rule(pi_qaru_rule_number => pi_qaru_rule_number
                                      ,pi_qaru_client_name => pi_qaru_client_name);
   ```

5. **動的SQL実行** (`src/plsql/pkg/qa_api_pkg.sql:97-108`)
   ```sql
   if l_qa_rule.qaru_category != 'APEX'
   then
     execute immediate l_qa_rule.qaru_sql bulk collect
       into l_qa_rules
       using pi_target_scheme, l_qa_rule.qaru_id, ...;
   ```

6. **除外オブジェクト処理** (`src/plsql/pkg/qa_api_pkg.sql:134`)
   ```sql
   qa_main_pkg.p_exclude_objects(pi_qa_rules => l_qa_rules);
   ```

7. **結果返却** (`src/plsql/pkg/qa_api_pkg.sql:135`)
   ```sql
   return l_qa_rules;
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか（`qa_api_pkg`から開始）
- [ ] 呼び出し関係を把握したか（パッケージ間の依存関係）
- [ ] データの変換ポイントを確認したか（`qa_rule_t`型への変換）
- [ ] エラーハンドリングを確認したか（例外節の処理）
- [ ] 最終的な出力を確認したか（`qa_rules_t`コレクション）

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要、インストール手順
2. `docs/code-to-docs/アーキテクチャ設計書/` - システム構成
3. `docs/code-to-docs/機能一覧/` - 機能概要

#### 特定機能を理解したい場合
1. `docs/code-to-docs/機能一覧/` - 機能の特定
2. `docs/code-to-docs/機能設計書/` - 詳細設計
3. `src/plsql/pkg/` - 関連ソースコード

#### 改修作業を行う場合
1. 対象機能の設計書（`docs/code-to-docs/機能設計書/`）
2. `docs/code-to-docs/画面設計書/` - UI影響確認
3. `docs/code-to-docs/単体テストケース一覧/` - テストケース確認

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| README.md | プロジェクト概要・インストール手順 | 初回参画時 |
| アーキテクチャ設計書 | システム構成・技術選定 | 全体設計理解時 |
| 機能一覧 | 機能の一覧と概要 | 機能調査時 |
| 機能設計書 | 各機能の詳細設計 | 実装・改修時 |
| 画面設計書 | APEX画面の設計 | UI関連作業時 |
| データベース設計書 | テーブル・カラム定義 | DB関連作業時 |
| テスト方針書 | テスト戦略・方針 | テスト計画時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: パッケージ仕様部と本体部の違いは？
A: 仕様部（SPEC）は公開インターフェースを定義し、本体部（BODY）は実装を含みます。ファイル内で`create or replace package`が仕様部、`create or replace package body`が本体部です。

```sql
-- 仕様部（インターフェース定義）
create or replace package qa_api_pkg as
  function tf_run_rule(...) return qa_rules_t;  -- シグネチャのみ
end qa_api_pkg;

-- 本体部（実装）
create or replace package body qa_api_pkg as
  function tf_run_rule(...) return qa_rules_t is
  begin
    -- 実装コード
  end;
end qa_api_pkg;
```

#### Q: `authid current_user`と`authid definer`の違いは？
A:
- `authid current_user`: 呼び出し元ユーザーの権限で実行（`qa_api_pkg`で使用）
- `authid definer`: パッケージ所有者の権限で実行（`qa_main_pkg`で使用）

#### Q: `$$plsql_unit`とは？
A: コンパイル時に現在のプログラムユニット名（パッケージ名）に置換されるPL/SQL予約語です。ログ出力時に呼び出し元を特定するために使用されます。

#### Q: `%TYPE`と`%ROWTYPE`の違いは？
A:
- `%TYPE`: 特定カラムと同じ型（例: `qa_rules.qaru_id%type`）
- `%ROWTYPE`: テーブル行全体と同じ構造（例: `qa_rules%rowtype`）

### デバッグのヒント

1. **Loggerテーブルの確認**
   ```sql
   select * from logger_logs order by id desc;
   select * from logger_logs_5_min;  -- 直近5分
   select * from logger_logs_60_min; -- 直近60分
   ```

2. **DBMS_OUTPUT有効化**
   ```sql
   set serveroutput on size unlimited
   exec dbms_output.enable(10000000);
   ```

3. **動的SQLのデバッグ**
   ```sql
   -- ルールSQLを確認
   select qaru_sql from qa_rules where qaru_rule_number = '23.1';
   ```

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| QUASTO | Quality Assurance Tool for Oracle databases |
| PL/SQL | Procedural Language/SQL（Oracle手続き型言語） |
| Oracle APEX | Application Express（ローコード開発プラットフォーム） |
| utPLSQL | PL/SQL用ユニットテストフレームワーク |
| DDL | Data Definition Language（CREATE/ALTER/DROP） |
| DML | Data Manipulation Language（INSERT/UPDATE/DELETE） |
| ルール | 品質基準を定義するSQLクエリ |
| クライアント | ルールを所有するプロジェクト/組織 |

### B. 主要ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `qa_api_pkg.sql` | API層パッケージ | ルール実行インターフェース |
| `qa_main_pkg.sql` | メインパッケージ | コアビジネスロジック |
| `qa_unit_tests_pkg.sql` | テストパッケージ | utPLSQL連携 |
| `qa_utils_pkg.sql` | ユーティリティ | 汎用関数群 |
| `qa_apex_app_pkg.sql` | APEX連携 | UI用機能 |
| `qa_rule_t.sql` | 型定義 | ルール結果オブジェクト |
| `qa_rules.sql` | テーブル定義 | ルール格納テーブル |
| `qaru_iu_trg.sql` | トリガー | 監査情報自動設定 |

### C. 参考資料

- [Oracle PL/SQL Language Reference](https://docs.oracle.com/en/database/oracle/oracle-database/19/lnpls/)
- [Oracle APEX Documentation](https://docs.oracle.com/en/database/oracle/apex/)
- [utPLSQL Documentation](http://utplsql.org/utPLSQL/latest/)
- [Logger Framework](https://github.com/OraOpenSource/Logger)
- [QUASTO GitHub Repository](https://github.com/mt-ag/quasto)
