# 機能設計書 10-ルールJSONエクスポート

## 概要

本ドキュメントは、QUASTOシステムにおけるルールJSONエクスポート機能の設計仕様を記述する。

### 本機能の処理概要

ルールJSONエクスポート機能は、QA_RULESテーブルのルールをJSON形式でエクスポートする機能である。qa_export_import_rules_pkg.f_export_rules_table_to_clob関数として実装されており、クライアント名とカテゴリでフィルタリングしたルールをJSON構造に変換してCLOB型で返却する。

**業務上の目的・背景**：ルール定義をバックアップしたり、他の環境に移行したりするために、ルールをポータブルな形式でエクスポートする必要がある。JSON形式は可読性が高く、他システムとの連携も容易なため、ルールの交換フォーマットとして採用されている。

**機能の利用シーン**：
- ルール定義のバックアップ作成
- 開発環境から本番環境へのルール移行
- 複数プロジェクト間でのルール共有
- ルール定義のバージョン管理（JSON形式でGit等に保存）

**主要な処理内容**：
1. クライアント名とカテゴリでQA_RULESテーブルからルールを取得
2. クライアント名 > カテゴリ > ルールの階層構造でJSONオブジェクトを構築
3. json_object_t、json_array_t型を使用してJSON構造を生成
4. json_serializeでpretty print形式に整形
5. CLOB型で結果を返却

**関連システム・外部連携**：APEX Rule Files画面からダウンロード機能として提供。APEXアプリケーションプロセス（GetRuleJsonAttachment）と連携。

**権限による制御**：ルールの参照権限を持つユーザーがエクスポート可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 9 | Rule Files | 主機能 | クライアント名を選択してルールをJSON形式でダウンロード |

## 機能種別

データ出力 / エクスポート

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_client_name | VARCHAR2 | Yes | クライアント名 | NOT NULL |
| pi_category | VARCHAR2 | No | カテゴリ（DDL/DML/APEX） | デフォルト: NULL（全カテゴリ） |

### 入力データソース

- APEX Rule Files画面からのパラメータ
- 他のPL/SQLプログラムからの呼び出し

## 出力仕様

### 出力データ

出力JSONの構造：
```json
{
  "version": 1.1,
  "qa_rules": [
    {
      "client_names": [
        {
          "client_name": "CLIENT1",
          "categories": [
            {
              "category": "DDL",
              "rules": [
                {
                  "rule_number": "R001",
                  "name": "ルール名",
                  "object_types": "TABLE",
                  "error_message": "エラーメッセージ",
                  "comment": "コメント",
                  "exclude_objects": "",
                  "error_level": 1,
                  "is_active": 1,
                  "sql": "SELECT ...",
                  "predecessor_ids": "",
                  "layer": "DATABASE"
                }
              ]
            }
          ]
        }
      ]
    }
  ]
}
```

| 項目名 | 型 | 説明 |
|--------|-----|------|
| version | NUMBER | JSONフォーマットバージョン（現在1.1） |
| client_names | ARRAY | クライアント名配列 |
| categories | ARRAY | カテゴリ配列 |
| rules | ARRAY | ルール配列 |

### 出力先

- 関数の戻り値（CLOB）
- APEX画面からダウンロード（ファイル）

## 処理フロー

### 処理シーケンス

```
1. パラメータログ記録
   └─ qa_logger_pkg.append_paramでパラメータを記録
2. ルール件数確認
   └─ 条件に合致するルールが存在するか確認
3. クライアント名ループ
   └─ 3.1 カテゴリループ
       └─ 3.2 ルールループ
           └─ 3.3 json_object_tでルールJSONを構築
       └─ 3.4 json_array_tでルール配列を構築
   └─ 3.5 json_array_tでカテゴリ配列を構築
4. メインJSONオブジェクト構築
   └─ version、qa_rulesを設定
5. JSON整形
   └─ json_serializeでpretty print
6. 結果返却
   └─ CLOB型で返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[パラメータログ記録]
    B --> C[ルール件数確認]
    C --> D{ルールあり?}
    D -->|No| E[NULL返却]
    D -->|Yes| F[クライアント名ループ]
    F --> G[カテゴリループ]
    G --> H[ルールループ]
    H --> I[l_rule_json構築]
    I --> J[l_rules_array_jsonに追加]
    J --> K{次のルールあり?}
    K -->|Yes| H
    K -->|No| L[l_category_json構築]
    L --> M{次のカテゴリあり?}
    M -->|Yes| G
    M -->|No| N[l_client_name_json構築]
    N --> O{次のクライアントあり?}
    O -->|Yes| F
    O -->|No| P[メインJSON構築]
    P --> Q[json_serialize pretty]
    Q --> R[CLOB返却]
    R --> S[終了]
    E --> S
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ルールなし時NULL | 条件に合致するルールがない場合はNULLを返却 | エクスポート時 |
| BR-02 | 階層構造 | client_names > categories > rulesの3階層構造 | JSON生成時 |
| BR-03 | バージョン番号 | JSONにはg_version_number（現在1.1）を含める | JSON生成時 |
| BR-04 | Pretty Print | json_serializeでpretty print形式に整形 | 出力時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ルール取得 | QA_RULES | SELECT | エクスポート対象ルールを取得 |

### テーブル別操作詳細

#### QA_RULES

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT（件数） | COUNT(1) | WHERE (pi_client_name IS NULL OR qaru_client_name = pi_client_name) AND (pi_category IS NULL OR qaru_category = pi_category) | 件数確認 |
| SELECT（クライアント） | qaru_client_name | 同上、GROUP BY | クライアント名ループ |
| SELECT（カテゴリ） | qaru_category | qaru_client_name = client、GROUP BY | カテゴリループ |
| SELECT（ルール） | 全属性 | qaru_client_name = client AND qaru_category = category | ルールループ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| NO_DATA_FOUND | データなし | 条件に合致するルールなし | NULLを返却（正常終了） |
| OTHERS | データベースエラー | SELECT時のエラー | ログに記録して再スロー |

### リトライ仕様

リトライは不要。

## トランザクション仕様

- 参照のみの処理のためトランザクション管理は不要

## パフォーマンス要件

- 実行時間：ルール数に依存（通常数秒以内）
- 大量ルール時のメモリ使用量に注意

## セキュリティ考慮事項

- エクスポートされるJSONにはSQLクエリが含まれるため、取り扱いに注意
- ダウンロードは認証済みユーザーのみ可能

## 備考

- g_version_number変数（パッケージレベル）でJSONバージョンを管理
- g_spool_activeフラグでデバッグ出力を制御

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: パッケージ変数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | qa_export_import_rules_pkg.sql | `src/plsql/pkg/qa_export_import_rules_pkg.sql` | パッケージレベル変数を確認 |

**読解のコツ**:
- **14行目**: g_version_number = 1.1（JSONバージョン）
- **18行目**: g_spool_active（spool時の出力制御フラグ）

#### Step 2: 関数仕様を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | qa_export_import_rules_pkg.sql | `src/plsql/pkg/qa_export_import_rules_pkg.sql` | f_export_rules_table_to_clob関数の仕様と実装を確認 |

**主要処理フロー**:
1. **20-31行目**: 関数仕様宣言
2. **82-239行目**: 関数実装本体
3. **107-111行目**: パラメータのログ記録
4. **113-117行目**: ルール件数確認
5. **119-131行目**: ルールなし時の処理（NULL返却）
6. **133-209行目**: 3重ループでJSON構築
7. **170-195行目**: l_rule_json構築（各属性をput）
8. **215-218行目**: メインJSON構築
9. **220-223行目**: json_serializeでpretty print

#### Step 3: JSON型の使用方法を理解する

**読解のコツ**:
- json_object_t：JSONオブジェクト（キー・値のペア）
- json_array_t：JSON配列
- .put()：値の追加
- .append()：配列への要素追加
- .to_clob()：CLOB型への変換
- json_serialize(...pretty)：整形済み出力

### プログラム呼び出し階層図

```
APEX Rule Files画面
    │
    ├─ qa_apex_app_pkg.p_download_rules_json
    │      │
    │      └─ qa_export_import_rules_pkg.f_export_rules_table_to_clob
    │             │
    │             ├─ SELECT COUNT(*) FROM qa_rules
    │             │
    │             ├─ SELECT qaru_client_name ... GROUP BY
    │             │      └─ カーソルループ
    │             │
    │             ├─ SELECT qaru_category ... GROUP BY
    │             │      └─ カーソルループ
    │             │
    │             ├─ SELECT * FROM qa_rules
    │             │      └─ カーソルループ
    │             │
    │             └─ json_serialize(... pretty)
    │
    └─ GetRuleJsonAttachment（APEXプロセス）
```

### データフロー図

```
[入力]                     [処理]                              [出力]

pi_client_name ─────────▶ f_export_rules_table_to_clob ─────▶ CLOB (JSON)
pi_category                    │
                               │
                               ├─▶ ルール件数確認
                               │
                               ├─▶ 3重ループ
                               │        │
                               │        ├─▶ クライアント名
                               │        │
                               │        ├─▶ カテゴリ
                               │        │
                               │        └─▶ ルール
                               │
                               ├─▶ json_object_t / json_array_t構築
                               │
                               └─▶ json_serialize (pretty)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| qa_export_import_rules_pkg.sql | `src/plsql/pkg/qa_export_import_rules_pkg.sql` | PKG | f_export_rules_table_to_clob関数 |
| qa_apex_app_pkg.sql | `src/plsql/pkg/qa_apex_app_pkg.sql` | PKG | p_download_rules_json（APEX連携） |
| qa_rules.sql | `src/ddl/tab/qa_rules.sql` | DDL | QA_RULESテーブル定義 |
| qa_logger_pkg.sql | `src/plsql/pkg/qa_logger_pkg.sql` | PKG | パラメータログ記録 |
| quasto_rules_template.json | `src/quasto_rules_template.json` | JSON | JSON出力フォーマットのテンプレート |
