# 機能設計書 13-ルールスクリプトエクスポート

## 概要

本ドキュメントは、QUASTOシステムにおける「ルールスクリプトエクスポート」機能の設計内容を記述するものである。本機能は、QA_RULESテーブルのルールを実行可能なSQLスクリプト形式でエクスポートし、SQL*Plusなどで直接実行可能なインポートスクリプトを生成する。

### 本機能の処理概要

本機能は、既存のルール定義を実行可能なPL/SQL匿名ブロック形式のスクリプトに変換する機能である。

**業務上の目的・背景**：JSON形式のエクスポートに加え、SQL*Plusなどのコマンドラインツールで直接実行可能なスクリプト形式が必要となる場合がある。特にCI/CDパイプラインや自動化スクリプトでの利用、APEXアプリケーションを介さないルール移行において有用である。

**機能の利用シーン**：DevOpsパイプラインでのルール自動デプロイ、DBA操作によるルール一括移行、バッチ処理でのルール適用時に利用される。JSON形式が利用できない環境でのルール移行手段としても有効。

**主要な処理内容**：
1. JSON形式のルールエクスポートデータを入力として受け取る
2. 各行を解析しPL/SQL形式に変換
3. SQL文の改行コードをchr(10)に置換
4. DBMS_LOB.APPENDを用いたCLOB連結スクリプトを生成
5. インポート用プロシージャ呼び出しコードを追加

**関連システム・外部連携**：f_export_rules_table_to_clobでエクスポートされたJSON CLOBを入力として使用。生成されたスクリプトは外部のSQL実行環境で使用される。

**権限による制御**：特定のロールや権限による制御はコード上確認されない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | 内部処理 | 直接の画面連携なし（プログラム呼び出し） |

## 機能種別

データ連携（エクスポート）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_clob | CLOB | Yes | JSON形式のルールエクスポートデータ | NULL不可 |

### 入力データソース

- f_export_rules_table_to_clob関数の出力（JSON形式CLOB）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | CLOB | 実行可能SQLスクリプト |

### 出力先

- 関数戻り値として呼び出し元に返却

### 出力スクリプト形式

```sql
set serveroutput on;
declare
  l_clob clob;
begin
dbms_lob.append(l_clob,q'[{JSON行1}]');
dbms_lob.append(l_clob,q'[{JSON行2}]');
...

quasto.qa_export_import_rules_pkg.p_import_clob_to_rules_table(pi_clob => l_clob);
end;
/
```

## 処理フロー

### 処理シーケンス

```
1. パラメータのログ記録
   └─ CLOBの先頭4000バイトをログ
2. スクリプトヘッダー生成
   └─ 'set serveroutput on;' + 'declare' + 'l_clob clob;' + 'begin'
3. JSON行ごとの処理ループ
   └─ REGEXP_SUBSTRで1行ずつ抽出
   └─ SQL行の改行コードをchr(10)に置換
   └─ DBMS_LOB.APPEND文を生成
4. インポート呼び出しコード追加
   └─ qa_export_import_rules_pkg.p_import_clob_to_rules_table呼び出し
5. スクリプトフッター追加
   └─ 'end;' + '/'
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[パラメータログ記録]
    B --> C[スクリプトヘッダー生成]
    C --> D[JSON CLOBを行分割]
    D --> E{各行に対してループ}
    E --> F{SQL行か?}
    F -->|Yes| G[改行をchr10に置換]
    F -->|No| H[そのまま処理]
    G --> I[DBMS_LOB.APPEND文生成]
    H --> I
    I --> E
    E -->|終了| J[インポート呼び出しコード追加]
    J --> K[スクリプトフッター追加]
    K --> L[CLOB返却]
    L --> M[正常終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 改行コード変換 | SQL文内の\nをchr(10)に変換 | "sql"フィールドを含む行 |
| BR-02 | 引用符エスケープ | Oracle q記法（q'[]'）を使用してエスケープ処理を簡略化 | すべての行 |
| BR-03 | インポート呼び出し | QUASTOスキーマのp_import_clob_to_rules_tableを呼び出す | スクリプト末尾 |

### 計算ロジック

- REGEXP_SUBSTRによる行分割: `regexp_substr(pi_clob, '.+', 1, level, 'm')`
- REGEXP_COUNTによる行数取得: `regexp_count(pi_clob, '.+', 1, 'm')`

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| なし | - | - | 本関数はDB操作を行わない（CLOB変換のみ） |

### テーブル別操作詳細

本関数は直接のデータベース操作を行わない。入力としてJSON CLOB（f_export_rules_table_to_clobの出力）を受け取り、実行可能スクリプトに変換して返却する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| NO_DATA_FOUND | システムエラー | 空のCLOBが渡された場合 | 有効なJSONデータを入力 |
| OTHERS | システムエラー | その他のランタイムエラー | ログを確認し問題を特定 |

### リトライ仕様

特になし

## トランザクション仕様

- 本関数はデータベース更新を行わないため、トランザクション制御は不要

## パフォーマンス要件

- REGEXP_SUBSTRによる行単位処理のため、大量行のJSONでは処理時間に注意
- DBMS_LOB.APPENDを多用するため、一時領域の使用量に注意

## セキュリティ考慮事項

- q記法によるSQLインジェクション対策
- qa_logger_pkgによる操作ログの記録

## 備考

- 本機能は「Experimental Function」とコメントされており、実験的機能である
- 生成スクリプト内でquasto.qa_export_import_rules_pkg.p_import_clob_to_rules_tableが呼び出されるため、実行環境にはQUASTOスキーマが必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | qa_export_import_rules_pkg.sql | `src/plsql/pkg/qa_export_import_rules_pkg.sql` | f_export_rules_table_to_clobの出力形式（82-239行目）を理解 |

**読解のコツ**: 入力となるJSON CLOBの構造を先に理解しておくと、変換ロジックが理解しやすい。

#### Step 2: メイン処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | qa_export_import_rules_pkg.sql | `src/plsql/pkg/qa_export_import_rules_pkg.sql` | f_export_rules_to_script_clob関数（242-316行目） |

**主要処理フロー**:
1. **252-255行目**: パラメータログ記録（先頭4000バイト）
2. **257-263行目**: スクリプトヘッダー生成（set serveroutput on; declare ... begin）
3. **264-275行目**: LATERAL句とCONNECT BYによる行分割（REGEXP_SUBSTR使用）
4. **277-294行目**: 各行の処理
   - **277-278行目**: "sql"を含む行の判定
   - **281-283行目**: qa_utils_pkg.f_replace_stringで\nをchr(10)に変換
   - **289-293行目**: DBMS_LOB.APPEND文の生成（q記法使用）
5. **296-299行目**: インポート呼び出しとフッター追加

**読解のコツ**: Oracle q記法（`q'[]'`）は、角括弧内の文字列をそのまま文字列リテラルとして扱う。シングルクォートのエスケープが不要になる。

#### Step 3: 補助関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | f_replace_string関数の動作 |

### プログラム呼び出し階層図

```
呼び出し元（任意）
    │
    └─ qa_export_import_rules_pkg.f_export_rules_to_script_clob
           │
           ├─ qa_logger_pkg.append_param (パラメータログ)
           │
           ├─ qa_utils_pkg.f_replace_string (\n -> chr(10))
           │
           └─ DBMS_LOB.APPEND (CLOB連結)
```

### データフロー図

```
[入力]                          [処理]                                [出力]

JSON CLOB                f_export_rules_to_script_clob
(f_export_rules          ──────────────────────────────▶  実行可能SQLスクリプト
_table_to_clobの                   │                      (CLOB)
出力)                              ▼
                         ┌─────────────────┐
                         │ スクリプトヘッダー │
                         │ set serveroutput │
                         │ declare/begin   │
                         └────────┬────────┘
                                  ▼
                         ┌─────────────────┐
                         │ 行ごとの変換    │
                         │ \n → chr(10)   │
                         │ DBMS_LOB.APPEND │
                         └────────┬────────┘
                                  ▼
                         ┌─────────────────┐
                         │ インポート呼出  │
                         │ end;/          │
                         └─────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| qa_export_import_rules_pkg.sql | `src/plsql/pkg/qa_export_import_rules_pkg.sql` | ソース | f_export_rules_to_script_clob関数を含む |
| qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | ソース | f_replace_string関数（文字列置換） |
| qa_logger_pkg.sql | `src/plsql/pkg/qa_logger_pkg.sql` | ソース | ロギング処理 |
