# 機能設計書 35-テーブル文字列変換

## 概要

本ドキュメントは、QUASTOにおけるvarchar2配列（PL/SQLテーブル）を区切り文字付き文字列に変換する機能について記載する。

### 本機能の処理概要

**業務上の目的・背景**：PL/SQLでは配列（コレクション）形式でデータを扱うことが多いが、ログ出力やSQL文への埋め込み、外部システムとのデータ交換では、区切り文字で連結された単一の文字列形式が必要になることがある。本機能は、varchar2_tab_t型の配列を指定した区切り文字で連結し、単一のVARCHAR2文字列として返却する。これにより、配列データの文字列化を標準化し、コードの再利用性と保守性を向上させる。

**機能の利用シーン**：ルール定義の除外オブジェクトリスト（qaru_exclude_objects）のようなコロン区切りのデータを扱う際や、複数の値をログに一括出力する際に使用される。また、SQLのLIKE条件やIN句の動的生成にも活用可能。

**主要な処理内容**：
1. varchar2_tab_t型の配列とセパレータを受け取る
2. 入力パラメータのNULLチェックを実施
3. 配列の要素をループで処理
4. 各要素をセパレータで連結して単一文字列を構築
5. 結果文字列を返却

**関連システム・外部連携**：外部システムとの直接連携はない。内部的にqa_logger_pkgを使用してエラーログを記録する。

**権限による制御**：本機能に特別な権限制御はない。パッケージを実行できるユーザーであれば使用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面と直接関連しない内部処理機能 |

## 機能種別

ユーティリティ / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_table | varchar2_tab_t | Yes | 変換対象のvarchar2配列 | NULLは不可、エラー発生 |
| pi_separator | VARCHAR2 | No | 区切り文字 | デフォルト':'、NULLは不可 |

### 入力データソース

呼び出し元からの引数

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | VARCHAR2 | 区切り文字で連結された文字列（最大4000文字） |

### 出力先

呼び出し元に返却（関数の戻り値）

## 処理フロー

### 処理シーケンス

```
1. f_get_table_as_string呼び出し
   └─ pi_table、pi_separatorを受け取る
2. パラメータログ記録
   └─ qa_logger_pkg.append_paramでpi_separatorをログ用パラメータに追加
3. NULLチェック
   └─ pi_tableまたはpi_separatorがNULLの場合、raise_application_error
4. ループ処理
   └─ pi_table.first から pi_table.last まで繰り返し
   └─ 最初の要素はそのまま、2番目以降はセパレータを前置して連結
5. 結果返却
   └─ l_string を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[パラメータログ記録]
    B --> C{pi_table IS NULL OR pi_separator IS NULL?}
    C -->|Yes| D[raise_application_error -20001]
    C -->|No| E[i = pi_table.first]
    E --> F{i <= pi_table.last?}
    F -->|No| G[l_stringを返却]
    F -->|Yes| H{l_string IS NULL?}
    H -->|Yes| I[l_string := pi_table i]
    H -->|No| J[l_string := l_string + separator + pi_table i]
    I --> K[i := i + 1]
    J --> K
    K --> F
    G --> L[終了]
    D --> M[例外発生]
    M --> N[qa_logger_pkg.p_qa_logでエラーログ]
    N --> O[raise]
    O --> P[異常終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-35-01 | デフォルトセパレータ | pi_separatorのデフォルト値は':'（コロン） | pi_separator未指定時 |
| BR-35-02 | NULL禁止 | pi_tableとpi_separatorの両方がNULLの場合はエラー | 常時 |
| BR-35-03 | DETERMINISTIC | 同じ入力に対して常に同じ出力を返す（キャッシュ可能） | 常時 |
| BR-35-04 | 先頭セパレータなし | 結果文字列の先頭にはセパレータを付けない | 常時 |

### 計算ロジック

連結ロジック：
```
結果 = 要素1 + セパレータ + 要素2 + セパレータ + ... + 要素N
例：['A', 'B', 'C'] + ':' → 'A:B:C'
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本機能はデータベース操作を行わない |

### テーブル別操作詳細

本機能はメモリ上の変換処理のみで、データベーステーブルへの直接操作はない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| -20001 | アプリケーションエラー | pi_tableまたはpi_separatorがNULL | 入力パラメータを正しく設定する |

### リトライ仕様

本機能にリトライ仕様はない。エラー発生時は呼び出し元に例外を伝播する。

## トランザクション仕様

本機能はトランザクションに影響を与えない（純粋な変換処理）。

## パフォーマンス要件

- DETERMINISTIC修飾子により、同一入力に対する結果がキャッシュ可能
- 配列サイズに比例した線形時間O(n)で処理
- 文字列連結は効率的に実装されているが、大量要素の場合はメモリ使用量に注意

## セキュリティ考慮事項

- 入力データをそのまま連結するのみで、セキュリティ上の特別な考慮は不要
- SQL文への埋め込みに使用する場合は、呼び出し側でSQLインジェクション対策を行うこと

## 備考

- 逆変換（文字列→配列）はf_get_string_as_table関数（機能No.36）で提供
- varchar2_tab_tはTABLE OF varchar2(4000 char)として定義されている
- 結果文字列がVARCHAR2の最大長（4000バイト）を超える場合のエラーハンドリングは呼び出し側で行う

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

変換対象の配列型を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | varchar2_tab_t.sql | `src/plsql/typ/varchar2_tab_t.sql` | TABLE OF varchar2(4000 char)として定義 |

**読解のコツ**: PL/SQLのネストされたテーブル（Nested Table）型を理解していると把握しやすい。

#### Step 2: 関数の仕様を理解する

関数シグネチャと戻り値を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | f_get_table_as_string仕様部（19-22行目） |

**主要ポイント**:
- **19行目**: 関数名とパラメータ定義
- **21行目**: pi_separatorのデフォルト値':'
- **22行目**: 戻り値VARCHAR2、DETERMINISTIC修飾子

#### Step 3: 関数の実装を理解する

実装の詳細を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | f_get_table_as_string本体（112-149行目） |

**主要処理フロー**:
- **117-118行目**: ローカル変数宣言（c_unit、l_param_list）
- **120行目**: 結果格納用変数l_string宣言
- **122-124行目**: パラメータログ記録（qa_logger_pkg.append_param）
- **126-129行目**: NULLチェックとエラー発生
- **131-139行目**: 配列要素のループ連結処理
- **141行目**: 結果返却
- **142-148行目**: 例外処理ブロック

#### Step 4: エラーハンドリングパターンを理解する

例外処理の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | 例外処理ブロック（142-148行目） |

**例外処理パターン**:
```sql
exception
  when others then
    qa_logger_pkg.p_qa_log(...);
    raise;
```

### プログラム呼び出し階層図

```
f_get_table_as_string (qa_utils_pkg)
    │
    ├─ qa_logger_pkg.append_param (パラメータログ記録)
    │
    ├─ raise_application_error (NULLエラー時)
    │
    └─ [エラー時] qa_logger_pkg.p_qa_log (エラーログ記録)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

pi_table
  ┌─────┐
  │要素1│──────┐
  ├─────┤      │
  │要素2│──────┼──▶ f_get_table_as_string ───▶ '要素1:要素2:要素3'
  ├─────┤      │              │
  │要素3│──────┘              │
  └─────┘                     ▼
                    ┌──────────────────┐
pi_separator ───▶   │ ループで連結処理  │
  ':'               └──────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | ソース | テーブル文字列変換機能を含むユーティリティパッケージ |
| varchar2_tab_t.sql | `src/plsql/typ/varchar2_tab_t.sql` | ソース | 入力配列の型定義 |
| qa_logger_pkg.sql | `src/plsql/pkg/qa_logger_pkg.sql` | ソース | エラーログ記録用パッケージ |
