# 機能設計書 36-文字列テーブル変換

## 概要

本ドキュメントは、QUASTOにおける区切り文字付き文字列をvarchar2配列（PL/SQLテーブル）に変換する機能について記載する。

### 本機能の処理概要

**業務上の目的・背景**：データベースに保存されたデータや外部システムから受け取ったデータは、しばしば区切り文字で連結された文字列形式で提供される。PL/SQLで個々の要素を処理するには、この文字列を配列に分割する必要がある。本機能は、指定した区切り文字で文字列を分割し、varchar2_tab_t型の配列として返却する。これにより、ルール定義の除外オブジェクトリスト（qaru_exclude_objects）などのコロン区切りデータを効率的に処理できる。

**機能の利用シーン**：ルール定義のqaru_exclude_objects列（コロン区切りのオブジェクト名リスト）をパースする際や、CSV形式のデータを個別の値に分解する際に使用される。また、動的に生成されたIN句の値リストを配列に変換する際にも活用可能。

**主要な処理内容**：
1. 区切り文字付き文字列とセパレータを受け取る
2. 入力パラメータのNULLチェックを実施
3. 階層クエリ（CONNECT BY）とREGEXP_SUBSTRを使用して文字列を分割
4. 分割された各要素をvarchar2_tab_t配列に追加
5. 結果配列を返却

**関連システム・外部連携**：外部システムとの直接連携はない。内部的にqa_logger_pkgを使用してエラーログを記録する。

**権限による制御**：本機能に特別な権限制御はない。パッケージを実行できるユーザーであれば使用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面と直接関連しない内部処理機能 |

## 機能種別

ユーティリティ / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_string | VARCHAR2 | Yes | 変換対象の区切り文字付き文字列 | NULLは不可、エラー発生 |
| pi_separator | VARCHAR2 | No | 区切り文字 | デフォルト':'、NULLは不可 |

### 入力データソース

呼び出し元からの引数（データベースカラム値、ユーザー入力等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | varchar2_tab_t | 分割された文字列の配列 |

### 出力先

呼び出し元に返却（関数の戻り値）

## 処理フロー

### 処理シーケンス

```
1. f_get_string_as_table呼び出し
   └─ pi_string、pi_separatorを受け取る
2. パラメータログ記録
   └─ qa_logger_pkg.append_paramでパラメータをログ用に追加
3. NULLチェック
   └─ pi_stringまたはpi_separatorがNULLの場合、raise_application_error
4. 配列初期化
   └─ l_table := new varchar2_tab_t()
5. 階層クエリ実行
   └─ CONNECT BY + REGEXP_SUBSTRで文字列を分割
6. ループ処理
   └─ 分割された各要素を配列に追加
7. 結果返却
   └─ l_table を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[パラメータログ記録]
    B --> C{pi_string IS NULL OR pi_separator IS NULL?}
    C -->|Yes| D[raise_application_error -20001]
    C -->|No| E[l_table := new varchar2_tab_t]
    E --> F[階層クエリ実行]
    F --> G{結果セットあり?}
    G -->|No| H[l_tableを返却]
    G -->|Yes| I[l_table.extend]
    I --> J[l_table last := column_value]
    J --> K{次の行あり?}
    K -->|Yes| I
    K -->|No| H
    H --> L[終了]
    D --> M[例外発生]
    M --> N[qa_logger_pkg.p_qa_logでエラーログ]
    N --> O[raise]
    O --> P[異常終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-36-01 | デフォルトセパレータ | pi_separatorのデフォルト値は':'（コロン） | pi_separator未指定時 |
| BR-36-02 | NULL禁止 | pi_stringとpi_separatorの両方がNULLの場合はエラー | 常時 |
| BR-36-03 | DETERMINISTIC | 同じ入力に対して常に同じ出力を返す（キャッシュ可能） | 常時 |
| BR-36-04 | 空要素スキップ | 連続するセパレータ間の空要素はスキップされる | 常時 |

### 計算ロジック

分割ロジック（REGEXP_SUBSTR使用）：
```sql
SELECT regexp_substr(pi_string, '[^' || pi_separator || ']+', 1, level) AS column_value
  FROM dual
CONNECT BY regexp_substr(pi_string, '[^' || pi_separator || ']+', 1, level) IS NOT NULL
```

例：'A:B:C' + ':' → ['A', 'B', 'C']

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | DUAL | SELECT | 階層クエリのベーステーブルとして使用 |

### テーブル別操作詳細

#### DUAL（Oracle組み込み表）

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | - | CONNECT BY句で階層的に行生成 | 参照のみ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| -20001 | アプリケーションエラー | pi_stringまたはpi_separatorがNULL | 入力パラメータを正しく設定する |

### リトライ仕様

本機能にリトライ仕様はない。エラー発生時は呼び出し元に例外を伝播する。

## トランザクション仕様

本機能はトランザクションに影響を与えない（SELECT文のみ使用する参照処理）。

## パフォーマンス要件

- DETERMINISTIC修飾子により、同一入力に対する結果がキャッシュ可能
- REGEXP_SUBSTRは正規表現処理のため、大量データでは性能に注意
- CONNECT BYによる階層クエリは要素数に比例した処理時間

## セキュリティ考慮事項

- 入力データをそのまま分割するのみで、セキュリティ上の特別な考慮は不要
- セパレータに正規表現のメタ文字を使用する場合は、予期しない動作に注意

## 備考

- 逆変換（配列→文字列）はf_get_table_as_string関数（機能No.35）で提供
- varchar2_tab_tはTABLE OF varchar2(4000 char)として定義されている
- 連続するセパレータがある場合（例：'A::B'）、空要素はスキップされる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

出力される配列型を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | varchar2_tab_t.sql | `src/plsql/typ/varchar2_tab_t.sql` | TABLE OF varchar2(4000 char)として定義 |

**読解のコツ**: PL/SQLのネストされたテーブル（Nested Table）型を理解していると把握しやすい。EXTEND、LASTメソッドの動作を理解することが重要。

#### Step 2: 関数の仕様を理解する

関数シグネチャと戻り値を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | f_get_string_as_table仕様部（30-33行目） |

**主要ポイント**:
- **30行目**: 関数名とパラメータ定義
- **32行目**: pi_separatorのデフォルト値':'
- **33行目**: 戻り値varchar2_tab_t、DETERMINISTIC修飾子

#### Step 3: 関数の実装を理解する

実装の詳細を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | f_get_string_as_table本体（151-189行目） |

**主要処理フロー**:
- **156-157行目**: ローカル変数宣言（c_unit、l_param_list）
- **159行目**: 結果格納用配列l_tableを初期化（new varchar2_tab_t()）
- **161-165行目**: パラメータログ記録（qa_logger_pkg.append_param）
- **167-170行目**: NULLチェックとエラー発生
- **172-179行目**: 階層クエリによる文字列分割とループ処理
- **181行目**: 結果返却
- **182-188行目**: 例外処理ブロック

#### Step 4: 階層クエリのロジックを理解する

CONNECT BY + REGEXP_SUBSTRの動作を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | 階層クエリ部分（172-175行目） |

**階層クエリの解説**:
```sql
SELECT regexp_substr(pi_string, '[^' || pi_separator || ']+', 1, level) AS column_value
  FROM dual
CONNECT BY regexp_substr(pi_string, '[^' || pi_separator || ']+', 1, level) IS NOT NULL
```
- `[^' || pi_separator || ']+`: セパレータ以外の1文字以上にマッチ
- `1, level`: 1文字目から開始し、level番目の出現を取得
- `CONNECT BY ... IS NOT NULL`: マッチがある限り階層を深く掘る

### プログラム呼び出し階層図

```
f_get_string_as_table (qa_utils_pkg)
    │
    ├─ qa_logger_pkg.append_param (パラメータログ記録)
    │
    ├─ raise_application_error (NULLエラー時)
    │
    ├─ 階層クエリ (DUAL + CONNECT BY)
    │      │
    │      └─ REGEXP_SUBSTR (正規表現分割)
    │
    └─ [エラー時] qa_logger_pkg.p_qa_log (エラーログ記録)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

pi_string                                                   l_table
'A:B:C'         ───▶ f_get_string_as_table ───▶         ┌─────┐
                              │                          │  A  │
pi_separator                  │                          ├─────┤
':'             ───▶          ▼                          │  B  │
                    ┌──────────────────┐                 ├─────┤
                    │ CONNECT BY +     │                 │  C  │
                    │ REGEXP_SUBSTR    │                 └─────┘
                    └──────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | ソース | 文字列テーブル変換機能を含むユーティリティパッケージ |
| varchar2_tab_t.sql | `src/plsql/typ/varchar2_tab_t.sql` | ソース | 出力配列の型定義 |
| qa_logger_pkg.sql | `src/plsql/pkg/qa_logger_pkg.sql` | ソース | エラーログ記録用パッケージ |
