# 機能設計書 37-文字列統一化

## 概要

本ドキュメントは、QUASTOにおける文字列を正規表現に基づいて統一されたフォーマットに変換する機能について記載する。

### 本機能の処理概要

**業務上の目的・背景**：システム内でファイル名、識別子、URLなどの文字列を統一されたフォーマットで扱う必要がある。ユーザー入力や外部システムからのデータには、スペース、特殊文字、大文字小文字の混在などの不統一が含まれることが多い。本機能は、正規表現ベースのルールに従って文字列を正規化し、システム全体で一貫した命名規則を適用する。これにより、ファイル名の衝突回避やデータの検索・比較処理の信頼性が向上する。

**機能の利用シーン**：JSONエクスポートファイル名の生成（p_download_rules_json）で、クライアント名をファイル名に適した形式に変換する際に使用される。また、データベースオブジェクト名の正規化、ユーザー入力の標準化などにも活用可能。

**主要な処理内容**：
1. 変換対象の文字列と変換オプションを受け取る
2. 入力パラメータの妥当性チェックを実施
3. スペースを置換文字に変換（オプション）
4. 正規表現にマッチしない文字を置換文字に変換
5. 大文字/小文字への変換（オプション）
6. 統一された文字列を返却

**関連システム・外部連携**：外部システムとの直接連携はない。内部的にqa_logger_pkgを使用してエラーログを記録する。

**権限による制御**：本機能に特別な権限制御はない。パッケージを実行できるユーザーであれば使用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面と直接関連しない内部処理機能 |

## 機能種別

ユーティリティ / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_string | VARCHAR2 | Yes | 変換対象の文字列 | NULLは不可、エラー発生 |
| pi_regexp | VARCHAR2 | No | 許可しない文字のパターン（否定形） | デフォルト'[^a-zA-Z0-9_]'、NULLは不可 |
| pi_replace_blank_space | VARCHAR2 | No | スペースを置換するか | デフォルト'Y'、'Y'または'N'のみ |
| pi_replacement_char | VARCHAR2 | No | 置換文字 | デフォルト'_'、NULLは不可 |
| pi_transform_case | VARCHAR2 | No | 大文字/小文字変換 | NULL、'l'（小文字）、'u'（大文字） |

### 入力データソース

呼び出し元からの引数（クライアント名、ファイル名、識別子等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | VARCHAR2 | 統一されたフォーマットの文字列（最大4000文字） |

### 出力先

呼び出し元に返却（関数の戻り値）

## 処理フロー

### 処理シーケンス

```
1. f_get_unified_string呼び出し
   └─ pi_string、各オプションを受け取る
2. パラメータログ記録
   └─ qa_logger_pkg.append_paramでパラメータをログ用に追加
3. 入力妥当性チェック
   └─ NULL、無効な値の場合はraise_application_error
4. スペース置換（pi_replace_blank_space = 'Y'の場合）
   └─ REPLACE(pi_string, ' ', pi_replacement_char)
5. 正規表現置換
   └─ REGEXP_REPLACE(l_string, pi_regexp, pi_replacement_char)
6. 大文字/小文字変換（pi_transform_caseが指定されている場合）
   └─ LOWER() または UPPER()
7. 結果返却
   └─ l_string を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[パラメータログ記録]
    B --> C{パラメータ妥当性チェック}
    C -->|無効| D[raise_application_error -20001]
    C -->|有効| E{pi_replace_blank_space = 'Y'?}
    E -->|Yes| F[スペースを置換文字に変換]
    E -->|No| G[スキップ]
    F --> H[REGEXP_REPLACEで非許可文字を置換]
    G --> H
    H --> I{pi_transform_case = 'l'?}
    I -->|Yes| J[LOWER 小文字に変換]
    I -->|No| K{pi_transform_case = 'u'?}
    K -->|Yes| L[UPPER 大文字に変換]
    K -->|No| M[変換なし]
    J --> N[l_stringを返却]
    L --> N
    M --> N
    N --> O[終了]
    D --> P[例外発生]
    P --> Q[qa_logger_pkg.p_qa_logでエラーログ]
    Q --> R[raise]
    R --> S[異常終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-37-01 | デフォルト正規表現 | '[^a-zA-Z0-9_]'（英数字とアンダースコア以外）を置換 | pi_regexp未指定時 |
| BR-37-02 | スペース置換優先 | スペース置換は正規表現置換より先に実行 | pi_replace_blank_space = 'Y'時 |
| BR-37-03 | 大文字/小文字変換 | 'l'で小文字、'u'で大文字に変換 | pi_transform_case指定時 |
| BR-37-04 | DETERMINISTIC | 同じ入力に対して常に同じ出力を返す | 常時 |

### 計算ロジック

変換ロジック（順序重要）：
```
1. スペース置換: 'Hello World' → 'Hello_World'（pi_replace_blank_space = 'Y'の場合）
2. 正規表現置換: 'Hello_World!' → 'Hello_World_'（'!'が'_'に）
3. 大文字/小文字: 'Hello_World_' → 'hello_world_'（pi_transform_case = 'l'の場合）
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本機能はデータベース操作を行わない |

### テーブル別操作詳細

本機能はメモリ上の変換処理のみで、データベーステーブルへの直接操作はない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| -20001 | アプリケーションエラー | pi_stringがNULL | 入力文字列を指定する |
| -20001 | アプリケーションエラー | pi_regexpがNULL | 正規表現を指定する |
| -20001 | アプリケーションエラー | pi_replace_blank_spaceが'Y'/'N'以外 | 有効な値を指定する |
| -20001 | アプリケーションエラー | pi_replacement_charがNULL | 置換文字を指定する |
| -20001 | アプリケーションエラー | pi_transform_caseが'l'/'u'/NULL以外 | 有効な値を指定する |

### リトライ仕様

本機能にリトライ仕様はない。エラー発生時は呼び出し元に例外を伝播する。

## トランザクション仕様

本機能はトランザクションに影響を与えない（純粋な変換処理）。

## パフォーマンス要件

- DETERMINISTIC修飾子により、同一入力に対する結果がキャッシュ可能
- REGEXP_REPLACEは正規表現処理のため、複雑なパターンでは性能に注意
- 通常の文字列変換は数マイクロ秒で完了

## セキュリティ考慮事項

- 入力データを変換するのみで、セキュリティ上の特別な考慮は不要
- ファイルパスやURL生成に使用する場合は、パストラバーサル攻撃に注意

## 備考

- p_download_rules_json（機能No.28）で、クライアント名をファイル名に変換する際に使用されている
- デフォルト設定では、英数字とアンダースコアのみを許可し、それ以外は'_'に置換
- 連続する置換文字は圧縮されない（例：'A  B' → 'A__B'）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 関数の仕様を理解する

関数シグネチャと各パラメータの役割を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | f_get_unified_string仕様部（44-51行目） |

**主要ポイント**:
- **44行目**: 関数名定義
- **46行目**: pi_stringの定義
- **47行目**: pi_regexpのデフォルト値'[^a-zA-Z0-9_]'
- **48行目**: pi_replace_blank_spaceのデフォルト値'Y'
- **49行目**: pi_replacement_charのデフォルト値'_'
- **50行目**: pi_transform_caseのデフォルト値NULL
- **51行目**: DETERMINISTIC修飾子

#### Step 2: 関数の実装を理解する

実装の詳細を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | f_get_unified_string本体（191-244行目） |

**主要処理フロー**:
- **199-200行目**: ローカル変数宣言（c_unit、l_param_list）
- **202行目**: 結果格納用変数l_string宣言
- **204-214行目**: パラメータログ記録（5つのパラメータ）
- **216-220行目**: 入力妥当性チェック（複合条件）
- **222-226行目**: スペース置換と正規表現置換
- **228-234行目**: 大文字/小文字変換
- **236行目**: 結果返却
- **237-243行目**: 例外処理ブロック

#### Step 3: 変換ロジックの詳細を理解する

REGEXP_REPLACEとCASE式の動作を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | 変換ロジック部分（222-234行目） |

**変換ロジックの解説**:
```sql
-- スペース置換 + 正規表現置換
l_string := regexp_replace(
  CASE
    WHEN pi_replace_blank_space = 'Y'
    THEN replace(pi_string, ' ', pi_replacement_char)
    ELSE pi_string
  END,
  pi_regexp,
  pi_replacement_char
);

-- 大文字/小文字変換
IF lower(pi_transform_case) = 'l' THEN
  l_string := lower(l_string);
ELSIF lower(pi_transform_case) = 'u' THEN
  l_string := upper(l_string);
END IF;
```

#### Step 4: 使用例を確認する

実際の使用パターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | qa_apex_app_pkg.sql | `src/plsql/pkg/qa_apex_app_pkg.sql` | p_download_rules_jsonでの使用（181行目） |

**使用例**:
```sql
l_client_name_unified := qa_utils_pkg.f_get_unified_string(
  pi_string => pi_client_name,
  pi_transform_case => 'l'
);
-- 結果をファイル名に使用: 'export_rules_' || l_client_name_unified || '_' || timestamp || '.json'
```

### プログラム呼び出し階層図

```
f_get_unified_string (qa_utils_pkg)
    │
    ├─ qa_logger_pkg.append_param (パラメータログ記録)
    │
    ├─ raise_application_error (バリデーションエラー時)
    │
    ├─ REPLACE (スペース置換)
    │
    ├─ REGEXP_REPLACE (正規表現置換)
    │
    ├─ LOWER / UPPER (大文字/小文字変換)
    │
    └─ [エラー時] qa_logger_pkg.p_qa_log (エラーログ記録)
```

### データフロー図

```
[入力]                          [処理]                           [出力]

pi_string
'Hello World!'  ───▶ f_get_unified_string ───▶ 'hello_world_'
                              │
pi_regexp                     │
'[^a-zA-Z0-9_]'               │
                              ▼
pi_replace_blank_space   ┌──────────────────────┐
'Y'                      │ 1. スペース→'_'     │
                         │ 2. 特殊文字→'_'     │
pi_replacement_char      │ 3. 小文字変換        │
'_'                      └──────────────────────┘

pi_transform_case
'l'
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | ソース | 文字列統一化機能を含むユーティリティパッケージ |
| qa_apex_app_pkg.sql | `src/plsql/pkg/qa_apex_app_pkg.sql` | ソース | 使用例を含むパッケージ（ファイル名生成） |
| qa_logger_pkg.sql | `src/plsql/pkg/qa_logger_pkg.sql` | ソース | エラーログ記録用パッケージ |
