# 機能設計書 38-文字列置換

## 概要

本ドキュメントは、QUASTOにおけるCLOB内の文字列を置換する機能について記載する。

### 本機能の処理概要

**業務上の目的・背景**：大量のテキストデータ（CLOB型）内の特定の文字列を置換する必要がある場面は多い。標準のREPLACE関数はVARCHAR2型に制限があるが、本機能はCLOB型のデータに対して文字列置換を行える。テンプレートの変数置換、動的SQLの生成、ログメッセージのカスタマイズなど、様々な場面で活用できる。

**機能の利用シーン**：大きなSQLスクリプトやJSON/XMLデータ内の特定のプレースホルダーを実際の値に置換する際に使用される。また、ルールエクスポート機能でのテンプレート処理やコード生成処理にも活用可能。

**主要な処理内容**：
1. 置換対象のCLOB文字列、検索文字列、置換文字列を受け取る
2. INSTR関数で検索文字列の位置を特定
3. 検索文字列が見つかった場合、SUBSTR関数で文字列を分割し置換
4. 検索文字列が見つからない場合、元の文字列をそのまま返却

**関連システム・外部連携**：外部システムとの直接連携はない。内部的にqa_logger_pkgを使用してエラーログを記録する。

**権限による制御**：本機能に特別な権限制御はない。パッケージを実行できるユーザーであれば使用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面と直接関連しない内部処理機能 |

## 機能種別

ユーティリティ / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_source_string | CLOB | Yes | 置換対象の元文字列 | - |
| pi_search_string | VARCHAR2 | Yes | 検索する文字列 | - |
| pi_replace_string | CLOB | Yes | 置換後の文字列 | - |

### 入力データソース

呼び出し元からの引数（CLOB型のデータ、テンプレート等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | CLOB | 置換後の文字列（検索文字列が見つからない場合は元の文字列） |

### 出力先

呼び出し元に返却（関数の戻り値）

## 処理フロー

### 処理シーケンス

```
1. f_replace_string呼び出し
   └─ pi_source_string、pi_search_string、pi_replace_stringを受け取る
2. ログ用に先頭4000文字を切り出し
   └─ DBMS_LOB.SUBSTRで処理
3. パラメータログ記録
   └─ qa_logger_pkg.append_paramでパラメータをログ用に追加
4. 検索文字列の位置を特定
   └─ INSTR(pi_source_string, pi_search_string)
5. 位置判定
   └─ 位置 > 0 の場合は置換処理、それ以外は元文字列を返却
6. 置換処理
   └─ 前半部分 + 置換文字列 + 後半部分
7. 結果返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[CLOB先頭4000文字を切り出し]
    B --> C[パラメータログ記録]
    C --> D[l_pos := INSTR]
    D --> E{l_pos > 0?}
    E -->|Yes| F[前半部分を切り出し]
    F --> G[置換文字列を連結]
    G --> H[後半部分を連結]
    H --> I[結果CLOBを返却]
    E -->|No| J[pi_source_stringをそのまま返却]
    I --> K[終了]
    J --> K

    subgraph 例外処理
    L[NO_DATA_FOUND] --> M[エラーログ記録]
    N[OTHERS] --> O[エラーログ記録]
    M --> P[raise]
    O --> P
    end
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-38-01 | 単一置換 | 最初に見つかった検索文字列のみ置換（全置換ではない） | 常時 |
| BR-38-02 | 検索失敗時 | 検索文字列が見つからない場合は元の文字列を返却 | 検索文字列が存在しない場合 |
| BR-38-03 | DETERMINISTIC | 同じ入力に対して常に同じ出力を返す | 常時 |
| BR-38-04 | ログ制限 | パラメータログには先頭4000文字のみ記録 | 常時 |

### 計算ロジック

置換ロジック：
```
結果 = SUBSTR(pi_source_string, 1, l_pos - 1)  -- 検索文字列の前の部分
     || pi_replace_string                        -- 置換文字列
     || SUBSTR(pi_source_string, l_pos + LENGTH(pi_search_string))  -- 検索文字列の後の部分
```

例：
- 入力：'Hello World', 'World', 'Oracle'
- 結果：'Hello Oracle'

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本機能はデータベース操作を行わない |

### テーブル別操作詳細

本機能はメモリ上の変換処理のみで、データベーステーブルへの直接操作はない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| NO_DATA_FOUND | 例外 | 置換対象データが存在しない場合 | 入力データを確認 |
| OTHERS | 例外 | その他のエラー | エラーログを確認 |

### リトライ仕様

本機能にリトライ仕様はない。エラー発生時は呼び出し元に例外を伝播する。

## トランザクション仕様

本機能はトランザクションに影響を与えない（純粋な変換処理）。

## パフォーマンス要件

- DETERMINISTIC修飾子により、同一入力に対する結果がキャッシュ可能
- CLOB操作のため、大きなデータでは性能に注意
- INSTR、SUBSTRはOracleの標準関数で最適化済み

## セキュリティ考慮事項

- 入力データを変換するのみで、セキュリティ上の特別な考慮は不要
- SQLインジェクション対策として、動的SQL生成時は注意が必要

## 備考

- 本機能は最初に見つかった検索文字列のみを置換する（全置換ではない）
- 全置換が必要な場合は、ループで呼び出すか、REGEXP_REPLACEの使用を検討
- CLOBのサイズ制限（最大4GB-1バイト）に注意

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 関数の仕様を理解する

関数シグネチャと各パラメータの役割を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | f_replace_string仕様部（61-66行目） |

**主要ポイント**:
- **61行目**: 関数名定義
- **63行目**: pi_source_stringはCLOB型
- **64行目**: pi_search_stringはVARCHAR2型
- **65行目**: pi_replace_stringはCLOB型
- **66行目**: 戻り値CLOB、DETERMINISTIC修飾子

#### Step 2: 関数の実装を理解する

実装の詳細を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | f_replace_string本体（246-296行目） |

**主要処理フロー**:
- **253-254行目**: ローカル変数宣言（c_unit、l_param_list）
- **256行目**: 検索位置格納用変数l_pos宣言
- **257-258行目**: ログ用の切り詰め変数宣言
- **260-261行目**: DBMS_LOB.SUBSTRで先頭4000文字を切り出し
- **263-269行目**: パラメータログ記録
- **272-273行目**: INSTR関数で検索文字列の位置を特定
- **274-280行目**: 検索成功時の置換処理
- **282行目**: 検索失敗時は元文字列を返却
- **283-295行目**: 例外処理ブロック

#### Step 3: 置換ロジックの詳細を理解する

SUBSTR関数による文字列操作を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | 置換処理部分（276-279行目） |

**置換ロジックの解説**:
```sql
RETURN SUBSTR(pi_source_string, 1, l_pos - 1)  -- 検索文字列の前の部分
    || pi_replace_string                        -- 置換文字列
    || SUBSTR(pi_source_string, l_pos + LENGTH(pi_search_string));  -- 検索文字列の後の部分
```

- `SUBSTR(..., 1, l_pos - 1)`: 先頭から検索位置の直前まで
- `pi_replace_string`: 置換文字列を挿入
- `SUBSTR(..., l_pos + LENGTH(...))`: 検索文字列の直後から末尾まで

#### Step 4: 例外処理パターンを理解する

エラーハンドリングの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | 例外処理ブロック（283-295行目） |

**例外処理パターン**:
- NO_DATA_FOUND: 置換対象データが存在しない場合
- OTHERS: その他の全ての例外

### プログラム呼び出し階層図

```
f_replace_string (qa_utils_pkg)
    │
    ├─ DBMS_LOB.SUBSTR (ログ用CLOBの切り詰め)
    │
    ├─ qa_logger_pkg.append_param (パラメータログ記録)
    │
    ├─ INSTR (検索位置特定)
    │
    ├─ SUBSTR (文字列切り出し) × 2
    │
    └─ [エラー時] qa_logger_pkg.p_qa_log (エラーログ記録)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

pi_source_string
'Hello World'      ───▶ f_replace_string ───▶ 'Hello Oracle'
                              │
pi_search_string              │
'World'            ───▶       │
                              ▼
pi_replace_string   ┌────────────────────────┐
'Oracle'            │ 1. INSTR で位置特定    │
                    │ 2. SUBSTR で分割       │
                    │ 3. || で連結           │
                    └────────────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | ソース | 文字列置換機能を含むユーティリティパッケージ |
| qa_logger_pkg.sql | `src/plsql/pkg/qa_logger_pkg.sql` | ソース | エラーログ記録用パッケージ |
