# 機能設計書 39-DBMS出力

## 概要

本ドキュメントは、QUASTOにおけるvarchar2/CLOBをDBMS_OUTPUTに出力する機能について記載する。

### 本機能の処理概要

**業務上の目的・背景**：開発・デバッグ時に、処理結果やデータ内容をコンソールに出力して確認したい場面は多い。標準のDBMS_OUTPUT.PUT_LINEは1行あたり32767バイトの制限があり、また大きなCLOBを出力する際には分割処理が必要となる。本機能は、VARCHAR2とCLOB型の両方に対応し、大きなデータも適切に分割して出力するユーティリティを提供する。

**機能の利用シーン**：開発中のデバッグ、スクリプト実行結果の確認、JSON/XMLデータの表示、ルールのSQL文の確認など、様々な場面でコンソール出力が必要な際に使用される。特に、生成されたSQLスクリプトやエクスポートデータの内容確認に有用。

**主要な処理内容**：
1. VARCHAR2版：入力文字列をそのままDBMS_OUTPUT.PUT_LINEで出力
2. CLOB版：バッファサイズ（32767バイト）単位でCLOBを分割して出力
3. 出力前にDBMS_OUTPUT.ENABLEでバッファサイズを拡張
4. エラー発生時はqa_logger_pkgでログを記録

**関連システム・外部連携**：Oracle標準のDBMS_OUTPUTパッケージを使用する。出力結果はSQL*Plus、SQL Developerなどのクライアントツールで確認可能。

**権限による制御**：本機能に特別な権限制御はない。パッケージを実行できるユーザーであれば使用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面と直接関連しない内部処理機能 |

## 機能種別

ユーティリティ / デバッグ支援

## 入力仕様

### 入力パラメータ

#### VARCHAR2版

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_string | VARCHAR2 | Yes | 出力する文字列 | - |

#### CLOB版

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_string | CLOB | Yes | 出力するCLOBデータ | - |

### 入力データソース

呼び出し元からの引数（デバッグデータ、生成されたSQL文等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | - | 戻り値なし（プロシージャ） |

### 出力先

DBMS_OUTPUTバッファ（コンソール出力）

## 処理フロー

### 処理シーケンス（VARCHAR2版）

```
1. p_print_to_dbms_output呼び出し
   └─ pi_string（VARCHAR2）を受け取る
2. パラメータログ記録
   └─ qa_logger_pkg.append_paramでパラメータを記録
3. コンソール出力
   └─ DBMS_OUTPUT.PUT_LINE(pi_string)
4. 処理完了
```

### 処理シーケンス（CLOB版）

```
1. p_print_to_dbms_output呼び出し
   └─ pi_string（CLOB）を受け取る
2. ローカル変数初期化
   └─ l_offset := 1, l_step := 32767
3. パラメータログ記録
   └─ qa_logger_pkg.append_paramで先頭4000文字を記録
4. バッファ有効化
   └─ DBMS_OUTPUT.ENABLE(buffer_size => 10000000)
5. ループ処理
   └─ DBMS_LOB.GETLENGTHまでl_step単位で出力
   └─ DBMS_OUTPUT.PUT_LINE(DBMS_LOB.SUBSTR(...))
6. 処理完了
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{入力型は?}
    B -->|VARCHAR2| C[パラメータログ記録]
    B -->|CLOB| D[l_offset = 1, l_step = 32767]
    C --> E[DBMS_OUTPUT.PUT_LINE]
    E --> F[終了]
    D --> G[パラメータログ記録]
    G --> H[DBMS_OUTPUT.ENABLE 10MB]
    H --> I{l_offset <= DBMS_LOB.GETLENGTH?}
    I -->|No| F
    I -->|Yes| J[DBMS_OUTPUT.PUT_LINE DBMS_LOB.SUBSTR]
    J --> K[l_offset += l_step]
    K --> I

    subgraph 例外処理
    L[NO_DATA_FOUND] --> M[エラーログ記録]
    N[OTHERS] --> O[エラーログ記録]
    M --> P[raise]
    O --> P
    end
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-39-01 | 分割出力 | CLOBは32767バイト単位で分割して出力 | CLOB版 |
| BR-39-02 | バッファ拡張 | CLOB出力前にバッファサイズを10MBに設定 | CLOB版 |
| BR-39-03 | DETERMINISTIC | 同じ入力に対して常に同じ出力を行う | 常時 |
| BR-39-04 | ログ制限 | パラメータログには先頭4000文字のみ記録 | CLOB版 |

### 計算ロジック

CLOB分割出力ロジック：
```
l_offset = 1
l_step = 32767
WHILE l_offset <= DBMS_LOB.GETLENGTH(pi_string) LOOP
  出力: DBMS_LOB.SUBSTR(pi_string, l_step, l_offset)
  l_offset := l_offset + l_step
END LOOP
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本機能はデータベース操作を行わない |

### テーブル別操作詳細

本機能はDBMS_OUTPUTへの出力のみで、データベーステーブルへの直接操作はない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| NO_DATA_FOUND | 例外 | 出力対象データが存在しない場合 | 入力データを確認 |
| OTHERS | 例外 | その他のエラー（バッファオーバーフロー等） | エラーログを確認 |

### リトライ仕様

本機能にリトライ仕様はない。エラー発生時は呼び出し元に例外を伝播する。

## トランザクション仕様

本機能はトランザクションに影響を与えない（DBMS_OUTPUTへの出力のみ）。

## パフォーマンス要件

- VARCHAR2版は即座に完了
- CLOB版は大きなデータでも分割出力により安定動作
- バッファサイズ10MBにより、大量出力にも対応

## セキュリティ考慮事項

- 出力データに機密情報が含まれないよう注意
- 本番環境では不要なデバッグ出力を無効化することを推奨
- DBMS_OUTPUTの出力はクライアントツールのコンソールに表示される

## 備考

- DBMS_OUTPUTはサーバーサイドでバッファリングされ、クライアントが取得するまで出力されない
- SQL*Plusでは`SET SERVEROUTPUT ON`が必要
- CLOB版の分割サイズ（32767バイト）はDBMS_OUTPUT.PUT_LINEの最大長に基づく

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: プロシージャの仕様を理解する

2つのオーバーロード版の仕様を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | p_print_to_dbms_output（VARCHAR2版）仕様部（73-76行目） |
| 1-2 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | p_print_to_dbms_output（CLOB版）仕様部（83-86行目） |

**主要ポイント**:
- **73行目**: VARCHAR2版のプロシージャ定義
- **76行目**: DETERMINISTIC修飾子
- **83行目**: CLOB版のプロシージャ定義
- **86行目**: DETERMINISTIC修飾子

#### Step 2: VARCHAR2版の実装を理解する

シンプルな実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | VARCHAR2版本体（298-324行目） |

**主要処理フロー**:
- **303-304行目**: ローカル変数宣言（c_unit、l_param_list）
- **306-308行目**: パラメータログ記録
- **310行目**: DBMS_OUTPUT.PUT_LINE(pi_string) - メイン処理
- **311-323行目**: 例外処理ブロック

#### Step 3: CLOB版の実装を理解する

分割出力の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | CLOB版本体（326-365行目） |

**主要処理フロー**:
- **331-332行目**: ローカル変数宣言（c_unit、l_param_list）
- **334行目**: l_offset（開始位置）= 1
- **335行目**: l_step（分割サイズ）= 32767
- **336行目**: l_clob_string（ログ用切り詰め）
- **339-342行目**: パラメータログ記録（先頭4000文字のみ）
- **344行目**: DBMS_OUTPUT.ENABLE(buffer_size => 10000000) - 10MBに拡張
- **345-351行目**: ループで分割出力
- **352-364行目**: 例外処理ブロック

#### Step 4: 分割出力ロジックの詳細を理解する

ループ処理の動作を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | ループ処理部分（345-351行目） |

**ループ処理の解説**:
```sql
LOOP
  EXIT WHEN l_offset > dbms_lob.getlength(pi_string);  -- 終了条件
  dbms_output.put_line(dbms_lob.substr(pi_string, l_step, l_offset));  -- 分割出力
  l_offset := l_offset + l_step;  -- 次の開始位置
END LOOP;
```

### プログラム呼び出し階層図

```
p_print_to_dbms_output (VARCHAR2版)
    │
    ├─ qa_logger_pkg.append_param (パラメータログ記録)
    │
    ├─ DBMS_OUTPUT.PUT_LINE (出力)
    │
    └─ [エラー時] qa_logger_pkg.p_qa_log (エラーログ記録)

p_print_to_dbms_output (CLOB版)
    │
    ├─ DBMS_LOB.SUBSTR (ログ用切り詰め)
    │
    ├─ qa_logger_pkg.append_param (パラメータログ記録)
    │
    ├─ DBMS_OUTPUT.ENABLE (バッファ拡張)
    │
    ├─ LOOP
    │      ├─ DBMS_LOB.GETLENGTH (長さ取得)
    │      ├─ DBMS_LOB.SUBSTR (分割取得)
    │      └─ DBMS_OUTPUT.PUT_LINE (出力)
    │
    └─ [エラー時] qa_logger_pkg.p_qa_log (エラーログ記録)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

VARCHAR2版:
pi_string        ───▶ p_print_to_dbms_output ───▶ DBMS_OUTPUT
'Hello World'                                      コンソール

CLOB版:
pi_string        ───▶ p_print_to_dbms_output ───▶ DBMS_OUTPUT
大きなCLOB                    │                   コンソール
                              ▼                   (複数行に分割)
                    ┌──────────────────────┐
                    │ 32767バイト単位で    │
                    │ 分割出力             │
                    └──────────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| qa_utils_pkg.sql | `src/plsql/pkg/qa_utils_pkg.sql` | ソース | DBMS出力機能を含むユーティリティパッケージ |
| qa_logger_pkg.sql | `src/plsql/pkg/qa_logger_pkg.sql` | ソース | エラーログ記録用パッケージ |
