# 機能設計書 5-オブジェクト除外

## 概要

本ドキュメントは、QUASTOシステムにおけるオブジェクト除外機能の設計仕様を記述する。

### 本機能の処理概要

オブジェクト除外機能は、特定のデータベースオブジェクトをルール実行結果から除外する機能である。QA_RULESテーブルのqaru_exclude_objects列にコロン区切りでオブジェクト名を指定することで、そのルールの検出結果から該当オブジェクトを自動的にフィルタリングする。

**業務上の目的・背景**：品質ルールは一般的な基準に基づいて定義されるが、特定のオブジェクトには例外的に適用しない場合がある。例えば、レガシーシステムとの互換性のために命名規則に従わないオブジェクトや、特別な理由で品質基準の対象外とするオブジェクトが存在する。この機能により、ルール自体を変更することなく、特定オブジェクトを検出対象から除外できる。

**機能の利用シーン**：
- レガシーオブジェクトを命名規則チェックから除外する場合
- 特定のテストデータ用オブジェクトを品質チェック対象外とする場合
- サードパーティ製品のオブジェクトを除外する場合
- 既知の問題を持つオブジェクトを一時的に検出対象から外す場合

**主要な処理内容**：
1. qaru_exclude_objects列にコロン区切りでオブジェクト名を格納
2. ルール実行後にp_exclude_objectsプロシージャで結果をフィルタリング
3. 除外オブジェクト名と検出オブジェクト名を大文字・小文字を無視して比較
4. 一致するオブジェクトをコレクションから削除
5. APEX画面から除外オブジェクトの設定が可能

**関連システム・外部連携**：APEX Add/Edit Rule画面のテキストエリアから除外オブジェクトを入力。

**権限による制御**：ルールの編集権限を持つユーザーのみ除外オブジェクトを設定可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 8 | Add/Edit Rule | 補助機能 | 除外オブジェクト設定（コロン区切りでオブジェクト名を指定） |

## 機能種別

データフィルタリング / 結果加工

## 入力仕様

### 入力パラメータ

#### p_exclude_objectsプロシージャ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_qa_rules | qa_rules_t | Yes | ルール実行結果コレクション | IN OUT NOCOPY |

#### qaru_exclude_objects列

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| qaru_exclude_objects | VARCHAR2(4000 CHAR) | No | 除外オブジェクト名（コロン区切り） | 例: OBJ1:OBJ2:OBJ3 |

### 入力データソース

- APEX Add/Edit Rule画面からのテキスト入力
- JSONインポート時のexclude_objectsフィールド

## 出力仕様

### 出力データ

p_exclude_objectsプロシージャは入力コレクションを直接変更（IN OUT NOCOPY）し、除外対象オブジェクトを削除する。

### 出力先

- 入力コレクションを直接変更（フィルタリング後のコレクション）

## 処理フロー

### 処理シーケンス

```
1. 入力コレクションの確認
   └─ コレクションが空でないことを確認
2. 各結果に対してループ処理
   └─ 2.1 QA_RULESからqaru_exclude_objectsを取得
       2.2 コロン区切りで分割してリスト化
       2.3 各除外オブジェクト名と結果オブジェクト名を比較（大文字・小文字無視）
       2.4 一致する場合はコレクションから削除
3. フィルタリング済みコレクションを返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{コレクション空?}
    B -->|Yes| C[終了]
    B -->|No| D[結果ループ開始]
    D --> E[qaru_exclude_objects取得]
    E --> F[コロン区切りで分割]
    F --> G{除外リスト空?}
    G -->|Yes| H[次の結果へ]
    G -->|No| I[除外オブジェクトループ]
    I --> J{オブジェクト名一致?}
    J -->|Yes| K[コレクションから削除]
    J -->|No| L[次の除外オブジェクト]
    K --> M[ループ終了]
    L --> I
    M --> H
    H --> N{次の結果あり?}
    N -->|Yes| D
    N -->|No| C
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | コロン区切り | 除外オブジェクト名はコロン（:）で区切って指定 | オブジェクト設定時 |
| BR-02 | 大文字・小文字無視 | オブジェクト名の比較は大文字・小文字を区別しない | フィルタリング時 |
| BR-03 | 完全一致 | オブジェクト名は完全一致で判定（部分一致ではない） | フィルタリング時 |
| BR-04 | NULL許容 | qaru_exclude_objectsがNULLの場合は除外処理をスキップ | フィルタリング時 |

### 計算ロジック

**除外オブジェクト分割ロジック**：
```sql
SELECT regexp_substr(l_excluded, '[^:]+', 1, level)
BULK COLLECT INTO l_excluded_objects
FROM dual
CONNECT BY regexp_substr(l_excluded, '[^:]+', 1, level) IS NOT NULL
```

**オブジェクト名比較ロジック**：
```sql
IF UPPER(l_excluded_objects(excl)) = UPPER(pi_qa_rules(i).object_name) THEN
    pi_qa_rules.DELETE(i);
    EXIT;
END IF;
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 除外オブジェクト取得 | QA_RULES | SELECT | qaru_exclude_objects値を取得 |
| 除外オブジェクト設定 | QA_RULES | UPDATE | qaru_exclude_objects値を更新 |

### テーブル別操作詳細

#### QA_RULES

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | qaru_exclude_objects | WHERE qaru_id = pi_qa_rules(i).qaru_id | p_exclude_objects内 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| OTHERS | データベースエラー | SELECT/処理時のエラー | ログに記録（例外は再スローしない） |

### リトライ仕様

リトライは不要。エラー発生時はログに記録し、処理を継続する（例外を握りつぶす）。

## トランザクション仕様

- 除外オブジェクト設定はAPEXフォームからのUPDATEで即時コミット
- p_exclude_objectsプロシージャ自体はメモリ操作のみ

## パフォーマンス要件

- フィルタリング処理：結果件数に比例（数ミリ秒〜数秒）

## セキュリティ考慮事項

- 除外オブジェクト設定はルール編集権限を持つユーザーのみ
- 除外設定の変更はqaru_updated_on, qaru_updated_by列で追跡可能

## 備考

- qaru_exclude_objectsはVARCHAR2(4000 CHAR)で最大4000文字
- result_cache修飾子でキャッシュ可能（f_get_excluded_objects）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、qaru_exclude_objects列の定義を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | qa_rules.sql | `src/ddl/tab/qa_rules.sql` | qaru_exclude_objects列の定義（14行目）とコメント（49行目）を確認 |

**読解のコツ**:
- コメント: `objects which should be excluded when using this rule. a colon delimted list of names`
- VARCHAR2(4000 CHAR)で十分な長さを確保

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | qa_main_pkg.sql | `src/plsql/pkg/qa_main_pkg.sql` | p_exclude_objectsプロシージャの仕様と実装を確認 |

**主要処理フロー**:
1. **172行目**: p_exclude_objectsプロシージャの仕様宣言
2. **842-897行目**: p_exclude_objectsプロシージャの実装本体
3. **851-853行目**: コレクションが空でないことを確認
4. **857-860行目**: QA_RULESからqaru_exclude_objectsを取得
5. **862-872行目**: コロン区切りで分割
6. **874-885行目**: オブジェクト名比較とコレクションからの削除

#### Step 3: 補助関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | qa_main_pkg.sql | `src/plsql/pkg/qa_main_pkg.sql` | f_get_excluded_objectsの実装を確認（result_cache） |

**主要処理フロー**:
- **533-566行目**: f_get_excluded_objects - 除外オブジェクト取得（result_cache修飾子付き）

#### Step 4: 呼び出し元を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | qa_api_pkg.sql | `src/plsql/pkg/qa_api_pkg.sql` | tf_run_ruleでのp_exclude_objects呼び出しを確認 |

**主要処理フロー**:
- **134行目**: tf_run_rule内でp_exclude_objectsを呼び出し

### プログラム呼び出し階層図

```
qa_api_pkg.tf_run_rule
    │
    └─ qa_main_pkg.p_exclude_objects
           │
           ├─ SELECT qaru_exclude_objects FROM qa_rules
           │
           ├─ regexp_substr (コロン区切り分割)
           │
           └─ コレクション.DELETE (一致オブジェクト削除)
```

### データフロー図

```
[入力]                     [処理]                              [出力]

qa_rules_t ──────────────▶ p_exclude_objects ──────────────▶ qa_rules_t
(検出オブジェクト)              │                           (フィルタリング済み)
                               │
                               ├─▶ qaru_exclude_objects取得
                               │
                               ├─▶ コロン区切り分割
                               │        └─▶ [OBJ1, OBJ2, OBJ3, ...]
                               │
                               └─▶ 名前比較・削除
                                        └─▶ UPPER(exclude) = UPPER(object_name)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| qa_rules.sql | `src/ddl/tab/qa_rules.sql` | DDL | qaru_exclude_objects列の定義 |
| qa_main_pkg.sql | `src/plsql/pkg/qa_main_pkg.sql` | PKG | p_exclude_objects, f_get_excluded_objects |
| qa_api_pkg.sql | `src/plsql/pkg/qa_api_pkg.sql` | PKG | p_exclude_objectsの呼び出し元 |
| qa_logger_pkg.sql | `src/plsql/pkg/qa_logger_pkg.sql` | PKG | エラーログ記録 |
