# 機能設計書 6-先行ルール依存関係

## 概要

本ドキュメントは、QUASTOシステムにおける先行ルール依存関係機能の設計仕様を記述する。

### 本機能の処理概要

先行ルール依存関係機能は、ルール実行順序を定義し、先行ルールが成功した場合のみ後続ルールを実行する機能である。QA_RULESテーブルのqaru_predecessor_ids列にコロン区切りで先行ルール番号を指定することで、ルール間の依存関係を表現する。また、循環参照（ループ）の検出機能も備える。

**業務上の目的・背景**：品質ルールには論理的な依存関係が存在することがある。例えば、「テーブル命名規則」ルールが成功した場合のみ「カラム命名規則」ルールを実行するといった制御が必要な場合がある。また、基盤的なルールの失敗時に派生ルールの実行を抑止することで、無駄なエラー出力を防ぎ、根本原因の特定を容易にする。

**機能の利用シーン**：
- 基盤ルールの成功を前提とした派生ルールを定義する場合
- ルール実行順序を明示的に制御したい場合
- 複数の前提条件を持つ複合的なルールを定義する場合
- エラーの連鎖を防ぎたい場合

**主要な処理内容**：
1. qaru_predecessor_ids列にコロン区切りで先行ルール番号を格納
2. qa_predecessor_order_vビューで先行ルールを考慮した実行順序を算出
3. f_check_for_loop関数で循環参照を検出
4. f_get_full_rule_pred関数で全先行ルールを再帰的に取得
5. tf_run_rules内で先行ルールの成功状態に基づき実行可否を判断

**関連システム・外部連携**：全ルール実行機能（tf_run_rules）と密接に連携し、実行順序と実行可否を制御する。

**権限による制御**：ルールの編集権限を持つユーザーのみ先行ルールを設定可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 8 | Add/Edit Rule | 補助機能 | 先行ルールIDの設定 |

## 機能種別

実行順序制御 / 依存関係管理

## 入力仕様

### 入力パラメータ

#### f_check_for_loop関数

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_qaru_rule_number | VARCHAR2 | No | チェック対象ルール番号（NULLで全ルール） | - |
| pi_client_name | VARCHAR2 | No | クライアント名（NULLで全クライアント） | - |

#### f_get_full_rule_pred関数

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_rule_number | VARCHAR2 | Yes | 対象ルール番号 | NOT NULL |

#### qaru_predecessor_ids列

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| qaru_predecessor_ids | VARCHAR2(4000 CHAR) | No | 先行ルール番号（コロン区切り） | 例: R001:R002:R003 |

### 入力データソース

- APEX Add/Edit Rule画面からのテキスト入力
- JSONインポート時のpredecessor_idsフィールド

## 出力仕様

### 出力データ

#### f_check_for_loop関数

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | VARCHAR2 | ループ検出時：ループを含むルール番号、正常時：NULL |

#### f_get_full_rule_pred関数

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | VARCHAR2 | カンマ区切りの全先行ルール番号リスト |

#### qa_predecessor_order_v

| 項目名 | 型 | 説明 |
|--------|-----|------|
| qaru_rule_number | VARCHAR2 | ルール番号 |
| predec | VARCHAR2 | 直接の先行ルール |
| step | NUMBER | 実行順序（階層レベル） |
| predecessor_list | VARCHAR2 | 全先行ルールリスト |

### 出力先

- 関数の戻り値
- qa_predecessor_order_vビュー（全ルール実行時の順序決定に使用）

## 処理フロー

### 処理シーケンス

```
1. ループ検出（f_check_for_loop）
   └─ 1.1 先行ルールを持つルールを取得
       1.2 階層クエリ（CONNECT BY）で循環参照をチェック
       1.3 ORA-1436例外でループ検出
2. 実行順序決定（qa_predecessor_order_v）
   └─ 2.1 先行ルールをコロン区切りで分割
       2.2 階層クエリで各ルールの実行階層（step）を算出
       2.3 最大stepを持つ行のみ取得（正規化）
3. 全先行ルール取得（f_get_full_rule_pred）
   └─ 3.1 先行ルールを再帰的に展開
       3.2 カンマ区切りでリスト化
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[先行ルールありルール取得]
    B --> C{ルールあり?}
    C -->|No| D[NULL返却]
    C -->|Yes| E[階層クエリ実行]
    E --> F{ORA-1436発生?}
    F -->|Yes| G[ループ検出ログ記録]
    G --> H[例外再スロー]
    F -->|No| I[次のルールへ]
    I --> J{全ルール処理完了?}
    J -->|No| B
    J -->|Yes| D
    D --> K[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | コロン区切り | 先行ルール番号はコロン（:）で区切って指定 | 先行ルール設定時 |
| BR-02 | ループ禁止 | 先行ルール依存関係に循環参照があってはならない | ルール実行前チェック |
| BR-03 | 先行ルール成功条件 | 先行ルールすべてが成功（検出0件）の場合のみ後続ルールを実行 | 全ルール実行時 |
| BR-04 | NULL許容 | qaru_predecessor_idsがNULLの場合は依存関係なし（即時実行可能） | ルール実行時 |

### 計算ロジック

**実行順序算出ロジック（qa_predecessor_order_v）**：
```sql
WITH splitted_pred AS (
    SELECT DISTINCT t.qaru_rule_number,
           TRIM(REGEXP_SUBSTR(t.qaru_predecessor_ids, '[^:]+', 1, levels.column_value)) AS predec
    FROM qa_rules t,
         TABLE(CAST(MULTISET(SELECT LEVEL FROM dual
                              CONNECT BY LEVEL <= LENGTH(REGEXP_REPLACE(t.qaru_predecessor_ids, '[^:]+')) + 1)
               AS SYS.ODCINUMBERLIST)) levels
)
SELECT qaru_rule_number, predec, LEVEL AS step
FROM splitted_pred
CONNECT BY NOCYCLE PRIOR qaru_rule_number = predec
START WITH predec IS NULL
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ループ検出 | QA_RULES | SELECT | qaru_predecessor_idsで階層クエリ |
| 実行順序取得 | qa_predecessor_order_v | SELECT | ルール実行順序の取得 |
| 先行ルール設定 | QA_RULES | UPDATE | qaru_predecessor_ids値を更新 |

### テーブル別操作詳細

#### QA_RULES

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | qaru_rule_number, qaru_predecessor_ids | WHERE qaru_predecessor_ids IS NOT NULL | f_check_for_loop |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ORA-01436 | CONNECT BY ループ | 先行ルール依存関係に循環参照 | ルール定義の依存関係を修正 |
| OTHERS | データベースエラー | 処理時のエラー | ログに記録して再スロー |

### リトライ仕様

リトライは不要。ループ検出時は設定の修正が必要。

## トランザクション仕様

- 参照のみの処理のためトランザクション管理は不要
- 先行ルール設定はAPEXフォームからのUPDATEで即時コミット

## パフォーマンス要件

- ループ検出：ルール数に依存（通常数秒以内）
- 実行順序取得：ルール数に依存

## セキュリティ考慮事項

- 先行ルール設定はルール編集権限を持つユーザーのみ

## 備考

- CONNECT BY NOCYCLEで無限ループを防止
- pragma exception_init(loop_detect_e, -1436)でORA-1436を名前付き例外として定義

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、qaru_predecessor_ids列とqa_running_rule_t型を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | qa_rules.sql | `src/ddl/tab/qa_rules.sql` | qaru_predecessor_ids列の定義（18行目）とコメント（53行目）を確認 |
| 1-2 | qa_running_rule_t.sql | `src/plsql/typ/qa_running_rule_t.sql` | 実行状態追跡用型の属性を確認 |

**読解のコツ**:
- コメント: `colon delimted list of predecessors (PIQA_ID). the rule will only be executed if there are no errors found for the predecessor`
- qa_running_rule_tのpredecessor属性で先行ルールを追跡

#### Step 2: ビューを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | qa_predecessor_order_v.sql | `src/plsql/vw/qa_predecessor_order_v.sql` | 先行ルール順序ビューの階層クエリを確認 |

**主要処理フロー**:
1. **10-20行目**: splitted_pred CTEで先行ルールをコロン区切りで分割
2. **21-26行目**: CONNECT BY NOCYCLEで階層構造を構築
3. **28行目**: max_step = stepで各ルールの最終階層のみ取得

#### Step 3: ループ検出関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | qa_main_pkg.sql | `src/plsql/pkg/qa_main_pkg.sql` | f_check_for_loop関数の仕様と実装を確認 |

**主要処理フロー**:
1. **181-185行目**: f_check_for_loop関数の仕様宣言
2. **899-960行目**: f_check_for_loop関数の実装本体
3. **909-911行目**: loop_detect_e例外をORA-1436にマッピング
4. **917-940行目**: 階層クエリでループ検出
5. **946-952行目**: ORA-1436例外時のログ記録と再スロー

#### Step 4: 全先行ルール取得関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | qa_main_pkg.sql | `src/plsql/pkg/qa_main_pkg.sql` | f_get_full_rule_pred関数の仕様と実装を確認 |

**主要処理フロー**:
- **192行目**: f_get_full_rule_pred関数の仕様宣言
- **962-1004行目**: f_get_full_rule_pred関数の実装（再帰呼び出し）

### プログラム呼び出し階層図

```
qa_api_pkg.tf_run_rules
    │
    ├─ qa_main_pkg.f_check_for_loop
    │      └─ SELECT ... CONNECT BY ... (ORA-1436検出)
    │
    ├─ qa_main_pkg.tf_get_rule_numbers
    │      └─ JOIN qa_predecessor_order_v (順序取得)
    │
    └─ ループ処理（先行ルール成功チェック）

qa_predecessor_order_v
    │
    └─ qa_main_pkg.f_get_full_rule_pred (再帰呼び出し)
```

### データフロー図

```
[入力]                     [処理]                              [出力]

qaru_predecessor_ids ──▶ コロン区切り分割 ──────────────▶ 先行ルールリスト
(R001:R002)                    │
                               │
                               ├─▶ 階層クエリ (CONNECT BY)
                               │        │
                               │        └─▶ step (実行順序)
                               │
                               └─▶ ループ検出
                                        │
                                        └─▶ ORA-1436 または NULL
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| qa_rules.sql | `src/ddl/tab/qa_rules.sql` | DDL | qaru_predecessor_ids列の定義 |
| qa_main_pkg.sql | `src/plsql/pkg/qa_main_pkg.sql` | PKG | f_check_for_loop, f_get_full_rule_pred, tf_get_rule_numbers |
| qa_api_pkg.sql | `src/plsql/pkg/qa_api_pkg.sql` | PKG | tf_run_rulesでの依存関係チェック |
| qa_predecessor_order_v.sql | `src/plsql/vw/qa_predecessor_order_v.sql` | VIEW | 先行ルール順序ビュー |
| qa_running_rule_t.sql | `src/plsql/typ/qa_running_rule_t.sql` | TYPE | 実行状態追跡型 |
