# 機能設計書 8-ルール存在確認

## 概要

本ドキュメントは、QUASTOシステムにおけるルール存在確認機能の設計仕様を記述する。

### 本機能の処理概要

ルール存在確認機能は、クライアント名に対応するルールがQA_RULESテーブルに存在するかを確認する機能である。qa_main_pkg.f_exist_rules関数として実装されており、ルール処理の事前チェックに使用される。

**業務上の目的・背景**：ルール関連の処理を実行する前に、対象となるルールが存在するかを確認する必要がある。存在しない場合に適切なエラーハンドリングやユーザーへの通知を行うために、この確認機能が提供されている。

**機能の利用シーン**：
- ルールエクスポート前にルールの存在を確認する場合
- クライアント名を指定した処理の事前検証
- APEX画面でのルール一覧表示前の確認

**主要な処理内容**：
1. クライアント名をパラメータとして受け取る（NULLの場合は全ルールを対象）
2. QA_RULESテーブルでEXISTS句を使用して存在確認
3. 存在する場合TRUE、存在しない場合FALSEを返却

**関連システム・外部連携**：エクスポート機能等の事前チェックとして使用される。

**権限による制御**：authid definerでパッケージ所有者権限で実行される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面から直接呼び出されることはなく、APIとして使用 |

## 機能種別

データ存在確認 / バリデーション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pi_qaru_client_name | VARCHAR2 | No | クライアント名（NULLで全ルール） | デフォルト: NULL |

### 入力データソース

- 他のPL/SQLプログラムからの関数呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | BOOLEAN | TRUE=存在する、FALSE=存在しない |

### 出力先

- 関数の戻り値（BOOLEAN）

## 処理フロー

### 処理シーケンス

```
1. パラメータログ記録
   └─ qa_logger_pkg.append_paramでパラメータを記録
2. 存在確認クエリ実行
   └─ EXISTS句で効率的に存在確認
3. 結果判定
   └─ COUNT結果が0ならFALSE、それ以外はTRUE
4. 結果返却
   └─ BOOLEAN値を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[パラメータログ記録]
    B --> C[EXISTS確認クエリ実行]
    C --> D{l_result = 0?}
    D -->|Yes| E[FALSE返却]
    D -->|No| F[TRUE返却]
    E --> G[終了]
    F --> G
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | NULL許容 | pi_qaru_client_nameがNULLの場合は全ルールを対象 | 関数呼び出し時 |
| BR-02 | 効率的確認 | EXISTS句を使用して最初の一致で検索を停止 | クエリ実行時 |

### 計算ロジック

**存在確認ロジック**：
```sql
SELECT DECODE(COUNT(1), 0, 0, 1)
INTO l_result
FROM dual
WHERE EXISTS (SELECT null
              FROM QA_RULES
              WHERE qaru_client_name = pi_qaru_client_name
              OR pi_qaru_client_name IS NULL);
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 存在確認 | QA_RULES | SELECT | EXISTS句で存在確認 |

### テーブル別操作詳細

#### QA_RULES

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | - | WHERE qaru_client_name = pi_qaru_client_name OR pi_qaru_client_name IS NULL | EXISTS句内 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| OTHERS | データベースエラー | SELECT時のエラー | ログに記録して再スロー |

### リトライ仕様

リトライは不要。

## トランザクション仕様

- 参照のみの処理のためトランザクション管理は不要

## パフォーマンス要件

- 実行時間：数ミリ秒以内（EXISTS句で効率的）

## セキュリティ考慮事項

- authid definerでパッケージ所有者権限で実行

## 備考

- EXISTS句により、最初の一致で検索が停止するため効率的
- DECODE関数で0/1をBOOLEANに変換

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 関数仕様を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | qa_main_pkg.sql | `src/plsql/pkg/qa_main_pkg.sql` | f_exist_rules関数の仕様と実装を確認 |

**主要処理フロー**:
1. **100-105行目**: f_exist_rules関数の仕様宣言
2. **460-491行目**: f_exist_rules関数の実装本体
3. **466-468行目**: パラメータのログ記録
4. **470-476行目**: EXISTS句を使用した存在確認クエリ
5. **478-483行目**: 結果判定（0ならFALSE、それ以外TRUE）

**読解のコツ**:
- DECODE(COUNT(1), 0, 0, 1)は「COUNT結果が0なら0、それ以外は1」を意味
- EXISTS句内で pi_qaru_client_name IS NULL の条件により、NULLパラメータ時は全ルールを対象

### プログラム呼び出し階層図

```
呼び出し元（例：エクスポート処理）
    │
    └─ qa_main_pkg.f_exist_rules
           │
           └─ SELECT ... EXISTS (SELECT ... FROM QA_RULES ...)
```

### データフロー図

```
[入力]                     [処理]                              [出力]

pi_qaru_client_name ──▶ qa_main_pkg.f_exist_rules ─────────▶ BOOLEAN
(または NULL)              │                               (TRUE/FALSE)
                           │
                           └─▶ EXISTS句クエリ
                                    │
                                    └─▶ COUNT = 0: FALSE
                                        COUNT > 0: TRUE
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| qa_main_pkg.sql | `src/plsql/pkg/qa_main_pkg.sql` | PKG | f_exist_rules関数 |
| qa_rules.sql | `src/ddl/tab/qa_rules.sql` | DDL | QA_RULESテーブル定義 |
| qa_logger_pkg.sql | `src/plsql/pkg/qa_logger_pkg.sql` | PKG | パラメータログ記録 |
