# QuickerSite コードリーディングガイドライン

## はじめに

このガイドラインは、QuickerSiteのコードベースを効率的に理解するための手引きです。
Classic ASP/VBScriptに精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、Classic ASP/VBScriptの基本構文と概念を解説します。

### 1.1 プログラム構造

Classic ASPは、HTMLとサーバーサイドスクリプト（VBScript）を混在させたWebアプリケーション技術です。
サーバーサイドコードは `<%` と `%>` で囲まれています。

```vbscript
// ファイル: asp/begin.asp:1-4
<%@LANGUAGE="VBSCRIPT" CODEPAGE="65001"%>
<%
Option Explicit

on error resume next
```

**解説:**
- `<%@LANGUAGE="VBSCRIPT" CODEPAGE="65001"%>`: 使用言語（VBScript）と文字コード（UTF-8）を宣言
- `Option Explicit`: 変数の明示的宣言を強制（未宣言変数の使用でエラー）
- `on error resume next`: エラーが発生しても処理を継続

### 1.2 データ型と変数

VBScriptはVariant型のみをサポートし、すべての変数は暗黙的にVariantとして扱われます。

```vbscript
// ファイル: asp/includes/functions.asp:86-93
Function isLeeg(byval value)
isLeeg=false
if isNull(value) then
isLeeg=true
else
if isEmpty(value) or trim(value)="" then isLeeg=true
end if
End Function
```

**主要なデータ型チェック:**
- `isNull()`: Null値のチェック
- `isEmpty()`: 未初期化値のチェック
- `isNumeric()`: 数値かどうかのチェック

**型変換関数（asp/includes/functions.asp内）:**

| 関数 | 説明 | 場所 |
|------|------|------|
| `convertGetal(value)` | 数値への変換（デフォルト0） | functions.asp:94-112 |
| `convertStr(value)` | 文字列への変換（デフォルト""） | functions.asp:134-145 |
| `convertBool(value)` | 論理値への変換（デフォルトfalse） | functions.asp:147-175 |
| `convertLng(value)` | 長整数への変換（デフォルトnull） | functions.asp:114-132 |

### 1.3 制御構造

**条件分岐:**

```vbscript
// ファイル: asp/process.asp:40-50
select case lcase(convertStr(request("pageAction")))

case "avataredit"
response.clear%><!-- #include file="process_avatar.asp"--><%response.flush
CleanupASP()
response.end
case "fileupload"
response.clear%><!-- #include file="process_fileupload.asp"--><%response.flush
CleanupASP()
response.end
```

**ループ:**

```vbscript
// ファイル: asp/includes/functions.asp:63-82
while instr(compressHTML,vbtab)<>0
    compressHTML=replace(compressHTML,vbtab," ",1,-1,1)
wend

while instr(compressHTML,vbcrlf)<>0
    compressHTML=replace(compressHTML,vbcrlf," ",1,-1,1)
wend
```

### 1.4 関数/メソッド定義

VBScriptでは`Function`（戻り値あり）と`Sub`（戻り値なし）の2種類があります。

```vbscript
// ファイル: asp/includes/functions.asp:59-83
function compressHTML(value)

    compressHTML=value

    while instr(compressHTML,vbtab)<>0
        compressHTML=replace(compressHTML,vbtab," ",1,-1,1)
    wend
    ' ... 省略 ...
end function
```

**ポイント:**
- 戻り値は関数名に代入する形式（`関数名 = 戻り値`）
- `byval`/`byref`で引数の渡し方を指定（デフォルトは`byref`）

### 1.5 モジュール/インポート

Classic ASPでは`#include`ディレクティブでファイルをインクルードします。

```vbscript
// ファイル: asp/begin.asp:56-68
<!-- #include file="config/rebrand.asp"-->
<!-- #include file="config/web_config.asp"-->
<!-- #include file="includes/artisteer.asp"-->
<!-- #include file="includes/sha256.asp"-->
<!-- #include file="includes/iso-8859-1.asp"-->
<!-- #include file="includes/fileexplorer.asp"-->
<!-- #include file="includes/constants.asp"-->
<!-- #include file="includes/dateFormatList.asp"-->
<!-- #include file="includes/excelFile.asp"-->
<!-- #include file="includes/messages.asp"-->
<!-- #include file="includes/database.asp"-->
<!-- #include file="includes/logVisit.asp"-->
<!-- #include file="includes/mail_message.asp"-->
```

**注意点:**
- `file=`は相対パス、`virtual=`は仮想ディレクトリからのパス
- インクルードはコンパイル時に展開されるため、条件分岐内でも常に読み込まれる

---

## 2. プロジェクト固有の概念

> このセクションでは、QuickerSite特有の概念を解説します。

### 2.1 フレームワーク固有の概念

QuickerSiteは独自のCMSフレームワークで、以下のコア概念があります：

**1. Customer（顧客/サイト）**
```vbscript
// ファイル: asp/includes/customer.asp:2-32
<%class cls_customer
Public resetDBConn
Public iId, sName, sURL, sBannerMenu, bEnableMainRSS, bRemoveTemplatesOnSetup,iFolderSize
Public bOnline, dUpdatedTS, dCreatedTS, dOnlineFrom, dResetStats, sTotalPW, bApplication, bIntranet, bCatalog, bMonitor
```

- 1つのデータベースで複数のWebサイト（Customer）を管理可能
- 各サイトは`tblCustomer`テーブルの1レコードに対応

**2. Page（ページ）**
```vbscript
// ファイル: asp/includes/page.asp:4-12
class cls_page
Public iId, iParentID, iListPageID, iCustomerID, sTitle, sExternalURLPrefix, sExternalURL, bOpenInNewWindow,sValueToBeValidated,sTitleToBeValidated
Public sValue, sValueTextOnly, iRang, bOnline, bDeleted, bContainerPage, bLossePagina, bHomepage, updatedTS,iUpdatedBy,dUpdatedOn
```

- CMSのコンテンツページを表現
- 階層構造（親子関係）をサポート
- テンプレートとの関連付け

**3. Template（テンプレート）**
- ページのレイアウトを定義
- `asp/includes/template.asp`で実装

### 2.2 プロジェクト独自のパターン

**クラスベースの設計パターン:**

QuickerSiteでは`cls_`プレフィックスでクラスを定義しています。

```vbscript
// ファイル: asp/includes/database.asp:4-62
Class cls_Database
private p_getConn
private p_getConnLabels
Private Sub Class_Initialize()
set p_getConn = nothing
set p_getConnLabels = nothing
End Sub
Public Function Execute(sql)
On Error Resume Next
dim connection
set connection = getConn()
Set Execute = connection.Execute(sql)
On Error Goto 0
End Function
```

**シングルトンパターン:**
```vbscript
// ファイル: asp/includes/database.asp:2-3
<%dim db
set db = new cls_database
```

グローバル変数`db`でデータベース接続を管理し、アプリケーション全体で共有します。

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `bs_*.asp` | Backsite（管理画面）ページ | `bs_login.asp`, `bs_menu.asp` |
| `ad_*.asp` | Admin（システム管理）ページ | `ad_default.asp`, `ad_login.asp` |
| `fs_*.asp` | Frontend Site（フロント編集）ページ | `fs_editPage.asp`, `fs_editListItem.asp` |
| `process_*.asp` | 処理ロジックページ | `process_login.asp`, `process_catalog.asp` |
| `includes/*.asp` | 共通インクルードファイル | `includes/functions.asp` |
| `*List.asp` | リスト表示系の定義 | `contactStatusList.asp`, `languageList.asp` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `cls_` | クラス定義 | `cls_page`, `cls_customer`, `cls_database` |
| `p_` | プライベートプロパティ | `p_getConn`, `p_template` |
| `i` | 整数型（Integer/ID） | `iId`, `iParentID`, `iCustomerID` |
| `s` | 文字列型（String） | `sTitle`, `sValue`, `sURL` |
| `b` | 論理型（Boolean） | `bOnline`, `bDeleted`, `bHomepage` |
| `d` | 日付型（Date） | `dCreatedTS`, `dUpdatedTS` |
| `rs` | レコードセット | `rsUL`, `rs` |
| `tbl` | テーブル名 | `tblPage`, `tblCustomer` |
| `convert*` | 型変換関数 | `convertGetal`, `convertStr`, `convertBool` |

### 3.3 プログラム分類一覧

| 種別 | プレフィックス | 説明 |
|------|---------------|------|
| 管理画面 | `bs_` | 一般ユーザー向け管理画面（Backsite） |
| システム管理 | `ad_` | システム管理者向け機能（Admin） |
| フロント編集 | `fs_` | フロントエンドからの編集機能 |
| 処理ロジック | `process_` | ビジネスロジック処理 |
| モジュール | `module` | 追加モジュール |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
QuickerSite-master/
├── asp/                          # サーバーサイドASPコード
│   ├── config/                   # 設定ファイル
│   │   ├── rebrand.asp          # ブランディング設定
│   │   └── web_config.asp       # メイン設定ファイル
│   ├── includes/                 # 共通インクルードファイル
│   │   ├── database.asp         # データベース接続クラス
│   │   ├── functions.asp        # ユーティリティ関数
│   │   ├── page.asp             # ページクラス
│   │   ├── customer.asp         # 顧客/サイトクラス
│   │   └── ...                  # その他の機能クラス
│   ├── assetmanager/            # アセット管理機能
│   ├── codemirror/              # コードエディタ
│   ├── uploadify214/            # ファイルアップロード
│   ├── begin.asp                # アプリケーション初期化
│   ├── process.asp              # メイン処理ルーター
│   ├── bs_*.asp                 # 管理画面ページ
│   ├── ad_*.asp                 # システム管理ページ
│   ├── fs_*.asp                 # フロント編集ページ
│   └── process_*.asp            # 処理ロジック
├── db/                           # データベースファイル
│   ├── QuickerSite.mdb          # メインデータベース（Access）
│   └── QuickerLabels.mdb        # ラベル/多言語データベース
├── js/                           # JavaScriptファイル
│   ├── JQuery183.js             # jQuery
│   ├── JQueryUI.js              # jQuery UI
│   └── ...                      # その他のライブラリ
├── css/                          # スタイルシート
├── ckeditor445/                  # CKEditorリッチテキストエディタ
├── modules/                      # 拡張モジュール
├── templates/                    # サイトテンプレート
├── UserFiles/                    # ユーザーアップロードファイル
├── fixedImages/                  # システム固定画像
├── backsite/                     # 管理画面エントリーポイント
├── common/                       # 共通リソース
├── default.asp                   # メインエントリーポイント
├── index.asp                     # インデックスページ
├── web.config                    # IIS設定
└── README.md                     # プロジェクト説明
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `asp/` | サーバーサイドロジック全般 | `begin.asp`, `process.asp` |
| `asp/config/` | アプリケーション設定 | `web_config.asp` |
| `asp/includes/` | 共通ライブラリ/クラス | `functions.asp`, `database.asp`, `page.asp` |
| `db/` | MS Accessデータベース | `QuickerSite.mdb` |
| `js/` | クライアントサイドJavaScript | `JQuery183.js` |
| `css/` | スタイルシート | `qs_ltr.css` |
| `ckeditor445/` | リッチテキストエディタ | - |
| `modules/` | 追加モジュール | `ajax.asp`, `module*.asp` |
| `templates/` | サイトテンプレート | - |
| `UserFiles/` | ユーザーアップロードコンテンツ | - |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

QuickerSiteは**フロントコントローラーパターン**を採用した従来型Webアプリケーションです。

```
+-------------------+     +-------------------+     +-------------------+
|   ブラウザ         |---->|   default.asp     |---->|   asp/begin.asp   |
|   (クライアント)    |     |   (エントリー)     |     |   (初期化)         |
+-------------------+     +-------------------+     +-------------------+
                                    |                         |
                                    v                         v
                          +-------------------+     +-------------------+
                          |   asp/process.asp |<----|   includes/*.asp  |
                          |   (ルーター)       |     |   (ライブラリ)     |
                          +-------------------+     +-------------------+
                                    |
                    +---------------+---------------+
                    |               |               |
                    v               v               v
          +--------------+  +--------------+  +--------------+
          | process_*.asp|  | bs_*.asp     |  | ad_*.asp     |
          | (処理ロジック) |  | (管理画面)    |  | (システム管理) |
          +--------------+  +--------------+  +--------------+
                    |               |               |
                    v               v               v
          +---------------------------------------------------+
          |              db/QuickerSite.mdb                   |
          |              (MS Access データベース)               |
          +---------------------------------------------------+
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| プレゼンテーション | HTMLレンダリング、ユーザー入力受付 | `default.asp`, `bs_*.asp` |
| ルーティング | リクエスト振り分け | `asp/process.asp` |
| ビジネスロジック | データ処理、業務ルール | `process_*.asp`, `asp/includes/*.asp` |
| データアクセス | データベース操作 | `asp/includes/database.asp` |
| 設定 | アプリケーション設定 | `asp/config/web_config.asp` |

### 5.3 データフロー

**リクエスト処理の流れ:**

1. **リクエスト受信** (`default.asp`)
   - IISがリクエストを受信し、default.aspにルーティング

2. **初期化** (`asp/begin.asp`)
   - 言語設定、セッション設定
   - 全インクルードファイルの読み込み
   - データベース接続の確立
   - 顧客オブジェクト（customer）の初期化

3. **ルーティング** (`asp/process.asp`)
   - `pageAction`パラメータに基づく処理振り分け
   - ページ情報の取得と検証

4. **処理実行** (`process_*.asp` など)
   - 対応する処理ファイルの実行
   - データベース操作

5. **レスポンス生成**
   - テンプレートの適用
   - HTMLの出力

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

```vbscript
// ファイル: default.asp:1-3
<!-- #include file="asp/begin.asp"-->
<%=run(execBeforePageLoad)%>
<!-- #include file="asp/process.asp"-->
```

**解説:**
- `asp/begin.asp`: アプリケーション全体の初期化
- `execBeforePageLoad`: カスタム前処理の実行（設定で指定可能）
- `asp/process.asp`: メインのルーティングと処理

### 6.2 ビジネスロジック

**ページクラス（cls_page）の主要メソッド:**

```vbscript
// ファイル: asp/includes/page.asp:58-69
Public function pickByCode(sPageCode)
ON Error Resume Next
if not isLeeg(sPageCode) then
dim rs
set rs=db.execute("select iID from tblPage where " & sqlCustId & " and sCode='"& ucase(cleanup(sPageCode)) & "'")
if not rs.eof then
pick(rs(0))
end if
set rs=nothing
dumperror "Pick Page By Code",err
end if
ON Error Goto 0
end function
```

**顧客クラス（cls_customer）の初期化:**

```vbscript
// ファイル: asp/begin.asp:153-156
'load customer
dim customer
set customer=new cls_customer
customer.pick(cId)
```

### 6.3 データアクセス

**データベースクラスの主要メソッド:**

```vbscript
// ファイル: asp/includes/database.asp:11-16
Public Function Execute(sql)
On Error Resume Next
dim connection
set connection = getConn()
Set Execute = connection.Execute(sql)
On Error Goto 0
End Function
```

```vbscript
// ファイル: asp/includes/database.asp:25-31
Public Function GetDynamicRS
On Error Resume Next
Set GetDynamicRS = server.CreateObject ("adodb.recordset")
GetDynamicRS.CursorType = 1
GetDynamicRS.LockType = 3
set GetDynamicRS.ActiveConnection = getConn()
On Error Goto 0
End Function
```

**使用例:**

| メソッド | 用途 | 例 |
|---------|------|-----|
| `db.Execute(sql)` | 読み取り専用クエリ | `db.execute("select * from tblPage")` |
| `db.GetDynamicRS` | 更新可能なレコードセット | フォームデータの保存時 |

### 6.4 ユーティリティ/共通機能

**主要なユーティリティ関数（asp/includes/functions.asp）:**

| 関数 | 説明 | 場所 |
|------|------|------|
| `isLeeg(value)` | 空値チェック | functions.asp:86-93 |
| `convertGetal(value)` | 数値変換 | functions.asp:94-112 |
| `convertStr(value)` | 文字列変換 | functions.asp:134-145 |
| `convertBool(value)` | 論理値変換 | functions.asp:147-175 |
| `compressHTML(value)` | HTML圧縮 | functions.asp:59-83 |
| `filterImages(v)` | Base64画像処理 | functions.asp:3-33 |
| `removeCRB(value)` | コードブロック除去 | functions.asp:35-42 |

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| クラスベースモデル | VBScriptクラスによるエンティティ表現 | 高 | includes/page.asp, includes/customer.asp |
| シングルトンDB | グローバルDB接続オブジェクト | 高 | includes/database.asp |
| インクルードチェーン | 多数のインクルードによる機能構築 | 高 | begin.asp |
| Select Caseルーティング | リクエストパラメータによる処理分岐 | 高 | process.asp |
| エラー抑制パターン | On Error Resume Next | 高 | 全般 |

### 各パターンの詳細

#### パターン1: クラスベースモデル

**目的:** エンティティの状態とふるまいをカプセル化

**実装例:**
```vbscript
// ファイル: asp/includes/page.asp:4-57
class cls_page
Public iId, iParentID, iListPageID, iCustomerID, sTitle, sExternalURLPrefix...
' ... プロパティ宣言 ...

Private Sub Class_Initialize
on error resume next
bNocache=false
bLoadInCache=false
iId=null
iCustomerID=cId
' ... 初期化処理 ...
on error goto 0
end sub
```

**解説:** VBScriptのClassを使用し、データベーステーブルの各レコードをオブジェクトとして表現。`Class_Initialize`でデフォルト値を設定。

#### パターン2: Select Caseルーティング

**目的:** リクエストパラメータに基づく処理の振り分け

**実装例:**
```vbscript
// ファイル: asp/process.asp:40-67
select case lcase(convertStr(request("pageAction")))

case "avataredit"
response.clear%><!-- #include file="process_avatar.asp"--><%response.flush
CleanupASP()
response.end
case "fileupload"
response.clear%><!-- #include file="process_fileupload.asp"--><%response.flush
CleanupASP()
response.end
case "unsubscribe"%><!-- #include file="process_unsubscribe.asp"--><%
case "send"%><!-- #include file="process_send.asp"--><%
' ... 他のケース ...
end select
```

**解説:** `pageAction`パラメータの値に応じて、対応する処理ファイルをインクルードして実行。

#### パターン3: エラー抑制パターン

**目的:** エラー発生時もアプリケーションを継続動作させる

**実装例:**
```vbscript
// ファイル: asp/includes/functions.asp:94-112
Function convertGetal(value)
if not isnull(value) then
if IsNumeric(value) then
if trim(value)<>"" then
on error resume next
convertGetal=cdbl(value)
dumperror "convertGetal:"&value,err
err.clear
on error goto 0
else
convertGetal=0
end if
```

**解説:** `On Error Resume Next`でエラーを抑制し、`dumperror`でログ出力後、`err.clear`でエラーをクリア。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: ページ表示フロー

**概要:** ユーザーがWebページにアクセスした際のコンテンツ表示処理

**処理フロー:**
```
default.asp → asp/begin.asp → asp/process.asp → テンプレート適用 → HTML出力
```

**詳細な追跡:**

1. **リクエスト受信** (`default.asp:1-3`)
   ```vbscript
   <!-- #include file="asp/begin.asp"-->
   <%=run(execBeforePageLoad)%>
   <!-- #include file="asp/process.asp"-->
   ```

2. **アプリケーション初期化** (`asp/begin.asp:151-196`)
   ```vbscript
   'load customer
   dim customer
   set customer=new cls_customer
   customer.pick(cId)

   ' ... 中略 ...

   'load selectedpage
   dim selectedPage
   set selectedPage=new cls_page
   selectedPage.pick(decrypt(left(Request("iId"),40)))
   if isNull(selectedPage.iId) then
       selectedPage.pickByCode(left(request("sCode"),25))
   end if
   ```

3. **ルーティング判定** (`asp/process.asp:40-267`)
   ```vbscript
   select case lcase(convertStr(request("pageAction")))
   case "avataredit"
   ' ... 各アクションの処理 ...
   case else
   ' ページ表示処理
   end select
   ```

4. **テンプレート適用とHTML出力** (`asp/process.asp:279`)
   ```vbscript
   if not printReplies then
   selectedPage.buildTemplate()
   ```

#### 例2: 管理画面ログインフロー

**概要:** 管理者が管理画面にログインする処理

**処理フロー:**
```
bs_login.asp → 認証処理 → bs_default.asp（成功時）
```

**詳細な追跡:**

1. **ログインフォーム表示** (`asp/bs_login.asp:1-8`)
   ```vbscript
   <!-- #include file="begin.asp"-->
   <%bUseArtLoginTemplate=true
   dim secondAdmin
   set secondAdmin=customer.secondAdmin
   ```

2. **認証処理** (`asp/bs_login.asp:8-26`)
   ```vbscript
   <%if Request.Form ("btnaction")="Login" then
   If LCase(session("captcha")) <> LCase(Left(Request.Form("captcha"),4)) Then
   ' CAPTCHA検証
   message.AddError("err_captcha")
   elseif logon.logon(sha256(Request.Form ("password"))) then
   ' パスワード認証成功
   if isLeeg(request("bs_page")) then
   Response.Redirect ("bs_default.asp")
   ```

3. **ロックアウト処理** (`asp/bs_login.asp:24-26`)
   ```vbscript
   application("bsLoginCount"&UserIP)=convertGetal(application("bsLoginCount"&UserIP))+1
   end if
   logon.lockBSAdmin()
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. README.md - プロジェクト概要
2. asp/config/web_config.asp - 設定構成の理解
3. asp/begin.asp - 初期化フローの理解
4. asp/process.asp - メインルーティングの理解

#### 特定機能を理解したい場合
1. asp/process.asp内の該当case文
2. 対応するprocess_*.aspファイル
3. asp/includes/内の関連クラスファイル

#### 改修作業を行う場合
1. 対象機能のASPファイル
2. asp/includes/内の関連クラス
3. asp/config/web_config.asp（設定変更の場合）

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| README.md | プロジェクト概要、インストール手順 | 初回参画時 |
| web.config | IIS設定 | デプロイ時 |
| asp/config/web_config.asp | アプリケーション設定 | 設定変更時 |
| LICENSE | ライセンス情報 | 利用条件確認時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: データベース接続はどこで設定されていますか？
A: `asp/config/web_config.asp`の40-55行目でデータベースパスと接続タイプを設定しています。
```vbscript
QS_DBS              = 1  '1 for Access
C_DATABASE          = server.MapPath (C_VIRT_DIR & "/db/QuickerSite.mdb")
```

#### Q: 新しいページタイプを追加するには？
A: `asp/process.asp`の`select case`文に新しいcaseを追加し、対応する処理ファイルを作成します。

#### Q: セッション管理はどこで行われていますか？
A: `asp/begin.asp`の15-16行目でセッション設定が行われ、認証状態は`asp/includes/logonEdit.asp`で管理されています。

#### Q: 多言語対応はどのように実装されていますか？
A: `db/QuickerLabels.mdb`データベースにラベルが格納され、`asp/includes/label.asp`と`asp/includes/language.asp`で管理されています。

### デバッグのヒント

1. **エラーの確認**: `On Error Resume Next`が多用されているため、`dumperror`関数の出力を確認
2. **SQLの確認**: `db.Execute()`に渡されるSQL文を`Response.Write`で出力して確認
3. **セッション状態**: `Session`オブジェクトの内容を確認
4. **Application変数**: `Application`オブジェクトの設定値を確認

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Backsite | QuickerSiteの管理画面（bs_プレフィックス） |
| Customer | QuickerSiteにおけるWebサイト/テナントの単位 |
| Page | CMSで管理されるコンテンツページ |
| Template | ページのレイアウトを定義するテンプレート |
| Intranet | 会員制エリア機能 |
| Catalog | 商品カタログ機能 |
| Feed | RSSフィード機能 |
| Contact | 顧客/会員情報 |
| CKEditor | リッチテキストエディタ |
| UFl (User Friendly URL) | SEOフレンドリーURL |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| default.asp | メインエントリーポイント | リクエスト受付 |
| asp/begin.asp | 初期化ファイル | 設定、インクルード、初期化 |
| asp/process.asp | メインルーター | リクエスト振り分け |
| asp/config/web_config.asp | 設定ファイル | DB接続、メール、各種設定 |
| asp/includes/database.asp | DBアクセスクラス | ADODB操作 |
| asp/includes/functions.asp | ユーティリティ関数 | 型変換、文字列処理 |
| asp/includes/page.asp | ページクラス | CMSページ操作 |
| asp/includes/customer.asp | 顧客クラス | サイト設定操作 |
| db/QuickerSite.mdb | メインDB | コンテンツ、設定 |
| db/QuickerLabels.mdb | ラベルDB | 多言語ラベル |

### C. 参考資料

- [QuickerSite公式サイト](https://quickersite.com/)
- [QuickerSiteデモサイト](https://demo.quickersite.com/)
- [Microsoft VBScript リファレンス](https://docs.microsoft.com/ja-jp/previous-versions/windows/internet-explorer/ie-developer/scripting-articles/d1wf56tt(v=vs.84))
- [Classic ASP リファレンス](https://docs.microsoft.com/ja-jp/previous-versions/iis/6.0-sdk/ms524664(v=vs.90))
- [ADO リファレンス](https://docs.microsoft.com/ja-jp/sql/ado/microsoft-activex-data-objects-ado)
- [jstemplates.com](https://jstemplates.com/) - QuickerSite用テンプレートジェネレーター
