# セキュリティ設計書

## 概要

本ドキュメントは、QuickerSite CMSプロジェクトのセキュリティ設計について記述する。QuickerSiteはClassic ASP（VBScript）で構築されたコンテンツ管理システムであり、本書では認証・認可設計、通信セキュリティ、データセキュリティ、入出力対策、監査ログの各観点からセキュリティ設計を整理する。

## 認証設計

### 認証方式

| 認証レベル | 認証方式 | 説明 |
| --- | --- | --- |
| 管理者認証（Admin） | パスワード認証 + SHA-256ハッシュ | システム管理者向け。`C_ADMINPASSWORD`で設定されたパスワードをSHA-256でハッシュ化して認証 |
| バックサイト認証 | パスワード認証 + SHA-256ハッシュ + CAPTCHA | コンテンツ管理者向け。`customer.adminPassword`または`secondAdmin.sPassword`で認証。CAPTCHAによるボット対策を実装 |
| セカンドアドミン認証 | パスワード認証 + SHA-256ハッシュ | 制限付き管理者向け。特定機能のみアクセス可能な権限制限付き認証 |
| イントラネット認証 | メール/ニックネーム + パスワード認証 | 一般ユーザー向け。ログインモードによりメールアドレスまたはニックネームで認証 |
| ページ認証 | パスワード認証 | 特定ページへのアクセス制限用。ページ単位でパスワード保護が可能 |

### パスワードハッシュ化

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| ハッシュアルゴリズム | SHA-256 | `asp/includes/sha256.asp`で実装。管理者パスワードの保存に使用 |
| MD5（レガシー） | MD5 | `asp/includes/md5.asp`で実装。一部機能で使用される可能性あり |
| 簡易暗号化 | ROT13変形（13文字シフト） | `asp/includes/encryption.asp`で実装。URL暗号化やCookie暗号化に使用 |

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | 30分 | `session.Timeout = 30` (`asp/begin.asp`) |
| セッションキー | `isAUTHENTICATED`, `isAUTHENTICATEDSecondAdmin`, `isAUTHENTICATEDasADMIN`, `isAUTHENTICATEDIntranet` | 顧客ID（cId）をプレフィックスとして使用 |
| セッション固定化対策 | 部分的実装 | ログアウト時にセッション変数をクリア。Cookie認証モード有効時はCookieにもパスワードハッシュを保存 |
| Cookie認証モード | 設定可能 | `QS_enableCookieMode=true`でCookie認証を有効化。アプリケーションリサイクル対策 |

### ログイン試行制限

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| 最大試行回数 | 5回（デフォルト） | `QS_number_of_allowed_attempts_to_login`で設定 |
| ロックアウト方式 | IPアドレスベース | `application("bsLoginCount"&UserIP)`でIPごとに試行回数をカウント |
| 通知 | メール通知 | ロックアウト時に管理者へメール通知（`dumpError`関数経由） |

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| システム管理者（Admin） | 全機能 | `C_ADMINPASSWORD`で認証。全ての管理機能にアクセス可能 |
| プライマリ管理者 | 全コンテンツ管理 | `customer.adminPassword`で認証。サイト管理の全機能にアクセス可能 |
| セカンドアドミン | 制限付き管理 | 約50項目の個別権限設定が可能（ページ編集、カタログ、フォーム、ニュースレター等） |
| イントラネットユーザー | コンテンツ閲覧・投稿 | ユーザー登録機能、個人フォルダへのファイルアップロード |
| 匿名ユーザー | 公開コンテンツ閲覧 | パスワード保護されていないページのみ閲覧可能 |

### セカンドアドミン権限一覧（主要項目）

| 権限カテゴリ | 権限フラグ | 説明 |
| --- | --- | --- |
| 基本設定 | bSetupGeneral, bSetupPageElements | サイト基本設定の変更権限 |
| コンテンツ管理 | bPagesAdd, bPagesDelete, bPagesMove | ページの追加・削除・移動権限 |
| カタログ | bCatalog | カタログ（商品管理）機能の使用権限 |
| フォーム | bForms, bFormExport | フォーム管理・エクスポート権限 |
| ニュースレター | bNewsletter | ニュースレター機能の使用権限 |
| イントラネット | bIntranet, bIntranetSetup, bIntranetContacts | イントラネット機能の管理権限 |
| テンプレート | bTemplates | テンプレート管理権限 |
| ファイル | bFiles | ファイルマネージャーの使用権限 |

### アクセス制御

アクセス制御は各ASPページの先頭でセッション変数をチェックする方式で実装されている。

```
' バックサイトセキュリティチェック（bs_security.asp）
if not convertBool(Session(cId & "isAUTHENTICATED")) and
   not convertBool(Session(cId & "isAUTHENTICATEDSecondAdmin")) then
    Response.Redirect ("noaccess.htm")
end if

' 管理者セキュリティチェック（ad_security.asp）
if not convertBool(Session(cId & "isAUTHENTICATEDasADMIN")) then
    Response.Redirect ("noaccess.htm")
end if
```

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | アプリケーション側での強制なし。Webサーバー（IIS）設定に依存 |
| HSTS | 未実装。Webサーバー設定で対応が必要 |
| キャッシュ制御 | `no-cache`, `pragma: no-cache`, `Expires: -1` を設定（`begin.asp`, `security.asp`） |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | Webサーバー設定（IIS）に依存。HTTPS推奨 |
| パスワード保存 | SHA-256ハッシュ（一方向暗号化） |
| セッションデータ | ASPセッションオブジェクト（サーバーサイド保存） |
| URL/Cookie暗号化 | ROT13変形（簡易可逆暗号化）- セキュリティ目的ではなく難読化目的 |

### 機密情報管理

| 項目 | 管理方式 |
| --- | --- |
| データベース接続文字列 | `asp/config/web_config.asp`に平文保存。Webルート外への移動推奨 |
| 管理者パスワード | SHA-256ハッシュ化してデータベースに保存 |
| SMTP認証情報 | `web_config.asp`に平文保存（`C_SMTPUSERNAME`, `C_SMTPUSERPW`） |
| データベースファイル | `/db/`ディレクトリに配置。ファイル名変更を推奨 |

### ファイルアップロード制限

| 項目 | 設定 |
| --- | --- |
| 許可拡張子 | 画像（jpg, png, gif等）、文書（doc, pdf, xls等）、メディア（mp3, mp4等）、圧縮（zip, rar等） |
| 禁止拡張子 | php, exe, jsp, aspx, py, bat, cmd, vbs, scr, hta等（`begin.asp`でブロック） |
| アップロード先 | `/userfiles/`ディレクトリ。IUSRに読み書き権限が必要 |

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS（クロスサイトスクリプティング） | `sanitize()`関数による出力エスケープ（`<`, `>`, `"`を実体参照に変換）。`Server.HTMLEncode()`の使用。`filterJS()`関数によるJavaScriptイベントハンドラの無効化 |
| SQLインジェクション | `cleanUp()`関数によるシングルクォートのエスケープ（`'` -> `''`）。パラメータ化クエリは未使用 |
| CSRF（クロスサイトリクエストフォージェリ） | 一部機能で`QS_secCodeHidden`によるトークン検証を実装。全機能での統一的な対策は未実装 |
| ディレクトリトラバーサル | ファイル操作時のパス検証は部分的実装 |
| コードインジェクション | `removeCRB()`関数によるASPコードレンダーブロック（`<%`, `%>`）の除去 |

### XSS対策関数

```vbscript
' sanitize関数 - HTML出力時のエスケープ
function sanitize(sValue)
    sanitize=quotRep(sValue)
end function

function quotRep(sValue)
    if isLeeg(sValue) then
        quotRep=sValue
    else
        quotRep=replace(sValue,"""","&quot;",1,-1,1)
        quotRep=replace(quotRep,"<","&lt;",1,-1,1)
        quotRep=replace(quotRep,">","&gt;",1,-1,1)
    end if
end function

' filterJS関数 - JavaScriptイベントハンドラの無効化
function filterJS(s)
    dim arrJS, jsKey
    arrJS=array("javascript:", "vbscript:", "onabort", "onactivate", ...)
    for jsKey=lbound(arrJS) to ubound(arrJS)
        s=replace(s,arrJS(jsKey),"[" & arrJS(jsKey) & "]",1,-1,1)
    next
    filterJS=s
end function
```

### SQLインジェクション対策関数

```vbscript
' cleanUp関数 - SQLインジェクション対策
function cleanUp(str)
    if isLeeg(str) then
        cleanUp=""
    else
        cleanUp=replace(str,"'","''",1,-1,1)
    end if
end function
```

## CAPTCHA

| 項目 | 設定 |
| --- | --- |
| 実装 | `asp/includes/captcha.asp` - ASP Security Image Generator v3.0 |
| コード長 | 4文字（英数字） |
| 適用箇所 | バックサイトログイン画面 |
| 検証方式 | セッション変数（`Session("CAPTCHA")`）との照合 |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| エラーログ | VBScriptランタイムエラー、セッション変数、訪問者情報 | メール送信（`C_ADMINEMAIL`宛） |
| 不正アクセス試行ログ | IPアドレス、試行回数、対象エリア（Admin/Backsite） | メール通知のみ |
| ユーザーログインログ | 最終ログイン日時（`lastLoginTS`） | データベース保存 |
| 訪問者ログ | アクセス統計（`logVisit.asp`） | データベース保存 |

### エラー通知

```vbscript
' ErrorReport関数 - エラーメール送信
Function ErrorReport(value, byref error)
    if not isLeeg(C_ADMINEMAIL) then
        dim theMail
        set theMail=new cls_mail_message
        theMail.receiver=C_ADMINEMAIL
        theMail.subject="Error op "& customer.siteName
        theMail.body=value & "<hr /><p>Visitor details:</p>" & getVisitorDetails
        theMail.send
        set theMail=nothing
    end if
End Function
```

## デフォルトパスワード対策

| 項目 | 対策 |
| --- | --- |
| デフォルトパスワードブロック | `blockDefaultPW=true`設定時、デフォルトパスワード（"admin"）でのアクセスをブロック |
| 初期設定強制 | デフォルトパスワード使用時は`bs_initialsetup.asp`へリダイレクトしパスワード変更を強制 |
| セカンドアドミン制限 | プライマリ管理者パスワードと同一のパスワード設定を禁止 |

## 備考

### セキュリティに関する推奨事項

1. **HTTPS強制**: Webサーバー（IIS）レベルでHTTPSリダイレクトを設定することを推奨
2. **データベースファイル保護**: `/db/`ディレクトリへの直接アクセスをWebサーバー設定でブロック
3. **設定ファイル保護**: `web_config.asp`の機密情報をWebルート外に移動するか、環境変数で管理
4. **パラメータ化クエリ**: SQLインジェクション対策として、パラメータ化クエリの導入を検討
5. **CSRF対策強化**: 全フォームでCSRFトークンを実装
6. **セッション管理強化**: セッションIDの再生成、Secure/HttpOnlyフラグの設定
7. **ログ管理強化**: データベースまたはファイルへの監査ログ永続化

### 技術的制約

- Classic ASP（VBScript）の制約により、最新のセキュリティフレームワークの適用が困難
- ADO/ODBCによるデータベースアクセスのため、パラメータ化クエリの実装が複雑
- IISに依存したセキュリティ設定（HTTPS、アクセス制限等）が多い
