# 機能設計書 117-CSRF対策

## 概要

本ドキュメントは、QuickerSite CMSにおける「CSRF対策」機能の設計仕様を定義する。

### 本機能の処理概要

Cross-Site Request Forgery（CSRF）攻撃からWebアプリケーションを保護するためのトークンベース認証機能である。セッションごとに一意のトークンを生成し、フォーム送信時に検証することで、不正なリクエストを検出・ブロックする。

**業務上の目的・背景**：CSRF攻撃は、ユーザーが認証済みの状態で悪意のあるサイトを訪問した際に、ユーザーの意図しない操作（データ変更、設定変更等）を実行させる攻撃である。QuickerSite CMSでは、管理画面での設定変更やコンテンツ編集など、多数の操作が可能なため、CSRF対策は必須のセキュリティ機能である。

**機能の利用シーン**：
- 管理画面のフォーム送信時
- ページ編集・削除操作時
- 設定変更操作時
- ファイルアップロード時
- メール送信フォーム送信時
- ゲストブック投稿時

**主要な処理内容**：
1. セッション開始時にCSRFトークン（secCode）を生成
2. フォームにhidden入力フィールドとしてトークンを埋め込み（QS_secCodeHidden）
3. URLパラメータとしてトークンを付与（QS_secCodeURL）
4. フォーム送信時にトークンを検証（checkCSRF、checkCSRF_Upload）
5. トークン不一致時に警告処理（sendCSRFWarning）

**関連システム・外部連携**：
- メール送信システム（CSRF検出時の管理者通知）
- 全ての状態変更を伴うフォーム

**権限による制御**：認証・非認証に関わらず、全てのフォーム送信で適用。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | 全管理画面 | 主機能 | フォーム送信時のCSRF検証 |
| 160 | メールページ | 使用 | checkCSRF()によるトークン検証 |
| 185 | フリーページ変換 | 使用 | QS_secCodeURLによるトークン付与 |
| 60 | ゲストブック編集 | 使用 | checkCSRF()によるトークン検証 |
| 81 | ピール設定 | 使用 | checkCSRF()によるトークン検証 |
| 92 | バナーメニュー編集 | 使用 | QS_secCodeHidden/checkCSRF |

## 機能種別

セキュリティ機能 / CSRF対策

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| QSSEC | String | Yes | CSRFトークン（フォーム送信時） | セッションのsecCodeと一致するか |

### 入力データソース

- HTTPリクエストパラメータ（Request("QSSEC")）
- セッション変数（session("secCode")）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| QS_secCodeHidden | String | hidden inputタグ形式のCSRFトークン |
| QS_secCodeURL | String | URLパラメータ形式のCSRFトークン |

### 出力先

- HTMLフォーム（hidden input）
- URL（クエリストリング）

### トークン形式

```html
<!-- QS_secCodeHidden出力例 -->
<input type='hidden' name='QSSEC' value='AB1CD2EF' />
```

```
<!-- QS_secCodeURL出力例 -->
QSSEC=AB1CD2EF
```

## 処理フロー

### 処理シーケンス

```
1. セッション開始
   └─ secCode関数呼び出し
       ├─ session("secCode")が空の場合
       │   └─ GeneratePassWord()でトークン生成
       └─ session("secCode")を返却

2. フォーム生成時
   ├─ QS_secCodeHidden呼び出し
   │   └─ <input type='hidden' name='QSSEC' value='トークン' /> 生成
   └─ QS_secCodeURL呼び出し（URL用）
       └─ "QSSEC=トークン" 生成

3. フォーム送信時
   ├─ checkCSRF呼び出し
   │   ├─ Request("QSSEC")取得
   │   ├─ session("secCode")と比較
   │   ├─ 一致 → 処理継続
   │   └─ 不一致 → sendCSRFWarning呼び出し
   │
   └─ checkCSRF_Upload呼び出し（アップロード時）
       ├─ 引数のqssec_valueを使用
       ├─ session("secCode")と比較
       └─ 不一致 → sendCSRFWarning呼び出し

4. CSRF検出時（sendCSRFWarning）
   ├─ Application変数でメール送信済みかチェック
   ├─ 未送信かつC_ADMINEMAIL設定あり
   │   ├─ cls_mail_message作成
   │   ├─ 管理者に警告メール送信
   │   └─ Application("CSRFMS" & UserIP)にフラグ設定
   └─ 処理終了（Response.End）
```

### フローチャート

```mermaid
flowchart TD
    A[セッション開始] --> B{session&#40;secCode&#41;<br>存在?}
    B -->|No| C[GeneratePassWord&#40;&#41;<br>でトークン生成]
    B -->|Yes| D[既存トークン使用]
    C --> E[session&#40;secCode&#41;に保存]
    E --> D

    F[フォーム生成] --> G[QS_secCodeHidden&#40;&#41;]
    G --> H[hidden inputタグ出力]

    I[フォーム送信] --> J[checkCSRF&#40;&#41;]
    J --> K{Request&#40;QSSEC&#41;<br>== session&#40;secCode&#41;?}
    K -->|Yes| L[処理継続]
    K -->|No| M[sendCSRFWarning&#40;&#41;]
    M --> N{メール未送信?}
    N -->|Yes| O[管理者にメール送信]
    N -->|No| P[スキップ]
    O --> Q[処理終了]
    P --> Q
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-117-01 | トークン生成 | セッション開始時にGeneratePassWordで8文字のトークンを生成 | 初回アクセス時 |
| BR-117-02 | トークン形式 | 英大文字と数字の組み合わせ（例：AB1CD2EF） | トークン生成時 |
| BR-117-03 | セッション単位 | CSRFトークンはセッション単位で管理 | 全リクエスト |
| BR-117-04 | フォーム埋め込み | 状態変更を伴う全フォームにトークンを埋め込み | フォーム生成時 |
| BR-117-05 | 検証必須 | 状態変更を伴う全リクエストでトークン検証 | POST/GET送信時 |
| BR-117-06 | メール通知制限 | 同一IPからのCSRF検出メールは1回のみ | CSRF検出時 |

### 計算ロジック

#### トークン生成（GeneratePassWord関数）

```vb
' 6-12文字のランダム文字列を生成
' 英大文字と数字の組み合わせ
Randomize
Lenght = Int(7 * Rnd) + 6  ' 6-12文字

For Index = 1 To Lenght
    LetterNumber = Int(25 * Rnd)
    CapitalNumber = Rnd
    If (CapitalNumber < 0.5) Then
        Password = Password & Chr(65 + LetterNumber)  ' A-Z
    Else
        Password = Password & Chr(97 + LetterNumber)  ' a-z
    End If
Next

' 最終的に8文字に整形（数字を2箇所挿入）
GeneratePassWord = Left(Password, 2) & Int(Rnd*10) & Mid(Password, 3, 2) & Int(Rnd*10) & Mid(Password, 5, 2)
```

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作なし（セッション変数とApplication変数のみ使用）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | CSRF検出 | トークン不一致 | sendCSRFWarningで管理者通知、処理中断 |
| - | トークン未設定 | QSSECパラメータなし | 空文字として比較、不一致として検出 |

### リトライ仕様

特になし（CSRF検出時は処理中断）

## トランザクション仕様

データベース操作を伴わないため、トランザクション管理は不要。

## パフォーマンス要件

- トークン生成: 1ms以内
- トークン検証: 1ms以内
- セッション変数アクセスのみで高速処理

## セキュリティ考慮事項

1. **トークンの予測困難性**: Randomizeと複数のRnd呼び出しで予測困難なトークンを生成
2. **セッション単位管理**: トークンはセッションに紐づき、セッション切れで無効化
3. **検証の徹底**: 状態変更を伴う全操作でcheckCSRFを呼び出し
4. **攻撃検知**: CSRF攻撃検出時に管理者へメール通知
5. **重複通知防止**: 同一IPからの通知は1回のみ（Application変数で制御）

## 備考

- トークンはセッションに保存されるため、セッションタイムアウト後は無効
- アップロード処理ではRequest("QSSEC")が使えないため、checkCSRF_Upload関数で別途対応
- 現在sendCSRFWarning内のtheMail.sendはコメントアウトされており、メール送信は無効化されている
- Response.Endもコメントアウトされており、CSRF検出後も処理が継続する可能性がある点に注意

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

CSRFトークン管理に使用するデータ構造を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | functions.asp | `asp/includes/functions.asp` 1607-1612行目 | secCode関数、session("secCode")の管理 |

**読解のコツ**: session("secCode")がトークンの保存先。初回アクセス時のみGeneratePassWord()で生成。

#### Step 2: エントリーポイントを理解する

CSRF対策の主要関数を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | functions.asp | `asp/includes/functions.asp` 1607-1641行目 | CSRF関連全関数 |

**主要関数一覧**:
1. **1607-1612行目**: `secCode` - トークン取得/生成
2. **1613-1615行目**: `QS_secCodeHidden` - hidden inputタグ生成
3. **1616-1618行目**: `QS_secCodeURL` - URLパラメータ形式生成
4. **1619-1623行目**: `checkCSRF` - トークン検証（通常フォーム）
5. **1624-1628行目**: `checkCSRF_Upload` - トークン検証（アップロード用）
6. **1629-1641行目**: `sendCSRFWarning` - CSRF検出時の警告処理

#### Step 3: トークン生成ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | functions.asp | `asp/includes/functions.asp` 800-829行目 | GeneratePassWord関数 |

**ポイント**: 6-12文字のランダム文字列を生成し、最終的に8文字（英字+数字）に整形。

### プログラム呼び出し階層図

```
[フォーム生成時]
QS_secCodeHidden()
    │
    └─ secCode()
           │
           ├─ isLeeg(session("secCode")) ... 空チェック
           │
           └─ GeneratePassWord() ... トークン生成（初回のみ）

[フォーム送信時]
checkCSRF()
    │
    ├─ convertStr(Request("QSSEC")) ... 送信されたトークン
    │
    ├─ secCode() ... セッションのトークン
    │
    └─ sendCSRFWarning() ... 不一致時
           │
           ├─ isLeeg(Application("CSRFMS" & UserIP)) ... 重複チェック
           │
           ├─ cls_mail_message ... メール送信クラス
           │
           └─ getVisitorDetails ... 訪問者情報取得

[アップロード時]
checkCSRF_Upload(qssec_value)
    │
    ├─ convertStr(qssec_value) ... 引数のトークン
    │
    ├─ secCode() ... セッションのトークン
    │
    └─ sendCSRFWarning() ... 不一致時
```

### データフロー図

```
[入力]                      [処理]                           [出力]

セッション開始 ──────────> secCode() ─────────────────> session("secCode")
                          (トークン生成)                   (8文字トークン)

フォーム生成 ───────────> QS_secCodeHidden() ──────────> hidden input
                              │
                              └─ secCode() ────────────> トークン取得

フォーム送信 ───────────> checkCSRF() ─────────────────> 検証結果
(QSSEC)                       │
                              ├─ Request("QSSEC") ────> 送信トークン
                              │
                              ├─ secCode() ───────────> セッショントークン
                              │
                              └─ 比較 ────────────────> 一致: 継続
                                                         不一致: sendCSRFWarning

CSRF検出 ──────────────> sendCSRFWarning() ────────────> メール送信
                              │                          (管理者通知)
                              └─ getVisitorDetails ───> 訪問者情報
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| functions.asp | `asp/includes/functions.asp` | ソース | CSRF関連関数定義 |
| begin.asp | `asp/begin.asp` | ソース | セッション初期化（secCode生成契機） |
| mail_message.asp | `asp/includes/mail_message.asp` | ソース | メール送信クラス（警告通知用） |

### 利用例

```asp
' フォーム生成時
<form method="post" action="process.asp">
    <%=QS_secCodeHidden%>
    <input type="submit" value="送信" />
</form>

' URL生成時
<a href="action.asp?<%=QS_secCodeURL%>&action=delete">削除</a>

' 送信処理時
<%
checkCSRF()  ' トークン検証（不一致時は処理中断）
' 以降の処理...
%>

' アップロード処理時（FreeASPUpload使用時）
<%
checkCSRF_Upload(Upload.Form("QSSEC"))
%>
```
